package com.instabug.library.view.pagerindicator;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.animation.ValueAnimator;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.OvalShape;
import android.util.AttributeSet;
import android.view.Gravity;
import android.widget.ImageView;
import android.widget.RelativeLayout;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.view.ViewUtils;

final class Dot extends RelativeLayout {

    private static final int DEFAULT_INACTIVE_DIAMETER_DP = 6;
    private static final int DEFAULT_ACTIVE_DIAMETER_DP = 9;
    private static final int DEFAULT_INACTIVE_COLOR = Color.WHITE;
    private static final int DEFAULT_ACTIVE_COLOR = Color.WHITE;
    private static final int DEFAULT_TRANSITION_DURATION_MS = 200;
    private static final boolean DEFAULT_INITIALLY_ACTIVE = false;
    private int inactiveDiameterPx;
    private int activeDiameterPx;
    private int inactiveColor;
    private int activeColor;
    private int transitionDurationMs;
    @Nullable
    private State state;
    private ShapeDrawable shape;
    private ImageView drawableHolder;
    @Nullable
    private AnimatorSet currentAnimator = null;

    public Dot(Context context) {
        this(context, null);
    }

    public Dot(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public Dot(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(attrs, defStyleAttr);
    }

    @SuppressLint({"ERADICATE_PARAMETER_NOT_NULLABLE", "CustomViewStyleable"})
    private void init(@Nullable AttributeSet attrs, int defStyleAttr) {
        // Use a TypedArray to process attrs
        final TypedArray attributes = getContext()
                .obtainStyledAttributes(attrs, com.instabug.library.R.styleable.IBDot, defStyleAttr, 0);

        // Convert default dimensions to px
        final int defaultActiveDiameterPx = ViewUtils.convertDpToPx(getContext(),
                DEFAULT_ACTIVE_DIAMETER_DP);
        final int defaultInactiveDiameterPx =
                ViewUtils.convertDpToPx(getContext(), DEFAULT_INACTIVE_DIAMETER_DP);

        // Assign provided attributes to member variables, or use the defaults if necessary
        inactiveDiameterPx = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDot_ibViewPagerInactiveDiameter,
                        defaultInactiveDiameterPx);
        activeDiameterPx = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDot_ibViewPagerActiveDiameter, defaultActiveDiameterPx);
        inactiveColor = attributes.getColor(com.instabug.library.R.styleable.IBDot_ibViewPagerInactiveColor, DEFAULT_INACTIVE_COLOR);
        activeColor = attributes.getColor(com.instabug.library.R.styleable.IBDot_ibViewPagerActiveColor, DEFAULT_ACTIVE_COLOR);
        transitionDurationMs = attributes
                .getInt(com.instabug.library.R.styleable.IBDot_ibViewPagerTransitionDuration, DEFAULT_TRANSITION_DURATION_MS);
        state = attributes.getBoolean(com.instabug.library.R.styleable.IBDot_ibViewPagerInitiallyActive, DEFAULT_INITIALLY_ACTIVE) ?
                State.ACTIVE : State.INACTIVE;

        // Attributes are no longer required
        attributes.recycle();

        // Ensure the view reflects the attributes
        reflectParametersInView();
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void reflectParametersInView() {
        // Reset root View so that the UI can be entirely recreated
        removeAllViews();

        // Make the root View bounds big enough to encompass the maximum diameter
        final int maxDimension = Math.max(inactiveDiameterPx, activeDiameterPx);
        setLayoutParams(new LayoutParams(maxDimension, maxDimension));

        // Set the gravity to centre for simplicity
        setGravity(Gravity.CENTER);

        // Create the drawable based on the current member variables
        final int diameter = (state == State.ACTIVE) ? activeDiameterPx : inactiveDiameterPx;
        final int color = (state == State.ACTIVE) ? activeColor : inactiveColor;
        shape = new ShapeDrawable(new OvalShape());
        shape.setIntrinsicWidth(diameter);
        shape.setIntrinsicHeight(diameter);
        shape.getPaint().setColor(color);

        // Add the drawable to the drawable holder
        drawableHolder = new ImageView(getContext());
        drawableHolder.setImageDrawable(null); // Forces redraw
        drawableHolder.setImageDrawable(shape);

        // Add the drawable holder to root View
        addView(drawableHolder);
    }

    private void animateDotChange(final int startSize, final int endSize, final int startColor,
                                  final int endColor, final int duration) {
        if (startSize < 0) {
            throw new IllegalArgumentException("startSize cannot be less than 0");
        } else if (endSize < 0) {
            throw new IllegalArgumentException("endSize cannot be less than 0");
        } else if (duration < 0) {
            throw new IllegalArgumentException("duration cannot be less than 0");
        }

        // To avoid conflicting animations, cancel any existing animation
        if (currentAnimator != null) {
            currentAnimator.cancel();
        }

        // Use an animator set to coordinate shape and color change animations
        currentAnimator = new AnimatorSet();
        currentAnimator.setDuration(duration);
        currentAnimator.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationStart(Animator animation) {
                // The state must be updated to reflect the transition
                if (state == State.INACTIVE) {
                    state = State.TRANSITIONING_TO_ACTIVE;
                } else if (state == State.ACTIVE) {
                    state = State.TRANSITIONING_TO_INACTIVE;
                }
            }

            @Override
            public void onAnimationEnd(Animator animation) {
                // Make sure state is stable (i.e. unchanging) at the end of the animation
                if (state != null && !state.isStable()) {
                    state = state.transitioningTo();
                }

                // Make sure the properties are correct
                changeSize(endSize);
                changeColor(endColor);

                // Declare the animation finished
                currentAnimator = null;
            }

            @Override
            public void onAnimationCancel(Animator animation) {
                // Make sure state is stable (i.e. unchanging) at the end of the animation
                if (state != null && !state.isStable()) {
                    state = state.transitioningFrom();
                }

                // Make sure the properties are correct
                changeSize(startSize);
                changeColor(startColor);

                // Declare the animation finished
                currentAnimator = null;
            }
        });

        ValueAnimator transitionSize = ValueAnimator.ofInt(startSize, endSize);
        transitionSize.addUpdateListener(animation -> {
            int size = (Integer) animation.getAnimatedValue();
            changeSize(size);
        });

        ValueAnimator transitionColor = ValueAnimator.ofFloat(0f, 1f);
        transitionColor.addUpdateListener(animation -> changeColor(endColor));

        if (currentAnimator != null) {
            currentAnimator.playTogether(transitionSize, transitionColor);
            currentAnimator.start();
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void changeSize(final int newSizePx) {
        shape.setIntrinsicWidth(newSizePx);
        shape.setIntrinsicHeight(newSizePx);
        drawableHolder.setImageDrawable(null); // Forces ImageView to update drawable
        drawableHolder.setImageDrawable(shape);
    }

    private void changeColor(final int newColor) {
        shape.getPaint().setColor(newColor);
    }

    public Dot setInactiveDiameterPx(final int inactiveDiameterPx) {
        if (inactiveDiameterPx < 0) {
            throw new IllegalArgumentException("inactiveDiameterPx cannot be less than 0");
        }

        this.inactiveDiameterPx = inactiveDiameterPx;
        reflectParametersInView();
        return this;
    }

    public Dot setInactiveDiameterDp(final int inactiveDiameterDp) {
        if (inactiveDiameterDp < 0) {
            throw new IllegalArgumentException("inactiveDiameterDp cannot be less than 0");
        }

        setInactiveDiameterPx(ViewUtils.convertDpToPx(getContext(), inactiveDiameterDp));
        return this;
    }

    public int getInactiveDiameter() {
        return inactiveDiameterPx;
    }

    public Dot setActiveDiameterPx(final int activeDiameterPx) {
        if (activeDiameterPx < 0) {
            throw new IllegalArgumentException("activeDiameterPx cannot be less than 0");
        }

        this.activeDiameterPx = activeDiameterPx;
        reflectParametersInView();
        return this;
    }

    public Dot setActiveDiameterDp(final int activeDiameterDp) {
        if (activeDiameterDp < 0) {
            throw new IllegalArgumentException("activeDiameterDp cannot be less than 0");
        }

        setActiveDiameterPx(activeDiameterDp);
        return this;
    }

    public int getActiveDiameter() {
        return activeDiameterPx;
    }

    public int getInactiveColor() {
        return inactiveColor;
    }

    public Dot setInactiveColor(final int inactiveColor) {
        this.inactiveColor = inactiveColor;
        reflectParametersInView();
        return this;
    }

    public int getActiveColor() {
        return activeColor;
    }

    public Dot setActiveColor(final int activeColor) {
        this.activeColor = activeColor;
        reflectParametersInView();
        return this;
    }

    public int getTransitionDuration() {
        return transitionDurationMs;
    }

    public Dot setTransitionDuration(final int transitionDurationMs) {
        if (transitionDurationMs < 0) {
            throw new IllegalArgumentException("transitionDurationMs cannot be less than 0");
        }

        this.transitionDurationMs = transitionDurationMs;
        return this;
    }

    public void toggleState(final boolean animate) {
        if (currentAnimator != null) {
            currentAnimator.cancel();
        }

        if (state != State.ACTIVE) {
            setActive(animate);
        } else if (state != State.INACTIVE) {
            setInactive(animate);
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "[Animation trying to start from illegal state]");
        }
    }

    public void setInactive(final boolean animate) {
        // Any existing animation will conflict with this animations and must be cancelled
        if (currentAnimator != null) {
            currentAnimator.cancel();
        }

        // Animate only if the animation is requested, is necessary, and will actually display
        final boolean shouldAnimate =
                animate && (state != State.INACTIVE) && (transitionDurationMs > 0);

        if (shouldAnimate) {
            animateDotChange(activeDiameterPx, inactiveDiameterPx, activeColor, inactiveColor,
                    transitionDurationMs);
        } else {
            // The UI must still be changed, just without animations
            changeSize(inactiveDiameterPx);
            changeColor(inactiveColor);
            state = State.INACTIVE;
        }
    }

    public void setActive(final boolean animate) {
        // Any existing animation will conflict with this animations and must be cancelled
        if (currentAnimator != null) {
            currentAnimator.cancel();
        }

        // Animate only if the animation is requested, is necessary, and will actually display
        final boolean shouldAnimate =
                animate && (state != State.ACTIVE) && (transitionDurationMs > 0);


        if (shouldAnimate) {
            animateDotChange(inactiveDiameterPx, activeDiameterPx, inactiveColor, activeColor,
                    transitionDurationMs);
        } else {
            // The UI must still be changed, just without animations
            changeSize(activeDiameterPx);
            changeColor(activeColor);
            state = State.ACTIVE;
        }
    }

    protected enum State {
        INACTIVE(true, null, null),
        ACTIVE(true, null, null),
        TRANSITIONING_TO_ACTIVE(false, ACTIVE, INACTIVE),
        TRANSITIONING_TO_INACTIVE(false, INACTIVE, ACTIVE);

        private final boolean isStable;
        @Nullable
        private final State to;
        @Nullable
        private final State from;

        State(final boolean isStable, @Nullable State to, @Nullable State from) {
            this.isStable = isStable;
            this.to = to;
            this.from = from;
        }

        public boolean isStable() {
            return isStable;
        }

        @Nullable
        public State transitioningTo() {
            return to;
        }

        @Nullable
        public State transitioningFrom() {
            return from;
        }
    }
}
