package com.instabug.library.util;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;

import com.instabug.library.Constants;
import com.instabug.library.settings.PerSessionSettings;
import com.instabug.library.settings.SettingsManager;

import java.util.HashSet;

import androidx.annotation.Nullable;

import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_BEHIND;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_FULL_SENSOR;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_FULL_USER;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_LOCKED;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_NOSENSOR;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_SENSOR;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_USER;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_USER_LANDSCAPE;
import static android.content.pm.ActivityInfo.SCREEN_ORIENTATION_USER_PORTRAIT;

/**
 * Static methods related to device orientation.
 */
public class OrientationUtils {

    private static HashSet<String> lockedActivities = new HashSet<>();

    private OrientationUtils() {
    }

    /**
     * Allows user to freely use portrait or landscape mode.
     */
    public static void unlockOrientation(Context context) {
        Activity activity = getActivity(context);
        if (activity != null) {
            if (!lockedActivities.contains(activity.getClass().getName())) {
                return;
            }
            InstabugSDKLogger.v(Constants.LOG_TAG, "Unlocking orientation for activity " + activity.toString());
            ComponentName name = new ComponentName(activity, activity.getClass());
            PackageManager pm = activity.getPackageManager();
            ActivityInfo info;
            try {
                info = pm.getActivityInfo(name, PackageManager.GET_META_DATA);
                activity.setRequestedOrientation(info.screenOrientation);
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
                activity.setRequestedOrientation(SCREEN_ORIENTATION_UNSPECIFIED);
            } catch (Exception e) {
                activity.setRequestedOrientation(SCREEN_ORIENTATION_UNSPECIFIED);
            }
        }
    }

    public static void lockScreenOrientation(Context context) {
        Activity activity = getActivity(context);
        if (activity != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Locking orientation for activity " + activity.toString());
            int rotation = activity.getWindowManager().getDefaultDisplay().getRotation();

            switch (activity.getResources().getConfiguration().orientation) {
                case Configuration.ORIENTATION_PORTRAIT:
                    //FIXME: maybe remove this and use sensor portrait and sensor landscape?
                    if (rotation == android.view.Surface.ROTATION_90
                            || rotation == android.view.Surface.ROTATION_180) {
                        activity.setRequestedOrientation(SCREEN_ORIENTATION_REVERSE_PORTRAIT);
                    } else {
                        activity.setRequestedOrientation(SCREEN_ORIENTATION_PORTRAIT);
                    }
                    break;

                case Configuration.ORIENTATION_LANDSCAPE:
                    if (rotation == android.view.Surface.ROTATION_0
                            || rotation == android.view.Surface.ROTATION_90) {
                        activity.setRequestedOrientation(SCREEN_ORIENTATION_LANDSCAPE);
                    } else {
                        activity.setRequestedOrientation(SCREEN_ORIENTATION_REVERSE_LANDSCAPE);
                    }
                    break;
                default:
                    break;
            }
            lockedActivities.add(activity.getClass().getName());
        }
    }

    public static void handelOrientation(Context context) {
        if (SettingsManager.getInstance().getRequestedOrientation() == PerSessionSettings.DEFAULT_ORIENTATION) {
            OrientationUtils.lockScreenOrientation(context);
        } else {
            Activity activity = getActivity(context);
            if (activity != null) {
                activity.setRequestedOrientation(getOrientation(SettingsManager.getInstance()
                        .getRequestedOrientation()));
            }
        }
    }

    @InstabugAnnotations.ScreenOrientation
    public static int getOrientation(int orientationValue) {
        switch (orientationValue) {
            case SCREEN_ORIENTATION_UNSPECIFIED:
                return SCREEN_ORIENTATION_UNSPECIFIED;
            case SCREEN_ORIENTATION_LANDSCAPE:
                return SCREEN_ORIENTATION_LANDSCAPE;
            case SCREEN_ORIENTATION_PORTRAIT:
                return SCREEN_ORIENTATION_PORTRAIT;
            case SCREEN_ORIENTATION_USER:
                return SCREEN_ORIENTATION_USER;
            case SCREEN_ORIENTATION_BEHIND:
                return SCREEN_ORIENTATION_BEHIND;
            case SCREEN_ORIENTATION_SENSOR:
                return SCREEN_ORIENTATION_SENSOR;
            case SCREEN_ORIENTATION_NOSENSOR:
                return SCREEN_ORIENTATION_NOSENSOR;
            case SCREEN_ORIENTATION_SENSOR_LANDSCAPE:
                return SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
            case SCREEN_ORIENTATION_REVERSE_LANDSCAPE:
                return SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
            case SCREEN_ORIENTATION_REVERSE_PORTRAIT:
                return SCREEN_ORIENTATION_REVERSE_PORTRAIT;
            case SCREEN_ORIENTATION_FULL_SENSOR:
                return SCREEN_ORIENTATION_FULL_SENSOR;
            case SCREEN_ORIENTATION_USER_LANDSCAPE:
                return SCREEN_ORIENTATION_USER_LANDSCAPE;
            case SCREEN_ORIENTATION_USER_PORTRAIT:
                return SCREEN_ORIENTATION_USER_PORTRAIT;
            case SCREEN_ORIENTATION_FULL_USER:
                return SCREEN_ORIENTATION_FULL_USER;
            case SCREEN_ORIENTATION_LOCKED:
                return SCREEN_ORIENTATION_LOCKED;
            default:
                return SCREEN_ORIENTATION_UNSPECIFIED;
        }
    }

    public static boolean isInLandscape(Context context) {
        return context != null && context.getResources().getConfiguration().orientation
                == Configuration.ORIENTATION_LANDSCAPE;
    }

    @Nullable
    private static Activity getActivity(Context context) {
        if (context != null) {
            while (context instanceof ContextWrapper) {
                if (context instanceof Activity) {
                    return (Activity) context;
                }
                context = ((ContextWrapper) context).getBaseContext();
            }
        }
        return null;
    }
}
