package com.instabug.library.sessionreplay.monitoring

import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable
import com.instabug.library.internal.filestore.CurrentSpanSelector
import com.instabug.library.internal.filestore.DeleteDirectory
import com.instabug.library.internal.filestore.MakeDirectoryWithAncestors
import com.instabug.library.internal.filestore.OperationScopeBuilder
import com.instabug.library.internal.filestore.SpansDirectoryFactory
import com.instabug.library.logscollection.GarbageCollector
import com.instabug.library.screenshot.analytics.AnalyticsEvent
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus
import com.instabug.library.sessionreplay.SRExecutionQueues
import com.instabug.library.sessionreplay.SRLoggingController.Status
import com.instabug.library.sessionreplay.SR_LOG_TAG
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.model.SRLog
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.threading.OrderedExecutorService

interface SRMonitorLifecycle {
    fun onSessionStarted(sessionId: String)
    fun onConfigurationsChanged(isMonitoringAvailable: Boolean)
    fun onBeingCleansed()
    fun onSessionEnded()
    fun onSRDisabled()
}

interface SRMonitoringDelegate : SRMonitorLifecycle, SRLoggingMonitor

class QueuedSRMonitoringDelegate(
    private val executor: OrderedExecutorService,
    private val loggingMonitor: ControlledSRLoggingMonitor,
    private val dataStore: SRMonitoringSpansDataStore,
    private val directoryFactory: SpansDirectoryFactory<SRMonitoringDirectory>,
    private val screenshotsAnalyticsEventBus: ScreenshotsAnalyticsEventBus,
    configurationsProvider: SRConfigurationsProvider,
    garbageCollector: GarbageCollector
) : SRMonitoringDelegate {

    private var runningSessionId: String? = null

    private var isMonitoringAvailable: Boolean = configurationsProvider.isMonitoringAvailable

    private var monitoringDirectory: SRMonitoringDirectory? = null
    private var screenshotsErrorsDisposable: IBGDisposable? = null

    init {
        executor.execute(SRExecutionQueues.Monitor) {
            directoryFactory()
                ?.also { directory -> monitoringDirectory = directory }
                ?.also(dataStore::init)
            "[Monitoring] Invoking garbage collector".logDWithThreadName(SR_LOG_TAG)
            garbageCollector.invoke()
        }
    }

    override fun onSessionStarted(sessionId: String) {
        executor.execute(SRExecutionQueues.Monitor) {
            runningSessionId = sessionId
            "[Monitoring] New session $sessionId started".logDWithThreadName(SR_LOG_TAG)
            takeIf { isMonitoringAvailable }
                ?.initMonitoring(sessionId)

        }
    }

    override fun onConfigurationsChanged(isMonitoringAvailable: Boolean) {
        executor.execute(SRExecutionQueues.Monitor) { handleConfigurations(isMonitoringAvailable) }
    }

    private fun handleConfigurations(isMonitoringAvailable: Boolean) {
        """
            [Monitoring] Handling configurations:
            Current availability: ${this.isMonitoringAvailable}
            New availability: $isMonitoringAvailable
        """.trimIndent().logDWithThreadName(SR_LOG_TAG)
        if (isMonitoringAvailable == this.isMonitoringAvailable) return
        this.isMonitoringAvailable = isMonitoringAvailable
        takeIf { isMonitoringAvailable }
            ?.apply { runningSessionId?.also(this::initMonitoring) }
            ?: shutdownMonitoring()
    }

    override fun onBeingCleansed() {
        executor.execute(SRExecutionQueues.Monitor) {
            "[Monitoring] Cleansing monitoring data".logDWithThreadName(SR_LOG_TAG)
            dataStore.cleanse()
        }
    }

    override fun onSessionEnded() {
        executor.execute(SRExecutionQueues.Monitor) {
            "[Monitoring] Session ended".logDWithThreadName(SR_LOG_TAG)
            runningSessionId = null
            takeIf { isMonitoringAvailable }
                ?.apply { directoryFactory.setCurrentSpanId(null) }
                ?.apply { monitoringDirectory = directoryFactory() }
                ?.apply { dataStore.onSpanEnded() }
                ?.apply { loggingMonitor.shutdown() }
        }
    }

    private fun disposeScreenshotsErrorsObserver() {
        screenshotsErrorsDisposable?.dispose()
        screenshotsErrorsDisposable = null
    }

    override fun onSRDisabled() {
        executor.execute(SRExecutionQueues.Monitor) {
            "[Monitoring] SR got disabled".logDWithThreadName(SR_LOG_TAG)
            runningSessionId = null
            shutdownMonitoring()
        }
    }

    override fun trackLog(log: SRLog, @Status status: Int) {
        executor.execute(SRExecutionQueues.Monitor) {
            loggingMonitor.trackLog(log, status)
        }
    }

    override fun trackScreenshot(@Status status: Int) {
        executor.execute(SRExecutionQueues.Monitor) {
            loggingMonitor.trackScreenshot(status)
        }
    }

    override fun analyzeAndTrackException(throwable: Throwable?) {
        executor.execute(SRExecutionQueues.Monitor) {
            loggingMonitor.analyzeAndTrackException(throwable)
        }
    }

    override fun analyzeAndTrackConfigurations(configurations: SRConfigurations) {
        executor.execute(SRExecutionQueues.Monitor) {
            loggingMonitor.analyzeAndTrackConfigurations(configurations)
        }
    }

    override fun onNewEvent(event: AnalyticsEvent) {
        executor.execute(SRExecutionQueues.Monitor) {
            loggingMonitor.onNewEvent(event)
        }
    }

    private fun initMonitoring(sessionId: String) {
        "[Monitoring] Initializing monitoring components".logDWithThreadName(SR_LOG_TAG)
        with(directoryFactory) {
            setCurrentSpanId(sessionId)
            monitoringDirectory = invoke()
        }
        loggingMonitor.init(sessionId)
        screenshotsErrorsDisposable =
            screenshotsErrorsDisposable ?: screenshotsAnalyticsEventBus.subscribe(this)
        with(requireNotNull(monitoringDirectory)) {
            OperationScopeBuilder(MakeDirectoryWithAncestors<SRMonitoringSessionDirectory>())
                .onSpan(CurrentSpanSelector())
                .buildAndExecute(this)
            dataStore.onSpanStarted(sessionId)
        }
    }

    private fun shutdownMonitoring() {
        "[Monitoring] Shutting down monitoring components".logDWithThreadName(SR_LOG_TAG)
        loggingMonitor.shutdown()
        disposeScreenshotsErrorsObserver()
        dataStore.shutdown().get()
        monitoringDirectory = directoryFactory.apply { setCurrentSpanId(null) }.invoke()
        monitoringDirectory?.let(DeleteDirectory<SRMonitoringDirectory>()::invoke)
    }
}