package com.instabug.library.sessionreplay

import android.content.Context
import androidx.annotation.WorkerThread
import com.instabug.library.FilesCacheDirectory
import com.instabug.library.GenericSpansCacheDirectory
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.util.extenstions.defensiveExecute
import com.instabug.library.util.extenstions.defensiveSubmit
import com.instabug.library.util.threading.OrderedExecutorService
import java.io.File
import java.util.concurrent.Future

private const val SR_LOGS_FILE_NAME = "logs.txt"
private const val SR_SCREENSHOTS_FILE_NAME = "screenshots"
private const val SR_CP_SCREENSHOTS_FILE_NAME = "screenshots-cp.zip"
private const val SR_CP_LOGS_FILE_NAME = "logs-cp.txt"

/**
 * Represents a single session-replay session directory. Also, provides easy access to directory contents.
 */
class SRSessionDirectory(
    val sessionId: String,
    private val parent: File
) : FilesCacheDirectory {
    override val filesDirectory: File
        get() = File(parent, sessionId)
    val logsFile: File
        get() = File(filesDirectory, SR_LOGS_FILE_NAME)

    val compressedLogsFile: File
        get() = File(filesDirectory, SR_CP_LOGS_FILE_NAME)
    val screenshots: Directory
        get() = Directory(filesDirectory, SR_SCREENSHOTS_FILE_NAME)
    val screenshotsZipFile: File
        get() = File(filesDirectory, SR_CP_SCREENSHOTS_FILE_NAME)
}

/**
 * An extension to [GenericSpansCacheDirectory] providing methods for executing atomic operations
 * on span directories.
 */
interface OperableSpansCacheDirectory<T> : GenericSpansCacheDirectory<T> {
    /**
     * Executes a [FileOperation] on the current span directory.
     * @param operation the [FileOperation] to execute.
     */
    fun <Out> operateOnCurrent(operation: FileOperation<T, Out>): Future<Out?>

    /**
     * Executes a [FileOperation] on the list of old spans directories.
     * @param operation the [FileOperation] to execute.
     */
    fun <Out> operateOnOld(operation: FileOperation<List<T>, Out>): Future<Out?>

    /**
     * Executes a [FileOperation] on the session-replay directory.
     * @param operation the [FileOperation] to submit.
     * @return future of results of the operation
     */
    infix fun <Out> operate(operation: FileOperation<File, Out>): Future<Out?>
}

private const val SR_DIR_NAME = "session-replay"

class SRFilesDirectory(
    private val executor: OrderedExecutorService,
    private val ctxGetter: () -> Context?,
    private val baseDirectoryGetter: (Context) -> File?
) : OperableSpansCacheDirectory<SRSessionDirectory> {

    private var currentSpanId: String? = null

    private val _filesDirectory: File?
        get() = ctxGetter()?.let(baseDirectoryGetter)
            ?.run { File(this, SR_DIR_NAME) }

    override val filesDirectory: File?
        @WorkerThread get() = executor.defensiveSubmit(
            key = SRExecutionQueues.LogsDirectory,
            errorMessage = "Failure while retrieving files dir",
            tag = SR_LOG_TAG
        ) { _filesDirectory }.get()

    private val _currentSpanDirectory: SRSessionDirectory?
        get() = currentSpanId?.let { spanId ->
            _filesDirectory?.let { parent -> SRSessionDirectory(spanId, parent) }
        }

    override val currentSpanDirectory: SRSessionDirectory?
        @WorkerThread get() = executor.defensiveSubmit(
            key = SRExecutionQueues.LogsDirectory,
            errorMessage = "Failure while retrieving current dir",
            tag = SR_LOG_TAG
        ) { _currentSpanDirectory }.get()

    private val _oldSpansDirectories: List<SRSessionDirectory>
        get() = _filesDirectory?.run {
            listFiles { file -> file.name != currentSpanId }
                ?.map { spanDir -> SRSessionDirectory(spanDir.name, this) }
        } ?: emptyList()

    override val oldSpansDirectories: List<SRSessionDirectory>
        @WorkerThread get() = executor.defensiveSubmit(
            key = SRExecutionQueues.LogsDirectory,
            errorMessage = "Failure while retrieving old dirs",
            tag = SR_LOG_TAG
        ) { _oldSpansDirectories }.get().orEmpty()

    override fun setCurrentSpanId(spanId: String?) {
        executor.defensiveExecute(
            key = SRExecutionQueues.LogsDirectory,
            errorMessage = "Failure while setting current span ID",
            tag = SR_LOG_TAG
        ) { currentSpanId = spanId }
    }

    override fun <Out> operateOnCurrent(operation: FileOperation<SRSessionDirectory, Out>): Future<Out?> =
        executor.submit(SRExecutionQueues.LogsDirectory) { _currentSpanDirectory?.let(operation::invoke) }

    override fun <Out> operateOnOld(operation: FileOperation<List<SRSessionDirectory>, Out>): Future<Out?> =
        executor.submit(SRExecutionQueues.LogsDirectory) { operation(_oldSpansDirectories) }

    override fun <Out> operate(operation: FileOperation<File, Out>): Future<Out?> =
        executor.submit(SRExecutionQueues.LogsDirectory) { _filesDirectory?.let(operation::invoke) }
}