package com.instabug.library.screenshot.instacapture

import android.app.Activity
import androidx.annotation.IntDef
import androidx.annotation.WorkerThread
import com.instabug.library.Constants
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus
import com.instabug.library.instacapture.screenshot.ViewsBitmapObservable
import com.instabug.library.internal.servicelocator.CoreServiceLocator.ignoredViewsIds
import com.instabug.library.internal.video.RequestPermissionActivityLauncher
import com.instabug.library.internal.video.ScreenRecordingServiceAction
import com.instabug.library.screenshot.ScreenshotCaptor
import com.instabug.library.screenshot.ScreenshotManager
import com.instabug.library.screenshot.instacapture.ScreenShotType.Companion.EXTRA
import com.instabug.library.screenshot.instacapture.ScreenShotType.Companion.INITIAL
import com.instabug.library.screenshot.instacapture.ScreenShotType.Companion.VISUAL_USER_STEP_SCREENSHOT
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogVerbose

@IntDef(value = [INITIAL, EXTRA, VISUAL_USER_STEP_SCREENSHOT])
@Retention(AnnotationRetention.SOURCE)
annotation class ScreenShotType {
    companion object {
        const val INITIAL = 0
        const val EXTRA = 1
        const val VISUAL_USER_STEP_SCREENSHOT = 2
    }
}

/**
 * an interface which will be implemented by different Strategies to capture screenshots
 * */
interface CapturingStrategy {
    fun captureScreenshot(
        activity: Activity, callback: ScreenshotCaptor.CapturingCallback
    )

    companion object Factory {

        fun fromScreenshotType(@ScreenShotType type: Int): CapturingStrategy =
            MediaProjectionCapturingStrategy
                .fromScreenshotType(type)
                ?: RxCapturingStrategy
    }
}


abstract class MediaProjectionCapturingStrategy : CapturingStrategy {
    companion object Factory {
        private val isMediaProjectionEnabled
            get() = SettingsManager.getInstance().isScreenshotByMediaProjectionEnabled
        /**
         *handle media projection screenshot capturing
         *@param type is the screenshot type {Extra or Initial}
         * when the type is Initial we request the Request Permission before capturing the screenshot
         * but when capturing extra the Permission will be handled in the Bug Reporting activity or Chat activity
         * */

        fun fromScreenshotType(@ScreenShotType type: Int): CapturingStrategy? {
            return takeIf { isMediaProjectionEnabled }
                ?.let { mapTypeToCapturingStrategy(type) }
        }

        private fun mapTypeToCapturingStrategy(type: Int) = when (type) {
            INITIAL -> InitialMediaProjectionCapturingStrategy
            EXTRA -> ExtraMediaProjectionCapturingStrategy
            else -> null
        }
    }

    @WorkerThread
    override fun captureScreenshot(
        activity: Activity, callback: ScreenshotCaptor.CapturingCallback
    ) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "start capture screenshot Using MediaProjection")
        runOrLogVerbose(
            message = "something went wrong while capturing screenshot Using MediaProjection"
        ) {
            activity
                .takeUnless { it.isFinishing }
                ?.also { stopScreenRecording() }
                ?.let { capture(activity, callback) }
        }.onFailure(callback::onCapturingFailure)
    }

    abstract fun capture(
        activity: Activity,
        callback: ScreenshotCaptor.CapturingCallback
    )

    private fun stopScreenRecording() {
        if (SettingsManager.getInstance().isScreenCurrentlyRecorded) {
            AutoScreenRecordingEventBus.getInstance()
                .post(ScreenRecordingServiceAction.CustomeActions.STOP_TRIM_KEEP)
        }
    }
}

object InitialMediaProjectionCapturingStrategy : MediaProjectionCapturingStrategy() {
    override fun capture(
        activity: Activity,
        callback: ScreenshotCaptor.CapturingCallback
    ) {
        RequestPermissionActivityLauncher.start(activity, false, true, callback)
    }
}

object ExtraMediaProjectionCapturingStrategy : MediaProjectionCapturingStrategy() {
    override fun capture(
        activity: Activity,
        callback: ScreenshotCaptor.CapturingCallback
    ) {
        ScreenshotManager.getInstance().takeScreenshot(callback)
    }
}


/**
 * capture screenshot using pixel copy if android is 26 or above otherwise it will draw views
 *
 * */
object RxCapturingStrategy : CapturingStrategy {
    @WorkerThread
    override fun captureScreenshot(
        activity: Activity, callback: ScreenshotCaptor.CapturingCallback
    ) {
        runOrLogVerbose(
            message = "something went wrong while capturing screenshot using rxjava"
        ) {
            ViewsBitmapObservable.get(activity, ignoredViewsIds)?.blockingFirst()
        }.onFailure(callback::onCapturingFailure)
            .getOrNull()
            ?.let(callback::onCapturingSuccess)

    }

}
