package com.instabug.library.networkv2.service;

import android.annotation.SuppressLint;

import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.model.AssetEntity;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;

/**
 * @author mesbah
 */
public class FilesService {
    private static FilesService INSTANCE;
    private final NetworkManager networkManager;

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of FilesService
     */
    public static FilesService getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new FilesService();
        }
        return INSTANCE;
    }

    private FilesService() {
        networkManager = new NetworkManager();
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void downloadFile(final AssetEntity assetEntity, final Request.Callbacks<AssetEntity, Throwable> downloadFileCallbacks) {
        if(assetEntity != null && downloadFileCallbacks != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Downloading file request");
            // create downloadFile request.
            Request downloadFileRequest = buildDownloadFileRequest(assetEntity);

            // do request with FILE_DOWNLOAD request type.
            networkManager.doRequestOnSameThread(RequestType.FILE_DOWNLOAD, downloadFileRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    if (requestResponse != null) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "downloadFile request onNext, Response code: " + requestResponse.getResponseCode());
                        InstabugSDKLogger.v(Constants.LOG_TAG, "Response body: " + requestResponse.getResponseBody());
                        downloadFileCallbacks.onSucceeded(assetEntity);
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "downloadFile request got error: " + error.getMessage());
                    downloadFileCallbacks.onFailed(error);
                }
            });
        }
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    Request buildDownloadFileRequest(AssetEntity assetEntity) {
        String url = assetEntity.getUrl();
        File file = assetEntity.getFile();

        // create file download request
        return new Request.Builder()
                .url(url)
                .method(RequestMethod.GET)
                .fileToDownload(file)
                .hasUuid(false)
                .disableDefaultParameters(true)
                .build();
    }
}
