package com.instabug.library.networkinterception.sanitization

import android.net.Uri
import androidx.annotation.VisibleForTesting
import com.instabug.library.model.NetworkLog
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.util.collections.toQueryString
import com.instabug.library.util.extenstions.extractQueryParams
import com.instabug.library.util.extenstions.toJsonObject

@VisibleForTesting
const val MASKED_VALUE = "*****"

class AutoMaskingSanitizer(
    private val autoMaskingHeaderKeys: Set<String>,
    private val autoMaskingQueryKeys: Set<String>,
) : Sanitizer<NetworkLog> {

    override fun sanitize(item: NetworkLog): NetworkLog =
        item.apply {
            sanitizeHeaders(this)
            sanitizeQueryParams(this)
        }

    private fun sanitizeHeaders(item: NetworkLog) {
        val requestHeadersJson = item.requestHeaders.toJsonObject()
        val responseHeadersJson = item.responseHeaders.toJsonObject()

        requestHeadersJson?.keys()?.forEach {
            if (shouldSanitizeHeader(it)) {
                requestHeadersJson.put(it, MASKED_VALUE)
            }
        }

        responseHeadersJson?.keys()?.forEach {
            if (shouldSanitizeHeader(it)) {
                responseHeadersJson.put(it, MASKED_VALUE)
            }
        }

        item.requestHeaders = requestHeadersJson.toString()
        item.responseHeaders = responseHeadersJson.toString()
    }

    private fun sanitizeQueryParams(item: NetworkLog) {
        item.url?.let { url ->
            val queryParams = url.extractQueryParams()?.toMutableMap()

            val uriBuilder = Uri.parse(url).buildUpon()

            queryParams?.forEach { query ->
                if (shouldSanitizeQuery(query.key)) {
                    queryParams[query.key] = MASKED_VALUE
                }
            }

            if (queryParams?.isNotEmpty() == true) {
                uriBuilder.encodedQuery(queryParams.toQueryString())
                item.url =
                    uriBuilder.build().toString()
            }
        }
    }

    private fun shouldSanitizeHeader(key: String): Boolean = autoMaskingHeaderKeys.contains(key.lowercase())

    private fun shouldSanitizeQuery(key: String): Boolean = autoMaskingQueryKeys.contains(key.lowercase())
}
