package com.instabug.library.model;

import static com.instabug.library.featuresflags.constants.Constants.KEY;
import static com.instabug.library.featuresflags.constants.Constants.VALUE;
import static com.instabug.library.tracking.InstabugTrackingStepsProvider.USER_STEPS_LIMIT;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.AppLaunchIDProvider;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.datahub.HubReportModifier;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.experiments.di.ServiceLocator;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.featuresflags.model.IBGFeatureFlag;
import com.instabug.library.internal.filestore.ActiveCurrentSpanSelector;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributesDbHelper;
import com.instabug.library.internal.storage.operation.ReadStateFromFileDiskOperation;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.model.v3Session.IBGInMemorySession;
import com.instabug.library.model.v3Session.IBGSessionMapper;
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils;
import com.instabug.library.sessionprofiler.SessionProfiler;
import com.instabug.library.sessionprofiler.model.timeline.SessionProfilerTimeline;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.settings.PerSessionSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.user.UserEvent;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.memory.MemoryUtils;
import com.instabug.library.visualusersteps.VisualUserStep;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

;

/**
 * @author mesbah.
 */
public class State implements Cacheable, Serializable {

    // keys used by toJson() & fromJson() methods
    public static final String KEY_APP_TOKEN = "application_token";
    public static final String KEY_APP_PACKAGE_NAME = "bundle_id";
    public static final String KEY_APP_VERSION = "app_version";
    private static final String KEY_BATTERY_LEVEL = "battery_level";
    private static final String KEY_BATTERY_STATUS = "battery_state";
    public static final String KEY_CARRIER = "carrier";
    public static final String KEY_CONSOLE_LOG = "console_log";
    public static final String KEY_CURRENT_VIEW = "current_view";
    public static final String KEY_DENSITY = "density";
    public static final String KEY_DEVICE = "device";
    public static final String KEY_DEVICE_ROOTED = "device_rooted";
    public static final String KEY_DURATION = "duration";
    public static final String KEY_EMAIL = "email";
    public static final String KEY_NAME = "name";
    public static final String KEY_PUSH_TOKEN = "push_token";
    public static final String KEY_INSTABUG_LOG = "instabug_log";
    public static final String KEY_LOCALE = "locale";
    private static final String KEY_MEMORY_FREE = "memory_free";
    private static final String KEY_MEMORY_TOTAL = "memory_total";
    private static final String KEY_MEMORY_USED = "memory_used";
    public static final String KEY_ORIENTATION = "orientation";
    public static final String KEY_OS = "os";
    public static final String KEY_REPORTED_AT = "reported_at";
    public static final String KEY_SCREEN_SIZE = "screen_size";
    public static final String KEY_SDK_VERSION = "sdk_version";
    public static final String KEY_STORAGE_FREE = "storage_free";
    public static final String KEY_STORAGE_TOTAL = "storage_total";
    public static final String KEY_STORAGE_USED = "storage_used";
    public static final String KEY_TAGS = "tags";
    public static final String KEY_USER_DATA = "user_data";
    public static final String KEY_USER_STEPS = "user_steps";
    public static final String KEY_WIFI_STATE = "wifi_state";
    public static final String KEY_USER_ATTRIBUTES = "user_attributes";
    public static final String KEY_NETWORK_LOGS = "network_log";
    public static final String KEY_USER_EVENTS = "user_events";
    public static final String KEY_VISUAL_USER_STEPS = "user_repro_steps";
    public static final String KEY_SESSIONS_PROFILER = "sessions_profiler";
    private static final String KEY_APP_STATUS = "app_status";
    public static final String KEY_EXPERIMENTS = "experiments";
    public static final String KEY_DEVICE_ARCHITECTURE = "device_architecture";
    public static final String UUID = "UUID";
    public static final String KEY_CURRENT_ACTIVITY = "current_activity";
    public static final String KEY_ACTIVITY_NAME = "activity_name";
    private static final String ACTIVITY_NAME_FALLBACK = "NA";
    private static final String CURRENT_VIEW_FALLBACK = "NA";
    private static final String[] USER_DATA_KEYS = {KEY_USER_ATTRIBUTES, KEY_EMAIL, KEY_NAME, KEY_PUSH_TOKEN};
    private static final String BUILD_PERCENTAGE = "build_percentage";

    private static final String APP_LAUNCH_ID = "app_launch_id";

    private static final String DEVICE_PERFORMANCE_CLASS = "dv_performance_class";
    public static final String TRIMMING_PERCENTAGE = "trimming_percentage";

    private static final String SESSION_ID = "session_id";

    private static final String ELIGIBLE_FOR_SCREENSHOTS = "eligible_for_screenshots";
    public static final String REPRO_CONFIGS = "repro_configs";

    protected static final float FULL_STATE_PERCENTAGE = 1.0f;

    public static final String KEY_FEATURES_FLAGS = "ff";
    public static final String VALUE_APP_STATUS_FOREGROUND = "foreground";
    @VisibleForTesting
    static final String VALUE_APP_STATUS_BACKGROUND = "background";

    public static final String PRODUCT_ANALYTICS_DATA_KEY = "ra";

    private long duration;
    private boolean isDeviceRooted;
    private int batteryLevel;
    private boolean wifiState;
    private long usedMemory;
    private long freeMemory;
    private long totalMemory;
    private long usedStorage;
    public long freeStorage;
    private long totalStorage;
    @Nullable
    private String sdkVersion;
    @Nullable
    private String locale;
    @Nullable
    private String device;
    @Nullable
    private String OS;
    @Nullable
    private String carrier;
    @Nullable
    private String appPackageName;
    @Nullable
    private String appVersion;
    @Nullable
    private String batteryState;
    @Nullable
    private String screenDensity;
    @Nullable
    private String screenSize;
    @Nullable
    private String ScreenOrientation;
    @Nullable
    private String currentView;
    @Nullable
    private String instabugLog;
    @Nullable
    private List<String> consoleLog;
    @Nullable
    private List<UserStep> userSteps;
    @Nullable
    private ArrayList<VisualUserStep> visualUserSteps;
    @Nullable
    private SessionProfilerTimeline sessionProfilerTimeline;
    @Nullable
    private String userEmail;
    @Nullable
    private String userName;
    @Nullable
    private String pushToken;
    @Nullable
    private String userData;
    private long reportedAt;
    @Nullable
    private String tags;
    @Nullable
    private String networkLogs;
    @Nullable
    private String userAttributes;
    @Nullable
    private String userEvents;
    @Nullable
    private Uri uri;
    @Nullable
    private String appStatus;
    @Nullable
    private List<String> experiments;

    @Nullable
    private List<IBGFeatureFlag> featureFlags;
    @Nullable
    private String deviceArchitecture;
    @Nullable
    private String uuid;
    @Nullable
    private String currentActivity;
    private boolean isMinimalState;
    private float buildPercentage = FULL_STATE_PERCENTAGE;
    @Nullable
    private String appToken;

    @Nullable
    private String appLaunchId;

    @Nullable
    private float trimmingPercentage = 0;

    @Nullable
    String sessionId;

    @Nullable
    String screenShotAnalytics;


    private boolean isEligibleForScreenshots = true;

    @Nullable
    private HashMap<Integer, Integer> reproConfigurationsMap;

    private int devicePerformanceClass = -1;

    public static State getState(Context context) {
        return getState(context, FULL_STATE_PERCENTAGE);
    }

    public static State getState(Context context, float percentage) {
        return new State.Builder(context)
                .build(true, true, true, percentage);
    }

    public static State getNonChangingState(Context context) {
        State state = new State()
                .setSdkVersion(DeviceStateProvider.getSdkVersion())
                .setLocale(DeviceStateProvider.getLocale(context))
                .setDevice(DeviceStateProvider.getDevice())
                .setIsDeviceRooted(DeviceStateProvider.isDeviceRooted())
                .setOS(DeviceStateProvider.getOS())
                .setAppVersion(DeviceStateProvider.getAppVersion(context))
                .setAppPackageName(DeviceStateProvider.getAppPackageName(context))
                .setScreenDensity(DeviceStateProvider.getScreenDensity(context))
                .setScreenSize(DeviceStateProvider.getScreenSize(context))
                .setCurrentView(CURRENT_VIEW_FALLBACK)
                .setCurrentActivity(ACTIVITY_NAME_FALLBACK)
                .setReportedAt(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds())
                .setDeviceArchitecture(DeviceStateProvider.getDeviceArchitecture())
                .setUuid(UserManager.getUUID())
                .setAppToken(TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken())
                .setDevicePerformanceClass(SettingsManager.getInstance().getDevicePerformanceClass());
        state.isMinimalState = true;
        return state;
    }

    @WorkerThread
    public static State getState(Context context, @Nullable Uri uri) {
        try {
            if (uri != null) {
                String stateJson = DiskUtils.with(context)
                        .readOperation(new ReadStateFromFileDiskOperation(uri))
                        .execute();
                String trimmedJsonState = stateJson.trim();
                if (!trimmedJsonState.equals("{}") && !trimmedJsonState.isEmpty()) {
                    State state = new State();
                    state.setUri(uri);
                    state.fromJson(stateJson);
                    return state;
                }
            }
        } catch (OutOfMemoryError | Exception e) {
            InstabugCore.reportError(e, "retrieving state throws an exception: " + e.getMessage() + ", falling back to non-changing");
            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving state throws an exception, falling back to non-changing", e);
        }
        State nonChangingState = getNonChangingState(context);
        nonChangingState.setUri(uri);
        return nonChangingState;
    }

    @Nullable
    public String getSdkVersion() {
        return sdkVersion;
    }

    private State setSdkVersion(String sdkVersion) {
        this.sdkVersion = sdkVersion;
        return this;
    }

    @Nullable
    public String getLocale() {
        return locale;
    }

    public State setLocale(String locale) {
        this.locale = locale;
        return this;
    }

    public long getDuration() {
        return duration;
    }

    public State setDuration(long duration) {
        this.duration = duration;
        return this;
    }

    @Nullable
    public String getDevice() {
        return device;
    }

    public State setDevice(String device) {
        this.device = device;
        return this;
    }

    public boolean isDeviceRooted() {
        return isDeviceRooted;
    }

    private State setIsDeviceRooted(boolean isDeviceRooted) {
        this.isDeviceRooted = isDeviceRooted;
        return this;
    }


    @Nullable
    public String getOS() {
        return OS;
    }

    public State setOS(String OS) {
        this.OS = OS;
        return this;
    }

    @Nullable
    public String getCarrier() {
        return carrier;
    }

    public State setCarrier(String carrier) {
        this.carrier = carrier;
        return this;
    }

    @Nullable
    public String getAppPackageName() {
        return appPackageName;
    }

    public State setAppPackageName(String appPackageName) {
        this.appPackageName = appPackageName;
        return this;
    }

    @Nullable
    public String getAppVersion() {
        return appVersion;
    }

    public State setAppVersion(String appVersion) {
        this.appVersion = appVersion;
        return this;
    }

    public int getBatteryLevel() {
        return batteryLevel;
    }

    @Nullable
    public Uri getUri() {
        return uri;
    }

    public void setUri(@Nullable Uri uri) {
        this.uri = uri;
    }

    public State setBatteryLevel(int batteryLevel) {
        this.batteryLevel = batteryLevel;
        return this;
    }

    @Nullable
    public String getBatteryState() {
        return batteryState;
    }

    public State setBatteryState(String batteryState) {
        this.batteryState = batteryState;
        return this;
    }

    public boolean isWifiEnable() {
        return wifiState;
    }

    private State setWifiState(boolean wifiState) {
        this.wifiState = wifiState;
        return this;
    }


    public long getUsedMemory() {
        return usedMemory;
    }

    private State setUsedMemory(long usedMemory) {
        this.usedMemory = usedMemory;
        return this;
    }

    public long getFreeMemory() {
        return freeMemory;
    }

    public State setFreeMemory(long freeMemory) {
        this.freeMemory = freeMemory;
        return this;
    }

    public long getTotalMemory() {
        return totalMemory;
    }

    public State setTotalMemory(long totalMemory) {
        this.totalMemory = totalMemory;
        return this;
    }

    public long getUsedStorage() {
        return usedStorage;
    }

    private State setUsedStorage(long usedStorage) {
        this.usedStorage = usedStorage;
        return this;
    }

    private long getFreeStorage() {
        return freeStorage;
    }

    private State setFreeStorage(long freeStorage) {
        this.freeStorage = freeStorage;
        return this;
    }

    public long getTotalStorage() {
        return totalStorage;
    }

    private State setTotalStorage(long totalStorage) {
        this.totalStorage = totalStorage;
        return this;
    }

    @Nullable
    public String getScreenDensity() {
        return screenDensity;
    }

    public State setScreenDensity(String screenDensity) {
        this.screenDensity = screenDensity;
        return this;
    }

    @Nullable
    public String getScreenSize() {
        return screenSize;
    }

    public State setScreenSize(String screenSize) {
        this.screenSize = screenSize;
        return this;
    }

    @Nullable
    public String getScreenOrientation() {
        return ScreenOrientation;
    }

    public State setScreenOrientation(String screenOrientation) {
        ScreenOrientation = screenOrientation;
        return this;
    }

    @Nullable
    public String getCurrentView() {
        return currentView;
    }

    public State setCurrentView(String currentView) {
        this.currentView = currentView;
        return this;
    }

    @Nullable
    public String getAppStatus() {
        return appStatus;
    }

    private State setAppStatus(String appStatus) {
        this.appStatus = appStatus;
        return this;
    }

    public State setScreenShotAnalytics(String screenShotAnalytics) {
        this.screenShotAnalytics = screenShotAnalytics;
        return this;
    }

    @Nullable
    public String getScreenShotAnalytics() {
        return screenShotAnalytics;
    }

    public State setAppStatusToBackground() {
        setAppStatus(VALUE_APP_STATUS_BACKGROUND);
        return this;
    }

    public State setAppStatusToForeground() {
        setAppStatus(VALUE_APP_STATUS_FOREGROUND);
        return this;
    }

    @Nullable
    public HashMap<Integer, Integer> getReproConfigurationsMap() {
        return reproConfigurationsMap;
    }


    public void setReproConfigurationsMap(@Nullable HashMap<Integer, Integer> reproConfigurationsMap) {
        this.reproConfigurationsMap = reproConfigurationsMap;
    }

    @Nullable
    public String getInstabugLog() {
        return instabugLog;
    }

    public void setInstabugLog(@Nullable String instabugLog) {
        this.instabugLog = instabugLog;
    }

    public void updateConsoleLog() {
        setConsoleLog(Builder.getConsoleLog(buildPercentage));
    }

    public void updateConsoleLog(ArrayList<ConsoleLog> logs) {
        List<String> consoleLogs = Builder.getConsoleLog(buildPercentage);
        if (consoleLogs == null) {
            consoleLogs = new LinkedList<>();

        }
        for (ConsoleLog log : logs) {
            try {
                consoleLogs.add(log.toJson());
            } catch (Throwable e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't add user console logs", e);
            }
        }
        setConsoleLog(consoleLogs);
    }


    public JSONArray getConsoleLog() {
        try {
            if (consoleLog != null)
                return new JSONArray(consoleLog);
        } catch (Throwable t) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't add user console logs", t);
            IBGDiagnostics.reportNonFatal(t, "couldn't add user console logs");
        }
        return new JSONArray();
    }

    public State appendConsoleLogs(@Nullable List<String> extraConsoleLogs) {
        List<String> localConsoleLogs;
        if (consoleLog == null) {
            localConsoleLogs = new ArrayList<>();
        } else {
            localConsoleLogs = consoleLog;
        }
        if (extraConsoleLogs != null) {
            localConsoleLogs.addAll(extraConsoleLogs);
        }
        consoleLog = localConsoleLogs;
        return this;
    }

    public void addConsoleLogs(@Nullable List<ConsoleLog> extraLogs) {
        if(extraLogs != null) {
            List<String> localConsoleLogs;
            if (consoleLog == null) {
                localConsoleLogs = new ArrayList<>();
            } else {
                localConsoleLogs = consoleLog;
            }
            for (ConsoleLog currentConsoleLog : extraLogs) {
                try {
                    localConsoleLogs.add(currentConsoleLog.toJson());
                } catch (JSONException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "couldn't add user console logs", e);

                }
            }
            consoleLog = localConsoleLogs;
        }
    }

    public State setConsoleLog(@Nullable List<String> consoleLogs) {
        this.consoleLog = consoleLogs;
        return this;
    }

    public void clearConsoleLogs() {
        if(consoleLog != null) {
            try {
                consoleLog.clear();
            } catch (UnsupportedOperationException exception) {
                consoleLog = new ArrayList<>();
            }
        }
    }

    public JSONArray getUserSteps() {
        return UserStep.toJson(userSteps);
    }

    public void clearUserSteps() {
        if(userSteps != null) {
            try {
                userSteps.clear();
            } catch (UnsupportedOperationException exception) {
                userSteps = new ArrayList<>();
            }
        }
    }

    @VisibleForTesting
    public State setUserSteps(List<UserStep> userSteps) {
        this.userSteps = userSteps;
        return this;
    }

    @Nullable
    public String getUserEmail() {
        return userEmail;
    }

    public State setUserEmail(String userEmail) {
        this.userEmail = userEmail;
        return this;
    }

    @Nullable
    public String getUserName() {
        return userName;
    }

    public State setUserName(@Nullable String userName) {
        this.userName = userName;
        return this;
    }

    public State updateIdentificationAttrs() {
        if (userEmail == null || userEmail.isEmpty()) {
            setUserEmail(UserManager.getUserEmail());
        }
        if (userName == null || userName.isEmpty()) {
            setUserName(UserManager.getUserName());
        }
        return this;
    }

    @Nullable
    public String getPushToken() {
        return pushToken;
    }

    public State setPushToken(@Nullable String pushToken) {
        this.pushToken = pushToken;
        return this;
    }

    @Nullable
    public String getUserData() {
        return userData;
    }

    public State setUserData(String userData) {
        this.userData = userData;
        return this;
    }

    public long getReportedAt() {
        return reportedAt;
    }

    private State setReportedAt(long reportedAt) {
        this.reportedAt = reportedAt;
        return this;
    }

    @Nullable
    public String getTags() {
        //to get the latest Tags added.
        return tags;
    }

    public State setTags(String tags) {
        this.tags = tags;
        return this;
    }

    private static String getTagsAsString(List<String> tags) {
        // Tags format: {"tags" : "tag1, tag2, tag3"}
        StringBuilder sb = new StringBuilder();

        if (tags != null && tags.size() > 0) {
            int size = tags.size();
            for (int i = 0; i < size; i++) {
                sb.append(tags.get(i));
                if (i != size - 1)
                    sb.append(", ");
            }
        }
        return sb.toString();
    }

    public State setTags(List<String> tags) {
        this.tags = getTagsAsString(tags);
        return this;
    }

    @Nullable
    public String getUserAttributes() {
        return userAttributes;
    }

    public State setUserAttributes(String userAttributes) {
        this.userAttributes = userAttributes;
        return this;
    }

    @Nullable
    public String getNetworkLogs() {
        return networkLogs;
    }

    public State setNetworkLogs(@Nullable String networkLogs) {
        this.networkLogs = networkLogs;
        return this;
    }

    @Nullable
    public String getUserEvents() {
        return userEvents;
    }

    public State setUserEvents(String userEvents) {
        this.userEvents = userEvents;
        return this;
    }

    public void updateUserEvents() throws JSONException {
        setUserEvents(UserEvent.toJson(InstabugUserEventLogger.getInstance().getUserEvents(buildPercentage))
                .toString());
    }

    public String getVisualUserSteps() {
        return VisualUserStep.toJsonString(visualUserSteps);
    }

    public void setVisualUserSteps(@Nullable ArrayList<VisualUserStep> visualUserSteps) {
        this.visualUserSteps = visualUserSteps;
    }

    @Nullable
    @VisibleForTesting
    public String getSessionProfilerTimeline() {
        return sessionProfilerTimeline == null ? null : sessionProfilerTimeline.toJson().toString();
    }

    public State setSessionProfilerTimeline(@Nullable SessionProfilerTimeline sessionProfilerTimeline) {
        this.sessionProfilerTimeline = sessionProfilerTimeline;
        return this;
    }

    public State setExperiments(@Nullable List<String> experiments) {
        this.experiments = experiments;
        return this;
    }

    public State setFeaturesFlags(@Nullable List<IBGFeatureFlag> featureFlags) {
        this.featureFlags = featureFlags;
        return this;
    }

    public State setDeviceArchitecture(@Nullable String deviceArchitecture) {
        this.deviceArchitecture = deviceArchitecture;
        return this;
    }

    public State setUuid(@Nullable String uuid) {
        this.uuid = uuid;
        return this;
    }

    @NonNull
    public String getCurrentActivityName() {
        return currentActivity != null ? currentActivity : ACTIVITY_NAME_FALLBACK;
    }

    @Nullable
    public String getUuid() {
        return uuid;
    }

    @Nullable
    public List<String> getExperiments() {
        return experiments;
    }

    @Nullable
    public List<IBGFeatureFlag> getFeatureFlags() {
        return featureFlags;
    }

    public List<StateItem<?>> getStateItems() {
        return getStateItems(true);
    }

    public List<StateItem<?>> getStateItems(boolean userIdentificationEnabled) {
        ArrayList<StateItem<?>> stateItems = new ArrayList<>();
        if (!isMinimalState) {
            stateItems.add(new StateItem<Integer>()
                    .setKey(KEY_BATTERY_LEVEL).setValue(getBatteryLevel()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_BATTERY_STATUS).setValue(getBatteryState()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_CARRIER).setValue(getCarrier()));
            if (userIdentificationEnabled) {
                stateItems.add(new StateItem<String>()
                        .setKey(KEY_EMAIL).setValue(getUserEmail()));
                stateItems.add(new StateItem<String>()
                        .setKey(KEY_NAME).setValue(getUserName()));
            }
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_PUSH_TOKEN).setValue(getPushToken()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_MEMORY_FREE).setValue(getFreeMemory()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_MEMORY_TOTAL).setValue(getTotalMemory()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_MEMORY_USED).setValue(getUsedMemory()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_ORIENTATION).setValue(getScreenOrientation()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_STORAGE_FREE).setValue(getFreeStorage()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_STORAGE_TOTAL).setValue(getTotalStorage()));
            stateItems.add(new StateItem<Long>()
                    .setKey(KEY_STORAGE_USED).setValue(getUsedStorage()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_TAGS).setValue(getTags()));
            stateItems.add(new StateItem<Boolean>()
                    .setKey(KEY_WIFI_STATE).setValue(isWifiEnable()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_USER_ATTRIBUTES).setValue(getUserAttributes()));
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_APP_STATUS).setValue(getAppStatus()));
            List<String> experiments = getExperiments();
            if (experiments != null && !experiments.isEmpty()) {
                JSONArray experimentsJsonArray = new JSONArray();
                for (String experiment : experiments) {
                    experimentsJsonArray.put(experiment);
                }
                stateItems.add(new StateItem<JSONArray>()
                        .setKey(KEY_EXPERIMENTS)
                        .setValue(experimentsJsonArray));
            }
        }

        List<IBGFeatureFlag> featureFlagList = getFeatureFlags();
        if (featureFlagList != null && !featureFlagList.isEmpty()) {
            JSONArray featureFlagsJsonArray = new JSONArray();
            for (IBGFeatureFlag ibgFeatureFlag : featureFlagList) {
                try {
                    JSONObject featureFlagJsonObject = new JSONObject();
                    featureFlagJsonObject.put(KEY, ibgFeatureFlag.getKey());
                    featureFlagJsonObject.put(VALUE,ibgFeatureFlag.getValue());
                    featureFlagsJsonArray.put(featureFlagJsonObject);
                } catch (Exception e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "something went wrong while feature flag object added");
                }

            }
            stateItems.add(new StateItem<JSONArray>()
                    .setKey(KEY_FEATURES_FLAGS)
                    .setValue(featureFlagsJsonArray));
        }

        stateItems.add(new StateItem<String>()
                .setKey(KEY_ACTIVITY_NAME).setValue(getCurrentActivityName()));

        stateItems.add(new StateItem<String>()
                .setKey(KEY_APP_PACKAGE_NAME).setValue(getAppPackageName()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_APP_VERSION).setValue(getAppVersion()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_CURRENT_VIEW).setValue(getCurrentView()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_DENSITY).setValue(getScreenDensity()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_DEVICE).setValue(getDevice()));
        stateItems.add(new StateItem<Boolean>()
                .setKey(KEY_DEVICE_ROOTED).setValue(isDeviceRooted()));
        stateItems.add(new StateItem<Long>()
                .setKey(KEY_DURATION).setValue(getDuration()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_LOCALE).setValue(getLocale()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_OS).setValue(getOS()));
        stateItems.add(new StateItem<Long>()
                .setKey(KEY_REPORTED_AT).setValue(getReportedAt()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_SCREEN_SIZE).setValue(getScreenSize()));
        stateItems.add(new StateItem<String>()
                .setKey(KEY_SDK_VERSION).setValue(getSdkVersion()));


        if (getDevicePerformanceClass() > DevicePerformanceClassUtils.PERFORMANCE_CLASS_UNDEFINED) {
            stateItems.add(new StateItem<Integer>()
                    .setKey(DEVICE_PERFORMANCE_CLASS).setValue(getDevicePerformanceClass()));
        }

        if (getTrimmingPercentage() > 0) {
            stateItems.add(new StateItem<Float>()
                    .setKey(TRIMMING_PERCENTAGE).setValue(getTrimmingPercentage()));
        }

        String deviceArchitecture = getDeviceArchitecture();
        if (deviceArchitecture != null && !deviceArchitecture.isEmpty()) {
            stateItems.add(new StateItem<String>()
                    .setKey(KEY_DEVICE_ARCHITECTURE)
                    .setValue(deviceArchitecture));
        }
        if (screenShotAnalytics != null) {
            stateItems.add(new StateItem<JSONObject>().setKey(PRODUCT_ANALYTICS_DATA_KEY).setValue(convertScreenShotAnalyticsToJson(getScreenShotAnalytics())));
        }
        if (sessionId != null) {
            stateItems.add(new StateItem<>().setKey(SESSION_ID).setValue(sessionId));
        }
        return stateItems;
    }

    public List<StateItem<?>> getEarlyStateItems() {
        List<StateItem<?>> stateItems = getStateItems(false);
        stateItems.add(new StateItem<String>()
                .setKey(KEY_APP_TOKEN).setValue(getAppToken()));
        return stateItems;
    }

    private JSONObject convertScreenShotAnalyticsToJson(@Nullable String screenShotAnalytics) {
        try {
            if (screenShotAnalytics != null) {
                return new JSONObject(screenShotAnalytics);
            }
            return new JSONObject();
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while converting screenShotAnalytics into a JSON object: " + e.getMessage());
            return new JSONObject();
        }
    }

    @Nullable
    private String getDeviceArchitecture() {
        return deviceArchitecture;
    }

    public static String[] getUserDataKeys() {
        return USER_DATA_KEYS.clone();
    }

    public double getBuildPercentage() {
        return buildPercentage;
    }

    public State setBuildPercentage(float percentage) {
        buildPercentage = percentage;
        return this;
    }

    public State setCurrentActivity(String currentActivity) {
        this.currentActivity = currentActivity;
        return this;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        try {
            JSONObject state = new JSONObject();
            List<StateItem<?>> stateItems = getStateItems();
            for (int i = 0; i < stateItems.size(); i++) {
                String key = stateItems.get(i).getKey();
                if (key != null) {
                    state.put(key, stateItems.get(i).getValue());
                }
            }
            state.put(UUID, uuid);
            ArrayList<StateItem> logsItems = getLogsItems(false);
            for (int i = 0; i < logsItems.size(); i++) {
                String key = logsItems.get(i).getKey();
                if (key != null) {
                    state.put(key, logsItems.get(i).getValue());
                }
            }
            state.put(BUILD_PERCENTAGE, buildPercentage);
            state.put(State.KEY_APP_TOKEN, appToken);
            state.put(APP_LAUNCH_ID, appLaunchId);
            state.put(DEVICE_PERFORMANCE_CLASS, devicePerformanceClass);
            state.put(ELIGIBLE_FOR_SCREENSHOTS, isEligibleForScreenshots);
            if(reproConfigurationsMap != null) {
                state.put(REPRO_CONFIGS,mapToJson(reproConfigurationsMap));
            }
            return state.toString();
        } catch (OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Could create state json string, OOM", e);
        }
        return new JSONObject().toString();
    }

    @Override
    public void fromJson(String stateAsJson) throws JSONException {
        JSONObject stateJsonObject = new JSONObject(stateAsJson);
        if (stateJsonObject.has(KEY_APP_PACKAGE_NAME))
            setAppPackageName(stateJsonObject.getString(KEY_APP_PACKAGE_NAME));
        if (stateJsonObject.has(KEY_APP_VERSION))
            setAppVersion(stateJsonObject.getString(KEY_APP_VERSION));
        if (stateJsonObject.has(KEY_BATTERY_LEVEL))
            setBatteryLevel(stateJsonObject.getInt(KEY_BATTERY_LEVEL));
        if (stateJsonObject.has(KEY_BATTERY_STATUS))
            setBatteryState(stateJsonObject.getString(KEY_BATTERY_STATUS));
        if (stateJsonObject.has(KEY_CARRIER))
            setCarrier(stateJsonObject.getString(KEY_CARRIER));
        if (stateJsonObject.has(KEY_CONSOLE_LOG))
            retrieveConsoleLogs(stateJsonObject);
        if (stateJsonObject.has(KEY_CURRENT_VIEW))
            setCurrentView(stateJsonObject.getString(KEY_CURRENT_VIEW));
        if (stateJsonObject.has(KEY_DENSITY))
            setScreenDensity(stateJsonObject.getString(KEY_DENSITY));
        if (stateJsonObject.has(KEY_DEVICE))
            setDevice(stateJsonObject.getString(KEY_DEVICE));
        if (stateJsonObject.has(KEY_DEVICE_ROOTED))
            setIsDeviceRooted(stateJsonObject.getBoolean(KEY_DEVICE_ROOTED));
        if (stateJsonObject.has(KEY_DURATION))
            setDuration(stateJsonObject.getLong(KEY_DURATION));
        if (stateJsonObject.has(KEY_EMAIL))
            setUserEmail(stateJsonObject.getString(KEY_EMAIL));
        if (stateJsonObject.has(KEY_NAME)) {
            setUserName(stateJsonObject.getString(KEY_NAME));
        }
        if (stateJsonObject.has(KEY_PUSH_TOKEN)) {
            setPushToken(stateJsonObject.getString(KEY_PUSH_TOKEN));
        }
        if (stateJsonObject.has(KEY_INSTABUG_LOG))
            setInstabugLog(stateJsonObject.getString(KEY_INSTABUG_LOG));
        if (stateJsonObject.has(KEY_LOCALE))
            setLocale(stateJsonObject.getString(KEY_LOCALE));
        if (stateJsonObject.has(KEY_MEMORY_FREE))
            setFreeMemory(stateJsonObject.getLong(KEY_MEMORY_FREE));
        if (stateJsonObject.has(KEY_MEMORY_TOTAL))
            setTotalMemory(stateJsonObject.getLong(KEY_MEMORY_TOTAL));
        if (stateJsonObject.has(KEY_MEMORY_USED))
            setUsedMemory(stateJsonObject.getLong(KEY_MEMORY_USED));
        if (stateJsonObject.has(KEY_ORIENTATION))
            setScreenOrientation(stateJsonObject.getString(KEY_ORIENTATION));
        if (stateJsonObject.has(KEY_OS))
            setOS(stateJsonObject.getString(KEY_OS));
        if (stateJsonObject.has(KEY_APP_STATUS))
            setAppStatus(stateJsonObject.getString(KEY_APP_STATUS));
        if (stateJsonObject.has(KEY_REPORTED_AT))
            setReportedAt(stateJsonObject.getLong(KEY_REPORTED_AT));
        if (stateJsonObject.has(KEY_SCREEN_SIZE))
            setScreenSize(stateJsonObject.getString(KEY_SCREEN_SIZE));
        if (stateJsonObject.has(KEY_SDK_VERSION))
            setSdkVersion(stateJsonObject.getString(KEY_SDK_VERSION));
        if (stateJsonObject.has(KEY_STORAGE_FREE))
            setFreeStorage(stateJsonObject.getLong(KEY_STORAGE_FREE));
        if (stateJsonObject.has(KEY_STORAGE_TOTAL))
            setTotalStorage(stateJsonObject.getLong(KEY_STORAGE_TOTAL));
        if (stateJsonObject.has(KEY_STORAGE_USED))
            setUsedStorage(stateJsonObject.getLong(KEY_STORAGE_USED));
        if (stateJsonObject.has(KEY_TAGS))
            setTags(stateJsonObject.getString(KEY_TAGS));
        if (stateJsonObject.has(KEY_USER_DATA))
            setUserData(stateJsonObject.getString(KEY_USER_DATA));
        if (stateJsonObject.has(KEY_USER_STEPS))
            setUserSteps(UserStep.fromJson(new JSONArray(stateJsonObject.getString
                    (KEY_USER_STEPS))));

        if (stateJsonObject.has(KEY_WIFI_STATE))
            setWifiState(stateJsonObject.getBoolean(KEY_WIFI_STATE));
        if (stateJsonObject.has(KEY_USER_ATTRIBUTES))
            setUserAttributes(stateJsonObject.getString(KEY_USER_ATTRIBUTES));
        if (stateJsonObject.has(KEY_NETWORK_LOGS))
            setNetworkLogs(stateJsonObject.getString(KEY_NETWORK_LOGS));
        if (stateJsonObject.has(KEY_USER_EVENTS))
            setUserEvents(stateJsonObject.getString(KEY_USER_EVENTS));
        if (stateJsonObject.has(KEY_VISUAL_USER_STEPS)) {
            setVisualUserSteps(VisualUserStep.fromJson(new JSONArray(stateJsonObject.getString
                    (KEY_VISUAL_USER_STEPS))));
        }
        if (stateJsonObject.has(KEY_SESSIONS_PROFILER)) {
            setSessionProfilerTimeline(SessionProfilerTimeline
                    .fromJson(new JSONObject(stateJsonObject.getString(KEY_SESSIONS_PROFILER))));
        }
        if (stateJsonObject.has(KEY_EXPERIMENTS)) {
            JSONArray experimentsJsonArray = stateJsonObject.getJSONArray(KEY_EXPERIMENTS);
            ArrayList<String> experiments = new ArrayList<>();
            for (int i = 0; i < experimentsJsonArray.length(); i++) {
                experiments.add((String) experimentsJsonArray.get(i));
            }
            setExperiments(experiments);

        }

        if (stateJsonObject.has(KEY_FEATURES_FLAGS)) {
            JSONArray featuresFlagsJsonArray = stateJsonObject.getJSONArray(KEY_FEATURES_FLAGS);
            ArrayList<IBGFeatureFlag> featureFlags = new ArrayList<>();
            for (int i = 0; i < featuresFlagsJsonArray.length(); i++) {
                JSONObject featureFlagJson = featuresFlagsJsonArray.getJSONObject(i);
                IBGFeatureFlag featureFlag = IBGFeatureFlag.fromJson(featureFlagJson);
                featureFlags.add(featureFlag);
            }
            setFeaturesFlags(featureFlags);
        }
        if (stateJsonObject.has(BUILD_PERCENTAGE)) {
            setBuildPercentage((float) stateJsonObject.getDouble(BUILD_PERCENTAGE));
        }
        setCurrentActivity(getActivityNameFromJson(stateJsonObject));

        setDeviceArchitecture(stateJsonObject.optString(KEY_DEVICE_ARCHITECTURE));
        setUuid(stateJsonObject.optString(uuid));

        if (stateJsonObject.has(KEY_APP_TOKEN)) {
            setAppToken(stateJsonObject.getString(KEY_APP_TOKEN));
        }
        if (stateJsonObject.has(APP_LAUNCH_ID)) {
            setAppLaunchId(stateJsonObject.getString(APP_LAUNCH_ID));
        }
        if (stateJsonObject.has(DEVICE_PERFORMANCE_CLASS)) {
            setDevicePerformanceClass(stateJsonObject.getInt(DEVICE_PERFORMANCE_CLASS));
        }

        if (stateJsonObject.has(TRIMMING_PERCENTAGE)) {
            setTrimmingPercentage((float) stateJsonObject.getDouble(TRIMMING_PERCENTAGE));
        }
        if (stateJsonObject.has(SESSION_ID)) {
            setSessionId(stateJsonObject.optString(SESSION_ID));
        }
        if (stateJsonObject.has(ELIGIBLE_FOR_SCREENSHOTS)) {
            setEligibleForScreenshots(stateJsonObject.optBoolean(ELIGIBLE_FOR_SCREENSHOTS));
        }

        if(stateJsonObject.has(REPRO_CONFIGS)) {
            setReproConfigurationsMap(jsonToReproMap(stateJsonObject.optJSONObject(REPRO_CONFIGS)));
        }
        if (stateJsonObject.has(PRODUCT_ANALYTICS_DATA_KEY))
            setScreenShotAnalytics(stateJsonObject.getString(PRODUCT_ANALYTICS_DATA_KEY));
    }


    @NonNull
    private String getActivityNameFromJson(JSONObject stateJson) {
        if (stateJson.has(KEY_ACTIVITY_NAME))
            return stateJson.optString(KEY_ACTIVITY_NAME, ACTIVITY_NAME_FALLBACK);

        if (stateJson.has(KEY_CURRENT_ACTIVITY))
            return stateJson.optString(KEY_CURRENT_ACTIVITY, ACTIVITY_NAME_FALLBACK);

        return ACTIVITY_NAME_FALLBACK;
    }

    private void retrieveConsoleLogs(JSONObject stateJsonObject) throws JSONException {
        JSONArray consoleLogs = new JSONArray(stateJsonObject.getString(KEY_CONSOLE_LOG));
        LinkedList<String> logs = new LinkedList<>();
        for (int i = 0; i < consoleLogs.length(); i++) {
            logs.add(consoleLogs.optString(i));
        }
        setConsoleLog(logs);
    }

    public ArrayList<StateItem> getLogsItems() {
        return getLogsItems(true);
    }

    private String getConsoleLogsForSync() {
        JSONArray logs = getConsoleLog();
        ConsoleLogsMapper.updateLogs(logs);
        return logs.toString();
    }

    private ArrayList<StateItem> getLogsItems(boolean forSync) {
        ArrayList<StateItem> logsItems = new ArrayList<>();
        addConsoleLogsItem(logsItems, forSync);
        logsItems.add(new StateItem<String>()
                .setKey(KEY_INSTABUG_LOG).setValue(getInstabugLog()));
        logsItems.add(new StateItem<String>()
                .setKey(KEY_USER_DATA).setValue(getUserData()));

        logsItems.add(new StateItem<String>()
                .setKey(KEY_NETWORK_LOGS).setValue(getNetworkLogs()));
        logsItems.add(new StateItem<String>()
                .setKey(KEY_USER_EVENTS).setValue(getUserEvents()));

        if (visualUserSteps != null) {
            logsItems.add(new StateItem<String>()
                    .setKey(KEY_VISUAL_USER_STEPS).setValue(getVisualUserSteps()));
        }

        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.TRACK_USER_STEPS)
                == Feature.State.ENABLED) {
            logsItems.add(new StateItem<String>()
                    .setKey(KEY_USER_STEPS).setValue(getUserSteps().toString()));
        }
        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.SESSION_PROFILER)
                == Feature.State.ENABLED) {
            if (sessionProfilerTimeline != null) {
                logsItems.add(new StateItem<String>()
                        .setKey(KEY_SESSIONS_PROFILER).setValue(getSessionProfilerTimeline()));
            }
        }
        return logsItems;
    }

    private void addConsoleLogsItem(@NonNull ArrayList<StateItem> logsItems, boolean forSync) {
        String consoleLogs = forSync
                ? getConsoleLogsForSync()
                : getConsoleLog().toString();
        StateItem<String> logsItem = new StateItem<String>()
                .setKey(KEY_CONSOLE_LOG)
                .setValue(consoleLogs);
        logsItems.add(logsItem);
    }

    @NonNull
    @Override
    public String toString() {
        try {
            return toJson();
        } catch (JSONException e) {
            e.printStackTrace();
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while getting state.toString()" + e
                    .getMessage(), e);
            return "error";
        }
    }

    @Override
    @SuppressLint("NP_METHOD_PARAMETER_TIGHTENS_ANNOTATION")
    public boolean equals(@Nullable Object state) {
        if (state == null) {
            return false;
        } else {
            if (state instanceof State) {
                State comparedState = ((State) state);
                return String.valueOf(comparedState.getAppVersion()).equals(String.valueOf
                        (getAppVersion()))
                        && comparedState.getBatteryLevel() == getBatteryLevel()
                        && String.valueOf(comparedState.getBatteryState()).equals(String.valueOf
                        (getBatteryState()))
                        && String.valueOf(comparedState.getCarrier()).equals(String.valueOf
                        (getCarrier()))
                        && String.valueOf(comparedState.getAppStatus()).equals(String.valueOf
                        (getAppStatus()))
                        && String.valueOf(comparedState.getConsoleLog()).equals(String.valueOf
                        (getConsoleLog()))
                        && String.valueOf(comparedState.getCurrentView()).equals(String.valueOf
                        (getCurrentView()))
                        && comparedState.getDuration() == getDuration()
                        && String.valueOf(comparedState.getDevice()).equals(String.valueOf(getDevice()))
                        && comparedState.getFreeMemory() == getFreeMemory()
                        && comparedState.getFreeStorage() == getFreeStorage()
                        && String.valueOf(comparedState.getLocale()).equals(String.valueOf(getLocale()))
                        && String.valueOf(comparedState.getOS()).equals(String.valueOf(getOS()))
                        && comparedState.getReportedAt() == getReportedAt()
                        && String.valueOf(comparedState.getScreenDensity()).equals(String.valueOf
                        (getScreenDensity()))
                        && String.valueOf(comparedState.getScreenOrientation()).equals(String.valueOf
                        (getScreenOrientation()))
                        && String.valueOf(comparedState.getScreenSize()).equals(String.valueOf
                        (getScreenSize()))
                        && String.valueOf(comparedState.getSdkVersion()).equals(String.valueOf
                        (getSdkVersion()))
                        && comparedState.getTotalMemory() == getTotalMemory()
                        && comparedState.getTotalStorage() == getTotalStorage()
                        && String.valueOf(comparedState.getTags()).equals(String.valueOf(getTags()))
                        && comparedState.getUsedMemory() == getUsedMemory()
                        && comparedState.getUsedStorage() == getUsedStorage()
                        && String.valueOf(comparedState.getUserData()).equals(String.valueOf
                        (getUserData()))
                        && String.valueOf(comparedState.getUserEmail()).equals(String.valueOf
                        (getUserEmail()))
                        && String.valueOf(comparedState.getUserName()).equals(String.valueOf
                        (getUserName()))
                        && String.valueOf(comparedState.getPushToken()).equals(String.valueOf(getPushToken()))
                        && String.valueOf(comparedState.getUserSteps()).equals(String.valueOf
                        (getUserSteps()))
                        && comparedState.isDeviceRooted() == isDeviceRooted()
                        && comparedState.isWifiEnable() == isWifiEnable()
                        && String.valueOf(comparedState.getInstabugLog()).equals(String.valueOf
                        (getInstabugLog()))
                        && String.valueOf(comparedState.getUserAttributes()).equals(String.valueOf
                        (getUserAttributes()))
                        && String.valueOf(comparedState.getNetworkLogs()).equals(String.valueOf
                        (getNetworkLogs()))
                        && String.valueOf(comparedState.getUserEvents()).equals(String.valueOf
                        (getUserEvents())) && String.valueOf(comparedState.getVisualUserSteps())
                        .equals(String.valueOf(getVisualUserSteps()))
                        && String.valueOf(comparedState.getSessionProfilerTimeline())
                        .equals(String.valueOf(getSessionProfilerTimeline()))
                        && comparedState.getDevicePerformanceClass() == getDevicePerformanceClass();
            } else {
                return false;
            }
        }
    }

    @Override
    public int hashCode() {
        return String.valueOf(getReportedAt()).hashCode();
    }

    public void updateVisualUserSteps() {
        setVisualUserSteps(Builder.getVisualUserSteps());
    }

    public State setAppLaunchId(@Nullable String appLaunchId) {
        this.appLaunchId = appLaunchId;
        return this;
    }

    @Nullable
    public String getAppLaunchId() {
        return appLaunchId;
    }

    public State setSessionId(@Nullable String sessionId) {
        this.sessionId = sessionId;
        return this;
    }

    @Nullable
    public String getSessionId() {
        return sessionId;
    }

    public boolean isMinimalState() {
        return isMinimalState;
    }

    @Nullable
    public String getAppToken() {
        return appToken;
    }

    public State setAppToken(@Nullable String appToken) {
        this.appToken = appToken;
        return this;
    }

    public int getDevicePerformanceClass() {
        return devicePerformanceClass;
    }

    public State setDevicePerformanceClass(int devicePerformanceClass) {
        this.devicePerformanceClass = devicePerformanceClass;
        return this;
    }

    public State setTrimmingPercentage(float trimmingPercentage) {
        this.trimmingPercentage = trimmingPercentage;
        return this;
    }

    public float getTrimmingPercentage() {
        return trimmingPercentage;
    }

    public void updateSessionIdFromLatestSession() {
        IBGInMemorySession latestSession = InstabugCore.getLatestV3Session();
        setSessionId(composeSessionId(latestSession));
    }

    public State setEligibleForScreenshots(boolean isEligible) {
        isEligibleForScreenshots = isEligible;
        return this;
    }

    public boolean isEligibleForScreenshots() {
        return isEligibleForScreenshots;
    }

    @Keep
    public enum Action {
        FINISHED, ERROR
    }

    public static class StateItem<V> implements Serializable {

        @Nullable
        String key;
        @Nullable
        V value;

        public StateItem() {
        }

        public StateItem(@Nullable String key, @Nullable V value) {
            this.key = key;
            this.value = value;
        }

        @Nullable
        public String getKey() {
            return key;
        }

        public StateItem<V> setKey(@Nullable String key) {
            this.key = key;
            return this;
        }

        @Nullable
        public V getValue() {
            return value;
        }

        public StateItem<V> setValue(@Nullable V value) {
            this.value = value;
            return this;
        }

        @NonNull
        @Override
        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public String toString() {
            return "key: " + getKey() + ", value: " + getValue();
        }
    }

    public static class Builder implements Serializable {
        private Context context;

        public Builder(Context context) {
            this.context = context;
        }


        @Nullable
        protected static List<String> getConsoleLog(float percentage) {
            return ConsoleLog.getConsoleLogs(percentage);
        }

        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public State build(boolean withInstabugLogs) {
            return build(withInstabugLogs, false, false, FULL_STATE_PERCENTAGE);
        }

        /**
         * Build simplified state for non fatals, by removing some unneeded fields to optimize memory
         * Removed fields:
         * ConsoleLog,UserSteps,UserEmail,UserName,PushToken,UserData,Tags,UserAttributes,NetworkLogs,UserEvents,InstabugLogs,SessionProfiler,Expirements
         *
         * @return state object
         */
        public State buildSimplifiedState() {
            State state = new State()
                    .setSdkVersion(DeviceStateProvider.getSdkVersion())
                    .setLocale(DeviceStateProvider.getLocale(context))
                    .setDuration(DeviceStateProvider.getActiveSessionDuration())
                    .setDevice(DeviceStateProvider.getDevice())
                    .setIsDeviceRooted(DeviceStateProvider.isDeviceRooted())
                    .setOS(DeviceStateProvider.getOS())
                    .setCarrier(DeviceStateProvider.getCarrier(context))
                    .setAppStatus(InstabugCore.getStartedActivitiesCount() > 0 ?
                            VALUE_APP_STATUS_FOREGROUND : VALUE_APP_STATUS_BACKGROUND)
                    .setAppVersion(DeviceStateProvider.getAppVersion(context))
                    .setAppPackageName(DeviceStateProvider.getAppPackageName(context))
                    .setBatteryLevel(DeviceStateProvider.getBatteryLevel(context))
                    .setBatteryState(DeviceStateProvider.getBatteryState(context))
                    .setWifiState(DeviceStateProvider.getWifiState(context))
                    .setFreeMemory(DeviceStateProvider.getFreeMemory(context))
                    .setUsedMemory(DeviceStateProvider.getUsedMemory(context))
                    .setTotalMemory(DeviceStateProvider.getTotalMemory(context))
                    .setFreeStorage(DeviceStateProvider.getFreeStorage())
                    .setUsedStorage(DeviceStateProvider.getUsedStorage())
                    .setTotalStorage(DeviceStateProvider.getTotalStorage())
                    .setScreenDensity(DeviceStateProvider.getScreenDensity(context))
                    .setScreenSize(DeviceStateProvider.getScreenSize(context))
                    .setScreenOrientation(DeviceStateProvider.getScreenOrientation(context))
                    .setCurrentView(InstabugCore.getCurrentView())
                    .setReportedAt(getReportedAt())
                    .setDeviceArchitecture(DeviceStateProvider.getDeviceArchitecture())
                    .setUuid(UserManager.getUUID())
                    .setAppToken(TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken())
                    .setDevicePerformanceClass(SettingsManager.getInstance().getDevicePerformanceClass())
                    .setTrimmingPercentage(getTrimmingPercentage())
                    .setSessionId(getSessionId())
                    .setAppLaunchId(getAppLaunchId());

            return state;
        }

        protected State buildSimplifiedEarlyState() {

            Context localContext = context;
            State state = new State()
                    .setSdkVersion(DeviceStateProvider.getSdkVersion())
                    .setDuration(DeviceStateProvider.getActiveSessionDuration())
                    .setDevice(DeviceStateProvider.getDevice())
                    .setIsDeviceRooted(DeviceStateProvider.isDeviceRooted())
                    .setOS(DeviceStateProvider.getOS())
                    .setAppStatus(InstabugCore.getStartedActivitiesCount() > 0 ?
                            VALUE_APP_STATUS_FOREGROUND : VALUE_APP_STATUS_BACKGROUND)
                    .setFreeStorage(DeviceStateProvider.getFreeStorage())
                    .setUsedStorage(DeviceStateProvider.getUsedStorage())
                    .setTotalStorage(DeviceStateProvider.getTotalStorage())
                    .setCurrentView(InstabugCore.getCurrentView())
                    .setReportedAt(getReportedAt())
                    .setDeviceArchitecture(DeviceStateProvider.getDeviceArchitecture())
                    .setUuid(UserManager.getUUID())
                    .setDevicePerformanceClass(SettingsManager.getInstance().getDevicePerformanceClass())
                    .setTrimmingPercentage(getTrimmingPercentage())
                    .setSessionId(getSessionId());

            if (localContext != null) {
                state.setLocale(DeviceStateProvider.getLocale(localContext))
                        .setCarrier(DeviceStateProvider.getCarrier(localContext))
                        .setAppVersion(DeviceStateProvider.getAppVersion(localContext))
                        .setAppPackageName(DeviceStateProvider.getAppPackageName(localContext))
                        .setBatteryLevel(DeviceStateProvider.getBatteryLevel(localContext))
                        .setBatteryState(DeviceStateProvider.getBatteryState(localContext))
                        .setWifiState(DeviceStateProvider.getWifiState(localContext))
                        .setFreeMemory(DeviceStateProvider.getFreeMemory(localContext))
                        .setUsedMemory(DeviceStateProvider.getUsedMemory(localContext))
                        .setTotalMemory(DeviceStateProvider.getTotalMemory(localContext))
                        .setScreenDensity(DeviceStateProvider.getScreenDensity(localContext))
                        .setScreenSize(DeviceStateProvider.getScreenSize(localContext))
                        .setScreenOrientation(DeviceStateProvider.getScreenOrientation(localContext))
                        .setAppPackageName(DeviceStateProvider.getAppPackageName(localContext));
            }
            return state;
        }

        public State build(boolean withInstabugLogs, boolean withExperiments, boolean withFeatureFlag, float percentage) {
            return build(withInstabugLogs, withExperiments, withFeatureFlag, percentage, true);
        }

        public State build(boolean withInstabugLogs, boolean withExperiments, boolean withFeatureFlag, float percentage, boolean withHubData) {
            State state = buildSimplifiedState();
            HubReportModifier reportModifier = null;
            if (withHubData) {
                reportModifier = new HubReportModifier.Builder()
                        .withIBGLogs(withInstabugLogs)
                        .buildWithDefaultStores();
                reportModifier.prepare(state, new ActiveCurrentSpanSelector<>());
            }
            state.setBuildPercentage(percentage)
                    .setConsoleLog(getConsoleLog(percentage))
                    .setUserSteps(getUserSteps(percentage))
                    .setUserEmail(getUserEmail())
                    .setUserName(getUserName())
                    .setPushToken(getPushNotificationToken())
                    .setUserData(getUserData())
                    .setTags(getTags())
                    .setUserAttributes(UserAttributesDbHelper.getSDKUserAttributes())
                    .setUserEvents(getUserEvents(percentage));


            if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.SESSION_PROFILER)
                    == Feature.State.ENABLED) {
                state.setSessionProfilerTimeline(getSessionProfilerTimeline(percentage));
            }

            if (withExperiments) {
                state.setExperiments(getExperiments(percentage));
            }
            if (withFeatureFlag){
                state.setFeaturesFlags(getFeaturesFlags(percentage));
            }

            /**
             * ToDo: Place holder to force merge conflict so changes to this method are applied to @link{com.instabug.library.model.StateBuilder}
             */


            state.setCurrentActivity(getCurrentActivity());
            if (withHubData && reportModifier != null) {
                reportModifier.finish();
            }
            return state;
        }

        public State buildEarlyState() {
            return buildSimplifiedEarlyState()
                    .setPushToken(getPushNotificationToken())
                    .setTags(getTags())
                    .setUserAttributes(UserAttributesDbHelper.getSDKUserAttributes())
                    .setExperiments(getExperiments(1.0f))
                    .setFeaturesFlags(getFeaturesFlags(1.0f))
                    .setCurrentActivity(getCurrentActivity())
                    .setAppToken(getEarlyAppToken())
                    .setFeaturesFlags(getFeaturesFlags(1.0f));


        }

        @Nullable
        private List<IBGFeatureFlag> getFeaturesFlags(Float percentage) {
            return FeaturesFlagServiceLocator.getFeatureFlagsManager().getAllFeaturesFlags(percentage);
        }

        @Nullable
        protected String getEarlyAppToken() {
            String mappedToken = TokenMappingServiceLocator
                    .getTokenMappingConfigs()
                    .getAvailableAppToken();
            if (mappedToken != null) {
                return mappedToken;
            }
            return SettingsManager.getInstance().getEarlyAppToken();
        }

        @Nullable
        protected List<String> getExperiments(float percentage) {
            List<String> experiments = ServiceLocator.getExperimentsManager().getExperiments(percentage);
            int numberOfDesiredExperiments = (int) Math.round(ServiceLocator.getExperimentsStoreLimit() * percentage);
            if (experiments == null || experiments.size() <= numberOfDesiredExperiments)
                return experiments;
            while (experiments.size() > 0 && experiments.size() > numberOfDesiredExperiments)
                experiments.remove(0);
            return experiments;
        }

        protected List<UserStep> getUserSteps(float percentage) {
            try {
                int fullLimit = CoreServiceLocator.getLimitConstraintApplier().applyConstraints(USER_STEPS_LIMIT);
                int numberOfDesiredSteps = Math.round(fullLimit * percentage);
                List<UserStep> steps = SessionReplayServiceLocator.getUserStepsProvider().getUserSteps();
                if (steps.size() <= numberOfDesiredSteps) return steps;
                int startIndex = steps.size() - numberOfDesiredSteps;
                return new ArrayList<>(steps.subList(startIndex, steps.size()));
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Unable to get user steps", e);
                return new ArrayList<>();
            }
        }

        protected static ArrayList<VisualUserStep> getVisualUserSteps() {
            return CoreServiceLocator.getReproStepsProxy().fetch();
        }

        protected SessionProfilerTimeline getSessionProfilerTimeline(float percentage) {
            return SessionProfiler.getInstance().fetch(percentage);
        }

        protected String getUserEmail() {
            return UserManager.getUserEmail();
        }

        @Nullable
        protected String getUserName() {
            return UserManager.getUserName();
        }

        protected String getPushNotificationToken() {
            return InstabugCore.getPushNotificationToken();
        }

        protected String getUserData() {
            return SettingsManager.getInstance().getUserData();
        }

        protected long getReportedAt() {
            return InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds();
        }

        protected String getTags() {
            return SettingsManager.getInstance().getTagsAsString();
        }

        protected String getUserEvents(float percentage) {
            if (!MemoryUtils.isLowMemory(context)) {
                try {
                    return UserEvent.toJson(InstabugUserEventLogger.getInstance().getUserEvents(percentage))
                            .toString();
                } catch (JSONException | OutOfMemoryError e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while parsing user events logs", e);
                }
            } else {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Running low on memory. Excluding UserEvents serialization from state builder.");
            }
            return "[]";
        }

        protected String getCurrentActivity() {
            InstabugInternalTrackingDelegate trackingDelegate = InstabugInternalTrackingDelegate.getInstance();
            if (trackingDelegate == null) return ACTIVITY_NAME_FALLBACK;
            Activity currentRealActivity = trackingDelegate.getCurrentRealActivity();
            if (currentRealActivity == null) return ACTIVITY_NAME_FALLBACK;
            return currentRealActivity.getClass().getName();
        }

        protected float getTrimmingPercentage() {
            if (SettingsManager.getInstance().isFeatureEnabled(IBGFeature.DEVICE_PERFORMANCE_CLASS, false)) {
                return PerSessionSettings.getInstance().getDeviceTrimmingPercentage();
            }
            return 0;
        }

        @Nullable
        protected String getSessionId() {
            IBGInMemorySession session = InstabugCore.getRunningV3Session();
            return composeSessionId(session);
        }

        @NonNull
        protected String getAppLaunchId() {
            return AppLaunchIDProvider.INSTANCE.getSpanId();
        }
    }

    @Nullable
    private static String composeSessionId(@Nullable IBGInMemorySession session) {
        String appToken = SettingsManager.getInstance().getAppToken();
        if (session == null || appToken == null) return null;
        return IBGSessionMapper.INSTANCE.getCompositeSessionId(session, appToken);
    }

    @Nullable
    private HashMap<Integer, Integer> jsonToReproMap(@Nullable JSONObject json) throws JSONException {
        if(json == null) return null;
        HashMap<Integer, Integer> map = new HashMap<Integer, Integer>();
        Iterator<String> keys = json.keys();
        while(keys.hasNext()) {
            String key = keys.next();
            Integer value = Integer.valueOf(json.get(key).toString());
            map.put(Integer.valueOf(key), value);
        }
        return map;
    }

    @Nullable
    private JSONObject mapToJson(@Nullable HashMap<Integer, Integer> map) throws JSONException {
        if(map == null) return null;
        JSONObject json = new JSONObject();
        for (Map.Entry<Integer, Integer> entry : map.entrySet()) {
            json.put(String.valueOf(entry.getKey()), entry.getValue());
        }
        return json;
    }
}
