package com.instabug.library.model;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.NetworkLogEntry;
import static com.instabug.library.sessionreplay.model.SRLogKt.getMaxSize;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.datahub.DataHubLog;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator;
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider;
import com.instabug.library.networkv2.BodyBufferHelper;
import com.instabug.library.sessionreplay.SRConstantsKt;
import com.instabug.library.sessionreplay.model.SRLog;
import com.instabug.library.sessionreplay.model.SRLogType;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by mNagy on 12/15/16.
 */

public class NetworkLog implements SRLog, DataHubLog {
    public static final String CONTENT_TYPE = "content-type";
    public static final String JSON = "application/json";
    public static final String PROTOBUF = "application/protobuf";
    public static final String XML_1 = "application/xml";
    public static final String XML_2 = "text/xml";
    public static final String HTML = "text/html";
    public static final String PLAIN_TEXT = "text/plain";
    public static final int NETWORK_LOG_LIMIT = 100;

    /**
     * A limit for both response/request bodies to limit max size of a table row to 1MB
     * A cursor should not read more than 1MB row at once, otherwise it throws {@linkplain android.database.sqlite.SQLiteBlobTooBigException}
     */
    public static final int SQL_RECORD_CHAR_LIMIT = 1_000_000; // chars
    public static final String FAILED_TO_PARSE_NETWORK_LOG_TO_JSON = "Failed to parse Network Log to JSON:";
    public static final String REQUEST_BODY_EXCEEDS_THE_MAXIMUM_SIZE_OF_1_KB = "The request body has not been logged because it exceeds the maximum size of %s Kb";
    public static final String RESPONSE_BODY_EXCEEDS_THE_MAXIMUM_SIZE_OF_1_KB = "The response body has not been logged because it exceeds the maximum size of %s Kb";
    private static final int KB = 1024;

    @Nullable
    private String date;
    @Nullable
    private String url;
    @Nullable
    private String request;
    @Nullable
    private String response;
    @Nullable
    private String method;
    @Nullable
    private String requestHeaders;
    @Nullable
    private String responseHeaders;
    // Total time (request + response)
    private long totalDuration;
    private int responseCode;
    private boolean userModified = false;
    private boolean autoMasked = false;

    @Nullable
    public String getDate() {
        return date;
    }

    public void setDate(@Nullable String date) {
        this.date = date;
    }

    @Nullable
    public String getUrl() {
        return url;
    }

    public void setUrl(@Nullable String url) {
        this.url = url;
    }

    @Nullable
    public String getRequest() {
        return request;
    }

    public void setRequest(@Nullable String request) {
        if (request != null) {
            this.request = validateSize(request, REQUEST_BODY_EXCEEDS_THE_MAXIMUM_SIZE_OF_1_KB);
        } else {
            this.request = null;
        }
    }

    @Nullable
    public String getResponse() {
        return response;
    }

    public void setResponse(@Nullable String response) {
        if (response != null) {
            this.response = validateSize(response, RESPONSE_BODY_EXCEEDS_THE_MAXIMUM_SIZE_OF_1_KB);
        } else {
            this.response = null;
        }
    }

    @NonNull
    private String validateSize(@NonNull String request, @NonNull String fallback) {
        int maxSize = getMaxSize(this);
        int maxSizeInKb = maxSize / KB;
        return BodyBufferHelper.isBodySizeAllowed(request, maxSize) ? request : String.format(fallback, maxSizeInKb);
    }

    @Nullable
    public String getMethod() {
        return method;
    }

    public void setMethod(@Nullable String method) {
        this.method = method;
    }

    public int getResponseCode() {
        return responseCode;
    }

    public void setResponseCode(int responseCode) {
        this.responseCode = responseCode;
    }

    @Nullable
    public String getRequestHeaders() {
        return requestHeaders;
    }

    @Nullable
    public String getResponseHeaders() {
        return responseHeaders;
    }

    public void setRequestHeaders(@Nullable String requestHeaders) {
        this.requestHeaders = requestHeaders;
    }

    public void setResponseHeaders(@Nullable String responseHeaders) {
        this.responseHeaders = responseHeaders;
    }

    public long getTotalDuration() {
        return totalDuration;
    }

    public void setTotalDuration(long totalDuration) {
        this.totalDuration = totalDuration;
    }

    public boolean isUserModified() {
        return userModified;
    }

    public void setUserModified(boolean userModified) {
        this.userModified = userModified;
    }

    public boolean isAutoMasked() {
        return autoMasked;
    }

    public void setAutoMasked(boolean autoMasked) {
        this.autoMasked = autoMasked;
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof NetworkLog)) return false;

        NetworkLog that = (NetworkLog) o;

        if (responseCode != that.responseCode) return false;
        if (date != null ? !date.equals(that.date) : that.date != null) return false;
        if (url != null ? !url.equals(that.url) : that.url != null) return false;
        if (request != null ? !request.equals(that.request) : that.request != null) return false;
        if (response != null ? !response.equals(that.response) : that.response != null)
            return false;
        if (method != null ? !method.equals(that.method) : that.method != null) return false;
        if (totalDuration != that.totalDuration) return false;
        if (responseHeaders != null ? !responseHeaders.equals(that.responseHeaders) : that.responseHeaders != null) {
            return false;
        }
        if (userModified != that.userModified) return false;
        return requestHeaders != null ? requestHeaders.equals(that.requestHeaders) : that.requestHeaders == null;
    }

    @Override
    public int hashCode() {
        int result = date != null ? date.hashCode() : 0;
        result = 31 * result + (url != null ? url.hashCode() : 0);
        result = 31 * result + (request != null ? request.hashCode() : 0);
        result = 31 * result + (response != null ? response.hashCode() : 0);
        result = 31 * result + (method != null ? method.hashCode() : 0);
        result = 31 * result + responseCode;
        result = 31 * result + (responseHeaders != null ? responseHeaders.hashCode() : 0);
        result = 31 * result + (requestHeaders != null ? requestHeaders.hashCode() : 0);
        result = 31 * result + Long.valueOf(totalDuration).hashCode();
        result = 31 * result + (userModified ? 1 : 0);

        return result;
    }

    @NonNull
    @Override
    public String toString() {
        return "NetworkLog{" +
                "date='" + date + '\'' +
                ", url='" + url + '\'' +
                ", request='" + request + '\'' +
                ", method='" + method + '\'' +
                ", responseCode=" + responseCode +
                ", headers='" + requestHeaders + '\'' +
                ", response='" + response + '\'' +
                ", response_headers='" + responseHeaders + '\'' +
                ", totalDuration='" + totalDuration + '\'' +
                ", modifiedByUser='" + userModified + '\'' +
                '}';
    }

    public void insert() {
        maskAndInsert();
    }

    private void maskAndInsert() {
        NetworkLog maskedNetworkLog = handleAutoMasking();
        CoreServiceLocator.getNetworkLogsDistributor().invoke(maskedNetworkLog);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public JSONObject toJsonObject() throws JSONException {
        JSONObject networkLog = new JSONObject();
        networkLog.put(NetworkLogEntry.COLUMN_DATE, getDate());
        networkLog.put(NetworkLogEntry.COLUMN_METHOD, getMethod());
        networkLog.put(NetworkLogEntry.COLUMN_RESPONSE_CODE, getResponseCode());
        networkLog.put(NetworkLogEntry.COLUMN_URL, getUrl());
        networkLog.put(NetworkLogEntry.COLUMN_NETWORK_TIME, getTotalDuration());
        networkLog.put(NetworkLogEntry.COLUMN_USER_MODIFIED, isUserModified());

        try {
            JSONObject headersObject = new JSONObject(getRequestHeaders());
            networkLog.put(NetworkLogEntry.COLUMN_HEADERS, headersObject);
        } catch (Exception e) {
            networkLog.put(NetworkLogEntry.COLUMN_HEADERS, getRequestHeaders());
        }
        try {
            JSONObject headersObject = new JSONObject(getResponseHeaders());
            networkLog.put(NetworkLogEntry.COLUMN_RESPONSE_HEADERS, headersObject);
        } catch (Exception e) {
            networkLog.put(NetworkLogEntry.COLUMN_RESPONSE_HEADERS, getResponseHeaders());
        }

        try {
            JSONObject requestObject = new JSONObject(getRequest());
            networkLog.put(NetworkLogEntry.COLUMN_REQUEST, requestObject);
        } catch (Exception e) {
            networkLog.put(NetworkLogEntry.COLUMN_REQUEST, getRequest());
        }

        try {
            JSONObject responseObject = new JSONObject(getResponse());
            networkLog.put(NetworkLogEntry.COLUMN_RESPONSE, responseObject);
        } catch (Exception e) {
            networkLog.put(NetworkLogEntry.COLUMN_RESPONSE, getResponse());

        }
        return networkLog;
    }

    @Override
    public long getTimestamp() {
        return date != null ? Long.parseLong(date) : 0;
    }

    @NonNull
    @Override
    public String getLogType() {
        return SRLogType.NETWORK_LOG;
    }

    @Nullable
    @Override
    public JSONObject getSrJsonRep() {
        try {
            JSONObject jsonObject = toJsonObject();
            //fill session replay data
            jsonObject
                    .put(SRConstantsKt.LOG_TYPE_KEY, getLogType())
                    .put(SRConstantsKt.TIMESTAMP_KEY, getTimestamp());
            return jsonObject;
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e, FAILED_TO_PARSE_NETWORK_LOG_TO_JSON, Constants.LOG_TAG);
            return null;
        }

    }

    @Nullable
    @Override
    public JSONObject getDataHubRep() {
        try {
            return toJsonObject();
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e, FAILED_TO_PARSE_NETWORK_LOG_TO_JSON, Constants.LOG_TAG);
            return null;
        }
    }

    public NetworkLog handleAutoMasking() {
        IBGNetworkInterceptionConfigurationProvider configurationProvider = NetworkInterceptionServiceLocator.getConfigurationProvider();
        if(!autoMasked && configurationProvider.isAutoMaskingEnabled()) {
            autoMasked = true;
            NetworkLog modifiedLog = NetworkInterceptionServiceLocator.getAutoMaskingSanitizer().sanitize(this);
            return modifiedLog == null ? this : modifiedLog;
        }

        return this;
    }


}
