package com.instabug.library.logging.disklogs;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.internal.orchestrator.Action;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.storage.operation.DiskOperation;
import com.instabug.library.internal.storage.operation.DiskOperationCallback;
import com.instabug.library.model.LogDescriptor;
import com.instabug.library.util.memory.MemoryUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * A class to provide API for writing session details Async or sync
 */
public class WriteLogDescriptorDiskOperator implements DiskOperation<Uri, Context> {

    private final File file;
    private LogDescriptor logDescriptor;

    /**
     * Construct the operator by giving it the logging file which will be used to write in
     * and logging descriptor that will hold the state of device in this session.
     *
     * @param file    log file
     * @param logData session details
     */
    WriteLogDescriptorDiskOperator(@NonNull File file, @NonNull LogDescriptor logData) {
        this.file = file;
        this.logDescriptor = logData;
    }

    /**
     * A method to execute writing operation sync
     *
     * @param context this context isn't used so it can be nullable.
     * @return a uri from file
     * @throws IOException
     */
    @Override
    public Uri execute(@Nullable Context context) throws IOException {
        writeFile(logDescriptor, context);
        return Uri.fromFile(file);
    }

    /**
     * A method to execute writing operation async
     *
     * @param context this context isn't used so it can be nullable.
     * @return a uri from file
     * @throws IOException
     */
    @Override
    public void executeAsync(final Context context, @Nullable final DiskOperationCallback<Uri> callback) {
        ActionsOrchestrator
                .obtainOrchestrator()
                .addWorkerThreadAction(new Action() {
                    @Override
                    public void run() throws Exception {
                        writeFile(logDescriptor, context);
                        if (callback != null) {
                            callback.onSuccess(Uri.fromFile(file));
                        }
                    }
                })
                .orchestrate();
    }

    /**
     * Internal method to write session details in file
     *
     * @param logDescriptor data
     * @throws IOException
     */

    private void writeFile(LogDescriptor logDescriptor, @Nullable Context context) throws IOException {
        if (context != null) {
            if (!MemoryUtils.isLowMemory(context)) {
                final FileOutputStream out = new FileOutputStream(file, true);
                final String logs = logDescriptor.toString();
                    try {
                        out.write(EncryptionManager.encrypt(logs.getBytes("UTF-8")));
                        out.write(EncryptionManager.LINE_FEED.getBytes("UTF-8"));
                    }catch (Throwable throwable){
                        Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to " + throwable.getMessage());
                    }
                 finally {
                    out.close();
                }
            } else {
                Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to low memory");
            }
        } else {
            Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to null context");
        }
    }


}
