package com.instabug.library.core.plugin;

import android.content.Context;

import androidx.annotation.CallSuper;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Instabug;
import com.instabug.library.internal.dataretention.DisposalPolicy;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import kotlin.annotation.AnnotationTarget;
import kotlin.annotation.Target;

/**
 * Created by tarek on 3/25/17.
 */

public abstract class Plugin {

    public static final int STATE_BACKGROUND = 0;
    public static final int STATE_FOREGROUND = 1;
    public static final int STATE_PROCESSING_ATTACHMENT = 2;
    @Nullable
    protected WeakReference<Context> contextWeakReference;
    @PluginState
    protected volatile int state = STATE_BACKGROUND;
    @PluginLifeCycleState
    private int lifeCycleState = PluginLifeCycleState.UNDEFINED;

    @CallSuper
    public void init(Context context) {
        contextWeakReference = new WeakReference<>(context);
        lifeCycleState = PluginLifeCycleState.INITIALIZED;
    }

    abstract public void wake();

    abstract public void sleep();

    abstract public void stop();

    abstract public void start(Context context);

    void wakeIfPossible() {
        if (canWake()) {
            wake();
            lifeCycleState = PluginLifeCycleState.WOKE;
        }
    }

    void sleepIfPossible() {
        if (canSleep()) {
            sleep();
            lifeCycleState = PluginLifeCycleState.SLEPT;
        }
    }

    void stopIfPossible() {
        if (canStop()) {
            stop();
            lifeCycleState = PluginLifeCycleState.STOPPED;
        }
    }

    void startIfPossible(Context context) {
        if (canStart()) {
            start(context);
            lifeCycleState = PluginLifeCycleState.STARTED;
        }
    }

    abstract public long getLastActivityTime();

    public void onLocaleChanged(Locale oldLocale, Locale newLocale) {
    }

    public void initDefaultPromptOptionAvailabilityState() {
    }

    private boolean canStop() {
        return lifeCycleState == PluginLifeCycleState.SLEPT
                || lifeCycleState == PluginLifeCycleState.STARTED
                || lifeCycleState == PluginLifeCycleState.WOKE;
    }

    private boolean canStart() {
        return lifeCycleState == PluginLifeCycleState.INITIALIZED
                || lifeCycleState == PluginLifeCycleState.STOPPED;
    }

    private boolean canWake() {
        return lifeCycleState == PluginLifeCycleState.STARTED
                || lifeCycleState == PluginLifeCycleState.SLEPT;
    }

    private boolean canSleep() {
        return lifeCycleState == PluginLifeCycleState.WOKE;
    }

    @Nullable
    public ArrayList<PluginPromptOption> getPromptOptions() {
        return null;
    }

    @Nullable
    public ArrayList<PluginPromptOption> getPluginOptions(boolean ignoreBaseFeature) {
        return null;
    }

    /**
     * Returns a list of disposal policies that has set the required retention periods for specified
     * categories of user's data and the minimum standards to be applied when destroying the information.
     *
     * @return list of {@link DisposalPolicy} (empty by default)
     */
    @NonNull
    public List<DisposalPolicy> getDataDisposalPolicies() {
        return Collections.emptyList();
    }

    @PluginState
    public int getState() {
        return state;
    }

    public void setState(@PluginState int state) {
        this.state = state;
    }

    @Nullable
    public Context getAppContext() {
        return Instabug.getApplicationContext();
    }

    public boolean isAppContextAvailable() {
        return contextWeakReference != null && contextWeakReference.get() != null;
    }

    public void invoke(PluginPromptOption pluginPromptOption) {

    }

    public abstract boolean isFeatureEnabled();

    @Retention(RetentionPolicy.SOURCE)
    @Target(allowedTargets = {AnnotationTarget.TYPE, AnnotationTarget.FIELD})
    @IntDef({
            STATE_FOREGROUND,
            STATE_BACKGROUND,
            STATE_PROCESSING_ATTACHMENT})
    public @interface PluginState {
    }

    @Retention(RetentionPolicy.SOURCE)
    @IntDef({
            PluginLifeCycleState.UNDEFINED,
            PluginLifeCycleState.INITIALIZED,
            PluginLifeCycleState.STARTED,
            PluginLifeCycleState.WOKE,
            PluginLifeCycleState.SLEPT,
            PluginLifeCycleState.STOPPED})
    public @interface PluginLifeCycleState {
        int UNDEFINED = -1;
        int INITIALIZED = 1;
        int STARTED = 2;
        int WOKE = 3;
        int SLEPT = 4;
        int STOPPED = 5;
    }

}
