package com.instabug.library;

import static com.instabug.library.FeaturesConstants.Default_STATE.SESSION_STITCHING_DEFAULT_STATE;
import static com.instabug.library.FeaturesConstants.SESSION_STITCHING;
import static com.instabug.library.FeaturesConstants.SESSION_STITCHING_TIME_OUT;
import static com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_V3_SESSIONS;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched;
import com.instabug.library.experiments.ExperimentsManager;
import com.instabug.library.experiments.di.ServiceLocator;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.internal.contentprovider.InstabugApplicationProvider;
import com.instabug.library.internal.resolver.LoggingSettingResolver;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.sharedpreferences.CorePrefPropertyKt;
import com.instabug.library.internal.sharedpreferences.SharedPreferencesMigrationEngine;
import com.instabug.library.internal.utils.memory.IBGLowMemoryHandler;
import com.instabug.library.model.FeaturesCache;
import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.service.FeaturesService;
import com.instabug.library.percentagefeatures.IBGPercentageFlagsResolver;
import com.instabug.library.sessionV3.configurations.RatingDialogDetectionConfigurationsImpl;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessioncontroller.SessionManualControllerServiceLocator;
import com.instabug.library.settings.PersistableSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.TokenMappingConfigsHandler;
import com.instabug.library.tokenmapping.TokenMappingConfigsHandlerImpl;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.memory.MemoryUtils;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Field;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Manages all plan-related feature updates and queries
 *
 * @author mSobhy
 */
public class InstabugFeaturesManager {
    @VisibleForTesting
    static final String STATE_SUFFIX = "STATE";
    @VisibleForTesting
    static final String AVAILABILITY_SUFFIX = "AVAIL";
    @VisibleForTesting
    static final String LAST_FETCHED_AT = "LAST_FETCHED_AT";
    static final Feature.State DEFAULT_FEATURE_STATE = Feature.State.ENABLED;
    static final Feature.State DEFAULT_CUSTOMIZED_FEATURE_STATE = Feature.State.DISABLED;
    private static final String EXPERIMENTAL_AVAILABILITY_SUFFIX = "EXP_AVAIL";
    private static final boolean DEFAULT_FEATURE_AVAILABILITY = true;
    private static final boolean DEFAULT_EXPERIMENTAL_FEATURE_AVAILABILITY = false;
    private static final boolean DEFAULT_CUSTOMIZED_FEATURE_AVAILABILITY = false;
    private static volatile InstabugFeaturesManager INSTANCE;
    private boolean instabugStateLogged = false;

    public final static String FEATURES_FETCHING_ERROR = "Something went wrong while do fetching features request";
    /**
     * Key to disable the whole SDK Temporary
     */
    private boolean temporaryDisabled = false;
    private ConcurrentHashMap<String, Feature.State> featuresState;
    /**
     * A map to check feature backend availability
     */
    private ConcurrentHashMap<String, Boolean> featuresAvailability;
    private ConcurrentHashMap<String, Boolean> experimentalFeaturesAvailability;

    private InstabugFeaturesManager() {
        featuresState = new ConcurrentHashMap<>(20, .9f, 2);
        featuresAvailability = new ConcurrentHashMap<>(20, .9f, 2);
        experimentalFeaturesAvailability = new ConcurrentHashMap<>(20, .9f, 2);
    }

    public static InstabugFeaturesManager getInstance() {
        if (INSTANCE == null) {  // have to do this inside the sync
            INSTANCE = new InstabugFeaturesManager();
        }
        return INSTANCE;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    static void release() {
        INSTANCE = null;
    }

    /**
     * Changes feature availability
     *
     * @param feature     to changa re availability of
     * @param isAvailable new state of {@code feature}
     */
    @VisibleForTesting
    void updateFeatureAvailability(@IBGFeature.Companion.IBGFeature String feature, boolean isAvailable) {
        updateFeatureObjectAvailability(feature, isAvailable);
    }

    /**
     * Changes feature availability
     *
     * @param feature     to changa re availability of
     * @param isAvailable new state of {@code feature}
     */
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    private void updateFeatureObjectAvailability(@IBGFeature.Companion.IBGFeature String feature, boolean isAvailable) {
        if (featuresAvailability.containsKey(feature) &&
                featuresAvailability.get(feature) == isAvailable) {
            return;
        }
        featuresAvailability.put(feature, isAvailable);
    }

    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    void updateExperimentalFeatureAvailability(@IBGFeature.Companion.IBGFeature String feature, boolean isAvailable) {
        if (experimentalFeaturesAvailability.containsKey(feature)
                && experimentalFeaturesAvailability.get(feature) == isAvailable) {
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Experimental feature " + feature + " availability to " +
                    isAvailable);
            experimentalFeaturesAvailability.put(feature, isAvailable);
        }
    }

    /**
     * @return check if the database transaction flag is set enabled/disabled
     */
    public boolean isDatabaseTransactionDisabled() {
        SharedPreferences sharedPreferences = CorePrefPropertyKt.getCorePreferences();
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IBGFeature.DATABASE_TRANSACTIONS_DISABLED, true);
    }

    /**
     * @param isDisabled set the database transaction enable/disabled controlled from BE flags.
     */
    public void setDatabaseTransactionDisabled(boolean isDisabled) {
        SharedPreferences sharedPreferences = CorePrefPropertyKt.getCorePreferences();
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IBGFeature.DATABASE_TRANSACTIONS_DISABLED, isDisabled).apply();
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public boolean isExperimentalFeatureAvailable(@IBGFeature.Companion.IBGFeature String feature) {
        if (experimentalFeaturesAvailability.containsKey(feature) && experimentalFeaturesAvailability.get(feature) != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Experimental Feature " + feature + " availability is " +
                    experimentalFeaturesAvailability.get(feature));
            return experimentalFeaturesAvailability.get(feature);
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Experimental Feature " + feature + " availability not found, returning " +
                    "" + DEFAULT_EXPERIMENTAL_FEATURE_AVAILABILITY);
            return DEFAULT_EXPERIMENTAL_FEATURE_AVAILABILITY;
        }
    }

    /**
     * @return {@code true} if features are fetched before by this device not in this session
     */
    public boolean isFeaturesFetchedBefore() {
        Context context = Instabug.getApplicationContext();
        return context != null && getLastFetchedAt() > 0L;
    }

    /**
     * Get last fetched at
     *
     * @return last fetched at
     */
    @VisibleForTesting
    long getLastFetchedAt() {
        SharedPreferences sharedPreferences = CorePrefPropertyKt.getCorePreferences();
        if (sharedPreferences == null) return 0L;
        return sharedPreferences.getLong(LAST_FETCHED_AT, 0L);
    }

    /**
     * Changes users preference of this feature state
     *
     * @param feature to change state of
     * @param state   new state of {@code feature}
     * @see com.instabug.library.Feature.State
     */
    public void setFeatureState(String feature, Feature.State state) {
        if (featuresState.get(feature) != state) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Setting " + feature + " state to " + state);
            featuresState.put(feature, state);
        }
    }

    /**
     * @param featureObj to check state of
     * @return current state of this feature, based on it's (availability & instabug state) and
     * user specified state
     * @see #isFeatureAvailable(Object)
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public Feature.State getFeatureState(Object featureObj) {
        if (temporaryDisabled) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "SDK is temporaryDisabled, returning disabled for feature: " + featureObj.toString());
            return Feature.State.DISABLED;
        }
        // if the plan is not available return disabled
        if (!isFeatureAvailable(IBGFeature.INSTABUG)) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "getFeatureState#!isFeatureAvailable, returning disabled for feature: " + featureObj.toString());
            return Feature.State.DISABLED;
        }
        // if Instabug is disabled return disable
        if (featuresState.get(IBGFeature.INSTABUG) == Feature.State.DISABLED) {
            if (instabugStateLogged) return Feature.State.DISABLED;
            InstabugSDKLogger.e(Constants.LOG_TAG, "Instabug is disabled ");
            instabugStateLogged = true;
            return Feature.State.DISABLED;
        }
        instabugStateLogged = false;

        boolean isFeatureAvailable = isFeatureAvailable(featureObj);
        if (!isFeatureAvailable) {
            return Feature.State.DISABLED;
        } else if (featuresState.containsKey(featureObj)) {
            return featuresState.get(featureObj);
        } else {
            // special condition for view hierarchy as it is disabled by default
            if (shouldDisableFeature(featureObj)) {
                return DEFAULT_CUSTOMIZED_FEATURE_STATE;
            }
            return DEFAULT_FEATURE_STATE;
        }
    }

    /**
     * @param featureObj to check availability of
     * @return {@code true} if this feature is available in user plan
     */
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public boolean isFeatureAvailable(Object featureObj) {
        if (featuresAvailability.containsKey(featureObj)) {
            return featuresAvailability.get(featureObj);
        } else if (shouldDisableFeature(featureObj)) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "isFeatureAvailable#shouldDisableFeature: " + featureObj.toString() + " return: DEFAULT_CUSTOMIZED_FEATURE_AVAILABILITY");
            return DEFAULT_CUSTOMIZED_FEATURE_AVAILABILITY;
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "isFeatureAvailable: " + featureObj.toString() + " return: DEFAULT_FEATURE_AVAILABILITY");
            return DEFAULT_FEATURE_AVAILABILITY;
        }
    }

    /**
     * Features that are disabled by default.
     *
     * @param feature
     * @return
     */
    private boolean shouldDisableFeature(Object feature) {
        return feature == IBGFeature.VIEW_HIERARCHY_V2
                || feature == IBGFeature.VP_CUSTOMIZATION
                || feature == IBGFeature.VZ_MESSAGES_CUSTOM_APPRATING_UI
                || feature == IBGFeature.PRODUCTION_USAGE_DETECTION
                || feature == IBGFeature.BE_USERS_KEYS;
    }

    public Feature.State getEncryptionState() {
        if (InstabugApplicationProvider.getInstance() != null) {
            Context context = InstabugApplicationProvider.getInstance().getApplication();
            if (context != null) {
                SharedPreferences sharedPreferences = context.getSharedPreferences("instabug", Context.MODE_PRIVATE);
                return sharedPreferences.getBoolean(IBGFeature.ENCRYPTION, false) ? Feature.State.ENABLED : Feature.State.DISABLED;
            }
        }
        return Feature.State.DISABLED;
    }

    public Feature.State getDbEncryptionState() {
        if (InstabugApplicationProvider.getInstance() != null) {
            Context context = InstabugApplicationProvider.getInstance().getApplication();
            if (context != null) {
                SharedPreferences sharedPreferences = context.getSharedPreferences("instabug", Context.MODE_PRIVATE);
                return sharedPreferences.getBoolean(IBGFeature.DB_ENCRYPTION, false) ? Feature.State.ENABLED : Feature.State.DISABLED;
            }
        }
        return Feature.State.DISABLED;
    }

    private void setEncryptionState(boolean isEnabled) {
        if (InstabugApplicationProvider.getInstance() != null) {
            Context context = InstabugApplicationProvider.getInstance().getApplication();
            if (context != null) {
                SharedPreferences sharedPreferences = context.getSharedPreferences("instabug", Context.MODE_PRIVATE);
                sharedPreferences.edit().putBoolean(IBGFeature.ENCRYPTION, isEnabled).apply();
            }
        }
    }

    private void setDbEncryptionState(boolean isEnabled) {
        if (InstabugApplicationProvider.getInstance() != null) {
            Context context = InstabugApplicationProvider.getInstance().getApplication();
            if (context != null) {
                SharedPreferences sharedPreferences = context.getSharedPreferences("instabug", Context.MODE_PRIVATE);
                sharedPreferences.edit().putBoolean(IBGFeature.DB_ENCRYPTION, isEnabled).apply();
            }
        }
    }

    /**
     * @param featureObj to check state of
     * @return current state of this feature, based on it's availability and user specified state
     * Ignoring instabug state
     * @see #isFeatureAvailable(Object)
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public Feature.State getFeatureAbsoluteState(Object featureObj) {
        if (temporaryDisabled) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "SDK is temp disabled, returing disable for " + featureObj.toString());
            return Feature.State.DISABLED;
        }

        boolean isFeatureAvailable = isFeatureAvailable(featureObj);
        if (!isFeatureAvailable) {
            return Feature.State.DISABLED;
        } else if (featuresState.containsKey(featureObj)) {
            return featuresState.get(featureObj);
        } else {
            // special condition for view hierarchy as it is disabled by default
            if (shouldDisableFeature(featureObj)) {
                return DEFAULT_CUSTOMIZED_FEATURE_STATE;
            }
            return DEFAULT_FEATURE_STATE;
        }
    }

    /**
     * Saves current state of Features to SharedPreferences
     *
     * @param context to use for getting SharedPreferences
     */
    public void saveFeaturesToSharedPreferences(final Context context) {
        if (context == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "unable to execute saveFeaturesToSharedPreferences."
                    + " Null context reference");
            return;
        }
        if (!MemoryUtils.isLowMemory(context)) {
            PoolProvider.getInstance().getIOExecutor().execute(() -> {
                try {
                    SharedPreferences sharedPreferences = CoreServiceLocator.getInstabugSharedPreferences(context,
                            SettingsManager.INSTABUG_SHARED_PREF_NAME);
                    if (sharedPreferences == null) {
                        InstabugSDKLogger.e(Constants.LOG_TAG,
                                "Couldn't save features because SharedPref is not available, Instabug will be paused");
                        Instabug.pauseSdk();
                        return;
                    }
                    SharedPreferences.Editor editor = sharedPreferences.edit();
                    for (Object feature : featuresAvailability.keySet()) {
                        if (feature instanceof String) {

                            editor.putBoolean((feature) + AVAILABILITY_SUFFIX,
                                    featuresAvailability.get(feature));
                        }
                    }
                    for (@IBGFeature.Companion.IBGFeature String feature : experimentalFeaturesAvailability.keySet()) {
                        String key = getExpFeaturePreferencesKey(feature);
                        boolean value = experimentalFeaturesAvailability.get(feature);
                        editor.putBoolean(key, value);
                    }
                    editor.apply();
                } catch (OutOfMemoryError oom) {
                    InstabugSDKLogger.e(Constants.LOG_TAG,
                            "Couldn't save features because memory is low, Instabug will be paused");
                    Instabug.pauseSdk();
                }
            });
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Couldn't save features because memory is low, Instabug will be paused");
            Instabug.pauseSdk();
        }
    }

    private String getExpFeaturePreferencesKey(@NonNull @IBGFeature.Companion.IBGFeature String featureName) {
        return featureName + EXPERIMENTAL_AVAILABILITY_SUFFIX;
    }

    /**
     * Restores old state of Features from SharedPreferences
     *
     * @param context to use for getting SharedPreferences
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    void restoreFeaturesFromSharedPreferences(final Context context) {
        if (!MemoryUtils.isLowMemory(context)) {
            SharedPreferences sharedPreferences = CoreServiceLocator.getInstabugSharedPreferences(context,
                    SettingsManager.INSTABUG_SHARED_PREF_NAME);
            if (sharedPreferences == null) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "Couldn't restoreFeaturesFromSharedPreferences because SharedPref is not available," +
                                "Instabug will be paused.");
                Instabug.pauseSdk();
                return;
            }

            if (!sharedPreferences.contains(IBGFeature.VP_CUSTOMIZATION + AVAILABILITY_SUFFIX)) {
                setLastFetchedAt(0, context);
                fetchPlanFeatures(context);
                return;
            }

            Field[] features = IBGFeature.class.getFields();
            for (Field feature : features) {
                // Experimental feature
                String key = getExpFeaturePreferencesKey(feature.getName());
                boolean savedExpFeatureAvailability = sharedPreferences.getBoolean(
                        key, DEFAULT_EXPERIMENTAL_FEATURE_AVAILABILITY);
                experimentalFeaturesAvailability.put(feature.getName(), savedExpFeatureAvailability);

                String featureAvailabilityKey = feature.getName() + AVAILABILITY_SUFFIX;
                boolean savedFeatureAvailability = sharedPreferences.getBoolean(feature.getName()
                        + AVAILABILITY_SUFFIX, getDefaultFeatureAvailability(feature.getName()));
                if (sharedPreferences.contains(featureAvailabilityKey)) {
                    featuresAvailability.put(feature.getName(), savedFeatureAvailability);
                } else {
                    if (!featuresAvailability.containsKey(feature.getName())) {
                        featuresAvailability.putIfAbsent(feature.getName(), savedFeatureAvailability);
                    }
                }

                if (!featuresState.containsKey(feature.getName())) {
                    Feature.State savedFeatureState = Feature.State.valueOf(sharedPreferences
                            .getString(feature.getName() + STATE_SUFFIX, getDefaultFeatureState(feature.getName())));
                    featuresState.putIfAbsent(feature.getName(), savedFeatureState);
                }
            }
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "Couldn't restoreFeaturesFromSharedPreferences because memory is low," +
                            "Instabug will be paused.");
            Instabug.pauseSdk();
        }
    }

    private String getDefaultFeatureState(@IBGFeature.Companion.IBGFeature String feature) {
        if (shouldDisableFeature(feature)) {
            return DEFAULT_CUSTOMIZED_FEATURE_STATE.name();
        }
        return DEFAULT_FEATURE_STATE.name();
    }

    private boolean getDefaultFeatureAvailability(@IBGFeature.Companion.IBGFeature String feature) {
        if (shouldDisableFeature(feature)) {
            return DEFAULT_CUSTOMIZED_FEATURE_AVAILABILITY;
        }
        return DEFAULT_FEATURE_AVAILABILITY;
    }

    /**
     * Fetch plan's features from server
     *
     * @param context to use for getting SharedPreferences
     */
    public synchronized void fetchPlanFeatures(final Context context) {
        // First, check if the SDK version updated and invalidate the hash value upon it.
        invalidateFeaturesRequestSettingsIfRequired();

        //Check for TTL expiry
        if (isTTLExpired(context) && !isSdkDecommissioned()) {
            FeaturesService.getInstance().getAppFeatures(new Request.Callbacks<String, Throwable>() {
                @Override
                public void onSucceeded(@Nullable String response) {
                    try {
                        if (response == null) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Features response is null");
                            return;
                        }
                        setLastFetchedAt(System.currentTimeMillis(), context);
                        InstabugSDKLogger.d(Constants.LOG_TAG, "Features fetched " +
                                "successfully");

                        updatePlanFeatures(response);
                        IBGCoreEventPublisher.post(new FeaturesFetched(response));
                        IBGCoreEventPublisher.post(IBGSdkCoreEvent.Features.Fetched.INSTANCE);
                        FeaturesCache featuresCache = getLastFeaturesSettings();
                        if (featuresCache != null && !featuresCache.isActive()) {
                            CoreServiceLocator.getSdkCleaningUtil().clearSdkData();
                        }
                    } catch (JSONException e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Something went " +
                                "wrong while parsing fetching features request's response", e);
                    }
                }

                @Override
                public void onFailed(Throwable throwable) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, FEATURES_FETCHING_ERROR, throwable);
                }
            });
        }
    }

    /**
     * Check If the last fetched at is more than TTL to decide if it should be retrieved again.
     *
     * @param context context to use in getting SharedPreferences
     * @return true if there is no cache or the last fetched exceeds the TTL.
     */
    private boolean isTTLExpired(Context context) {
        FeaturesCache featuresCache = getLastFeaturesSettings();
        if (featuresCache != null) {
            return (System.currentTimeMillis() - getLastFetchedAt()) > featuresCache.getTtl();
        }
        return true;
    }

    /**
     * Check if the SDK is decommissioned or not based on the previously cached values for
     * TTL and isActive.
     *
     * @return true if sdk is decommissioned.
     */
    @VisibleForTesting
    boolean isSdkDecommissioned() {
        FeaturesCache featuresCache = getLastFeaturesSettings();
        if (featuresCache != null && featuresCache.isSdkDecommissioned()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Instabug SDK has been totally disabled, please contact Instabug support team at contactus@instabug.com for help");
            return true;
        }
        return false;
    }

    /**
     * Check If there is cache SDK version is changed to:
     * - Reset the TTL value to default 0.
     * - Invalidate the hash of {@link com.instabug.library.networkv2.request.Header#IF_MATCH}
     */
    @VisibleForTesting
    void invalidateFeaturesRequestSettingsIfRequired() {
        FeaturesCache featuresCache = getLastFeaturesSettings();
        if (featuresCache != null && featuresCache.getSdkVersion() != null) {
            if (featuresCache.getSdkVersion().equalsIgnoreCase(BuildConfig.SDK_VERSION)) {
                return;
            }
            try {
                featuresCache.setTtl(0);
                featuresCache.setHash("");
                SettingsManager.getInstance().setFeaturesCache(featuresCache);
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to update previously cached feature settings due to: " + e.getMessage());
            }
        }
    }

    /**
     * Getting the previously cached FeaturesCache
     *
     * @return {@link FeaturesCache} holds all feature plan request settings.
     */
    @Nullable
    FeaturesCache getLastFeaturesSettings() {
        FeaturesCache featuresCache = null;
        try {
            featuresCache = SettingsManager.getInstance().getFeaturesCache();
            if (featuresCache != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Previously cached feature settings: " + featuresCache.toJson());
            }
        } catch (JSONException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Failed to load previously cached feature settings due to: " + e.getMessage());
        }
        return featuresCache;
    }

    @VisibleForTesting
    void updatePlanFeatures(String response) throws JSONException {
        JSONObject responseJsonObject = new JSONObject(response);
        boolean isActive = responseJsonObject.optBoolean("is_active", true);
        updateFeatureAvailability(IBGFeature.INSTABUG, isActive);
        if (isActive) {
            Instabug.resumeSdk();
        } else {
            Instabug.pauseSdk();
            InstabugSDKLogger.w(Constants.LOG_TAG, "SDK is disabled. " +
                    "Please make sure you are using a valid application token");
        }

        boolean pushNotificationsEnabled = responseJsonObject.optBoolean
                ("push_notifications", false);
        updateFeatureAvailability(IBGFeature.PUSH_NOTIFICATION,
                pushNotificationsEnabled);

        boolean whiteLabelEnabled = responseJsonObject.optBoolean
                ("white_label", false);
        updateFeatureAvailability(IBGFeature.WHITE_LABELING, whiteLabelEnabled);

        boolean customFontEnabled = responseJsonObject.optBoolean
                ("custom_font", false);
        updateFeatureAvailability(IBGFeature.CUSTOM_FONT, customFontEnabled);

        boolean inAppMessagingEnabled = responseJsonObject.optBoolean
                ("in_app_messaging", false);
        updateFeatureAvailability(IBGFeature.IN_APP_MESSAGING,
                inAppMessagingEnabled);

        boolean multipleAttachmentsEnabled = responseJsonObject.optBoolean
                ("multiple_attachments", false);
        updateFeatureAvailability(IBGFeature.MULTIPLE_ATTACHMENTS,
                multipleAttachmentsEnabled);

        boolean userStepsEnabled = responseJsonObject.optBoolean
                ("user_steps", false);
        updateFeatureAvailability(IBGFeature.TRACK_USER_STEPS, userStepsEnabled);

        boolean reproStepsEnabled = responseJsonObject.optBoolean
                ("repro_steps", false);
        updateFeatureAvailability(IBGFeature.REPRO_STEPS, reproStepsEnabled);

        boolean consoleLogEnabled = responseJsonObject.optBoolean
                ("console_log", false);
        updateFeatureAvailability(IBGFeature.CONSOLE_LOGS, consoleLogEnabled);

        boolean ibgLogEnabled = responseJsonObject.optBoolean("ibg_log", false);
        updateFeatureAvailability(IBGFeature.INSTABUG_LOGS, ibgLogEnabled);

        boolean networkLogsEnabled = responseJsonObject.optBoolean("network_log", false);
        updateFeatureAvailability(IBGFeature.NETWORK_LOGS, networkLogsEnabled);

        boolean userDataEnabled = responseJsonObject.optBoolean("user_data",
                true);
        updateFeatureAvailability(IBGFeature.USER_DATA, userDataEnabled);

        boolean surveysEnabled = responseJsonObject.optBoolean("surveys", false);
        updateFeatureAvailability(IBGFeature.SURVEYS, surveysEnabled);

        boolean viewHierarchyV2Enabled = responseJsonObject.optBoolean
                ("view_hierarchy_v2", false);
        updateFeatureAvailability(IBGFeature.VIEW_HIERARCHY_V2, viewHierarchyV2Enabled);

        boolean userEventsEnabled = responseJsonObject.optBoolean
                ("user_events", false);
        updateFeatureAvailability(IBGFeature.USER_EVENTS, userEventsEnabled);

        boolean disclaimerEnabled = responseJsonObject.optBoolean("disclaimer_text", false);
        updateFeatureAvailability(IBGFeature.DISCLAIMER, disclaimerEnabled);

        boolean sessionProfilerEnabled = responseJsonObject.optBoolean("sessions_profiler", false);
        updateFeatureAvailability(IBGFeature.SESSION_PROFILER, sessionProfilerEnabled);

        boolean featuresRequestEnabled = responseJsonObject.optBoolean("feature_requests", false);
        updateFeatureAvailability(IBGFeature.FEATURE_REQUESTS, featuresRequestEnabled);

        boolean vpCustomizationsEnabled = responseJsonObject.optBoolean("vp_customizations", false);
        updateFeatureAvailability(IBGFeature.VP_CUSTOMIZATION, vpCustomizationsEnabled);

        boolean experimentalPromptFREnabled = responseJsonObject.optBoolean("experimental_prompt_fr",
                DEFAULT_EXPERIMENTAL_FEATURE_AVAILABILITY);
        updateExperimentalFeatureAvailability(IBGFeature.FEATURE_REQUESTS, experimentalPromptFREnabled);

        boolean announcementsEnabled = responseJsonObject.optBoolean("announcements", false);
        updateFeatureAvailability(IBGFeature.ANNOUNCEMENTS, announcementsEnabled);

        boolean userAttributesEnabled = responseJsonObject.optBoolean("be_user_attributes", false);
        updateExperimentalFeatureAvailability(IBGFeature.BE_USER_ATTRIBUTES, userAttributesEnabled);

        boolean disableSigning = responseJsonObject.optBoolean("disable_signing", false);
        updateFeatureAvailability(IBGFeature.BE_DISABLE_SIGNING, !disableSigning);

        boolean usersKeysEnabled = responseJsonObject.optBoolean("users_keys", false);
        SettingsManager.getInstance().setUsersPageEnabled(usersKeysEnabled);

        boolean vzMessagesCustomAppRatingUi = responseJsonObject.optBoolean("vz_messages_custom_app_rating_ui", false);
        updateFeatureAvailability(IBGFeature.VZ_MESSAGES_CUSTOM_APPRATING_UI, vzMessagesCustomAppRatingUi);

        boolean databaseTransactionsDisabled = responseJsonObject.optBoolean("android_db_transaction_disabled", true);
        setDatabaseTransactionDisabled(databaseTransactionsDisabled);

        boolean productionUsageDetection = responseJsonObject.optBoolean("production_usage_detection", false);
        updateFeatureAvailability(IBGFeature.PRODUCTION_USAGE_DETECTION, productionUsageDetection);

        JSONObject sdkLogFeatureJson = responseJsonObject.optJSONObject("sdk_log_v2");
        LoggingSettingResolver.getInstance().setLoggingFeatureSettings(sdkLogFeatureJson);

        CoreServiceLocator.getDevicePerformanceClassConfig().handle(responseJsonObject);

        TokenMappingConfigsHandler tokenMappingHandler = new TokenMappingConfigsHandlerImpl();
        tokenMappingHandler.handle(responseJsonObject);

        JSONObject sessionsConfig = responseJsonObject.optJSONObject("sessions");
        String json = sessionsConfig == null ? "{}" : sessionsConfig.toString();
        SettingsManager.getInstance().setSessionsSyncConfigurations(json);
        if (sessionsConfig != null)
            IBGSessionServiceLocator.getConfigurationsHandler().handle(sessionsConfig.optJSONObject(FLAG_V3_SESSIONS));
        updateSessionStitching(sessionsConfig);
        IBGSessionServiceLocator.getConfigurationsHandler().handlePeriodicUpdateFlags(responseJsonObject);

        SessionManualControllerServiceLocator.getSessionManualControllerConfigs().handle(responseJsonObject);

        boolean encryptionEnabled = responseJsonObject.optBoolean("android_encryption", false);
        Feature.State encryptionUpdatedState = encryptionEnabled ? Feature.State.ENABLED : Feature.State.DISABLED;
        Feature.State oldEncryptionState = getEncryptionState();
        setEncryptionState(encryptionEnabled);
        SettingsManager.getInstance().setFeatureEnabled(IBGFeature.ENCRYPTION, encryptionEnabled);

        if (oldEncryptionState != encryptionUpdatedState) {
            SharedPreferencesMigrationEngine.startMigration(encryptionEnabled, Instabug.getApplicationContext());
            IBGCoreEventPublisher.post(IBGSdkCoreEvent.EncryptionStateChanged.INSTANCE);
        }
        double dbEncryptionPercentage = responseJsonObject.optDouble("an_db_encryption_v2", 0);
        IBGPercentageFlagsResolver.resolve(IBGFeature.DB_ENCRYPTION, dbEncryptionPercentage);
        setDbEncryptionState(SettingsManager.getInstance().getFeatureState(IBGFeature.DB_ENCRYPTION, false) == Feature.State.ENABLED);

        boolean isSessionScreenOffMonitorEnabled = responseJsonObject.optBoolean("an_exp_session_screenoff", true);
        SettingsManager.getInstance().setFeatureEnabled(IBGFeature.SCREEN_OFF_MONITOR, isSessionScreenOffMonitorEnabled);

        ExperimentsManager experimentsManager = ServiceLocator.getExperimentsManager();
        if (experimentsManager != null) {
            experimentsManager.updateExperimentsFeature(responseJsonObject);
        }
        FeaturesFlagServiceLocator.getFeaturesFlagsConfigsHandler().handleConfigs(responseJsonObject);
        CoreServiceLocator.getCompositeFeatureFlagHandler().handleConfigs(responseJsonObject);
        NetworkInterceptionServiceLocator.configurationHandler().handleConfigs(responseJsonObject);

        saveQueueThresholds(responseJsonObject);
        extractCustomIdentifiedEmail(responseJsonObject);
        RatingDialogDetectionConfigurationsImpl.INSTANCE.handle(responseJsonObject);
        CoreServiceLocator.getNetworkDiagnosticsConfigurationHandler().handle(responseJsonObject);
        CoreServiceLocator.getSdkExperimentsManager().handleConfigs(responseJsonObject);
        IBGLowMemoryHandler.handleConfiguration(responseJsonObject);
    }

    private void saveQueueThresholds(JSONObject features) {

        long dequeueThreshold = features.optLong("android_db_time_to_dequeue_threshold", 4000L);
        long completionThreshold = features.optLong("android_db_time_to_completion_threshold", 5000L);

        PersistableSettings persistableSettings = PersistableSettings.getInstance();
        if (persistableSettings != null) {
            persistableSettings.saveDequeueThreshold(dequeueThreshold);
            persistableSettings.saveCompletionThreshold(completionThreshold);
        }

    }

    private void extractCustomIdentifiedEmail(JSONObject responseJsonObject) {
        boolean isCustomIdentifiedEmailUsed = responseJsonObject.optBoolean("crashes_custom_identified_email", false);
        updateFeatureAvailability(IBGFeature.CRASHES_CUSTOM_IDENTIFIED_EMAIL, isCustomIdentifiedEmailUsed);
    }

    /**
     * Set last fetched at
     *
     * @param planFeaturesFetchedAt last fetched at
     * @param context               to use for getting SharedPreferences
     */
    @VisibleForTesting
    public void setLastFetchedAt(long planFeaturesFetchedAt, Context context) {
        SharedPreferences sharedPreferences = CoreServiceLocator.getInstabugSharedPreferences(context,
                SettingsManager.INSTABUG_SHARED_PREF_NAME);
        if (sharedPreferences == null) return;
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putLong(LAST_FETCHED_AT, planFeaturesFetchedAt);
        editor.apply();
    }

    public void setTemporaryDisabled() {
        temporaryDisabled = true;
    }

    private void updateSessionStitching(@Nullable JSONObject sessionsJsonObject) {
        if (sessionsJsonObject == null) return;
        boolean sessionStitchingEnabled = sessionsJsonObject.optBoolean(SESSION_STITCHING, SESSION_STITCHING_DEFAULT_STATE);
        SettingsManager.getInstance().setFeatureEnabled(IBGFeature.SDK_STITCHING, sessionStitchingEnabled);
        updateFeatureAvailability(IBGFeature.SDK_STITCHING, sessionStitchingEnabled);
        if (sessionsJsonObject.has(SESSION_STITCHING_TIME_OUT)) {
            int sessionStitchingTimeout = sessionsJsonObject.optInt(SESSION_STITCHING_TIME_OUT);
            SettingsManager.getInstance().setSessionStitchingTimeout(sessionStitchingTimeout);
        }
    }

    public boolean isSessionStitchingEnabled() {
        if (isFeaturesFetchedBefore()) {
            return SettingsManager
                    .getInstance()
                    .getFeatureState(IBGFeature.SDK_STITCHING, SESSION_STITCHING_DEFAULT_STATE)
                    == Feature.State.ENABLED;
        }
        return true;
    }

}
