package com.instabug.library.tracking;

import android.app.Activity;
import android.app.Application;
import android.content.ComponentCallbacks2;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.SystemClock;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.appcompat.app.AppCompatActivity;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.PresentationManager;
import com.instabug.library.SessionManager;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.eventbus.AppStateEvent;
import com.instabug.library.core.eventbus.AppStateEventBus;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.utils.memory.IBGLowMemoryHandler;
import com.instabug.library.logging.InstabugUserEventLogger;
import com.instabug.library.sessionV3.manager.SessionLifecycleHandler;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.KeyboardEventListener;
import com.instabug.library.util.threading.PoolProvider;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

public class InstabugActivityLifecycleListener implements Application.ActivityLifecycleCallbacks,
        ComponentCallbacks2 {
    private boolean isConfigChanged = false;

    private long lastResumeEvent = 0;
    private final Set<String> set = new HashSet<>();
    private boolean isPaused = false;

    private static final String NOTIFY_CHANGES_KEY = "IBG-NOTIFY_CHANGES_KEY";
    private final Map<Integer, InstabugFragmentLifecycleListener> fragmentLifecycleListenerMap;

    @NonNull
    private StartedActivitiesCounter startedActivitiesCounter;
    Map<Integer, KeyboardEventListener> activityKeyboardEventListenerMap;

    public InstabugActivityLifecycleListener() {
        activityKeyboardEventListenerMap = new HashMap<>();
        fragmentLifecycleListenerMap = new HashMap<>();
        //Direct access here as calling CoreServiceLocator.getStartedActivitiesCounter()
        // will cause a regression in app start up time
        startedActivitiesCounter = StartedActivitiesCounterSingleton.INSTANCE;
    }

    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle bundle) {
        set.add(activity.getClass().getSimpleName());
        InstabugInternalTrackingDelegate.getInstance().handleActivityCreatedEvent(activity);
        registerFragmentCallbacks(activity);
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityCreated(activity, bundle);
        SessionLifecycleHandler.INSTANCE.onActivityCreated(activity, bundle);
    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        startedActivitiesCounter.onActivityStarted();
        CoreServiceLocator.getScreenOffHandler().onActivityStarted();
        InstabugInternalTrackingDelegate.getInstance().handleActivityStartedEvent(activity);
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityStarted(activity);
    }

    @Override
    public void onActivityResumed(@NonNull final Activity activity) {
        IBGWindowCallbacksHandler.registerWindowCallbacksIfNeeded(activity);
        registerKeyboardListener(activity);
        if (SettingsManager.getInstance().isInBackground()) {
            AppStateEventBus.INSTANCE.post(new AppStateEvent.ForegroundAppStateEvent());
        }
        PoolProvider.postIOTask(new Runnable() {
            @Override
            public void run() {
                if (SettingsManager.getInstance().isInBackground()) {
                    if (DeviceStateProvider.getFreeInternalStorage() < 50) {
                        // device is running on low disk space. disabling instabug
                        Instabug.pauseSdk();
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Instabug was disabled temporary because of low disk storage " +
                                "'< 50MB' and it will be resumed next session one there is available disk storage");
                        isPaused = true;
                        return;
                    }

                    if (IBGLowMemoryHandler.isSDKPausedOnLowMemory()) {
                        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.INSTABUG, Feature.State.ENABLED);
                        Instabug.resumeSdk();
                        IBGLowMemoryHandler.setSDKPausedOnLowMemory(false);
                    }

                    // start session
                    SessionManager.getInstance().handleSessionStartEvent();
                    SettingsManager.getInstance().setInBackground(false);
                }

                PoolProvider.postMainThreadTask(new Runnable() {
                    @Override
                    public void run() {
                        /*
                            this block of code does some things
                            1. Makes sure that on changing orientation the front facing activity is the one that being
                               triggered
                            2. An issue that make bugreporting called twice on dismissing this because we are always
                               killing it by starting it again and forcing it to close from outside the activity itself.

                            At last this code SHOULD make sure that notifyActivityChanged() is only called when the current
                            activity is changed from any activity to NON-INSTABUG Activity.
                         */
                        PresentationManager presentationManager = PresentationManager.getInstance();
                        if (SystemClock.elapsedRealtime() - lastResumeEvent < 300) {
                            return;
                        }
                        if (presentationManager != null) {
                            if (presentationManager.getCurrentActivityName().equalsIgnoreCase(activity.getLocalClassName())
                                    && isConfigChanged) {
                                lastResumeEvent = SystemClock.elapsedRealtime();
                            }
                            presentationManager.setCurrentActivity(activity);

                            if (isConfigChanged) {
                                isConfigChanged = false;
                            } else {
                                PoolProvider.postOrderedIOTask(NOTIFY_CHANGES_KEY, () -> {
                                    if (!(activity instanceof _InstabugActivity)) {
                                        presentationManager.notifyActivityChanged();
                                    }
                                });

                            }
                        }
                    }
                });
            }
        });
        InstabugInternalTrackingDelegate.getInstance().handleActivityResumedEvent(activity);
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityResumed(activity);
        SessionLifecycleHandler.INSTANCE.onActivityResumed(activity);
    }

    private void registerFragmentCallbacks(Activity activity) {
        if (activity instanceof AppCompatActivity && !(activity instanceof _InstabugActivity)) {
            InstabugFragmentLifecycleListener instabugFragmentLifecycleListener = new InstabugFragmentLifecycleListener();
            ((AppCompatActivity) activity).getSupportFragmentManager()
                    .registerFragmentLifecycleCallbacks(instabugFragmentLifecycleListener, true);
            fragmentLifecycleListenerMap.put(activity.hashCode(), instabugFragmentLifecycleListener);
        }
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
        if (!(activity instanceof _InstabugActivity)) {
            Locale appLocale = Locale.getDefault();
            InstabugSDKLogger.d(Constants.LOG_TAG, "Setting app locale to " + appLocale.toString());
            SettingsManager.getInstance().setAppLocale(appLocale);
        }
        InstabugInternalTrackingDelegate.getInstance().handleActivityPausedEvent(activity);
        IBGWindowCallbacksHandler.unregisterWindowCallbacks(activity);
        unregisterKeyboardListener(activity);
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityPaused(activity);
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        startedActivitiesCounter.onActivityStopped();
        InstabugInternalTrackingDelegate.getInstance().handleActivityStoppedEvent(activity);
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityStopped(activity);
        SessionLifecycleHandler.INSTANCE.onActivityStopped(activity);
    }

    @VisibleForTesting
    void registerKeyboardListener(Activity activity) {
        if (activity == null)
            return;
        KeyboardEventListener listener = new KeyboardEventListener(activity,
        isKeyboardOpen -> CoreServiceLocator.getReproStepsProxy().logKeyboardEvent(isKeyboardOpen));
        activityKeyboardEventListenerMap.put(activity.hashCode(), listener);
    }

    private void unregisterKeyboardListener(Activity activity) {
        if (activity == null)
            return;

        // unregister keyboardListener
        KeyboardEventListener listener = activityKeyboardEventListenerMap.get(activity.hashCode());
        if (listener != null)
            listener.unregisterKeyboardListener();
        activityKeyboardEventListenerMap.remove(activity.hashCode());
    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle bundle) {
        InstabugSDKLogger.d(Constants.LOG_TAG, activity.getClass().getSimpleName() + " SaveInstanceState");
    }

    @Override
    public void onActivityDestroyed(Activity activity) {
        set.remove(activity.getClass().getSimpleName());
        if (set.isEmpty()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "app is getting terminated, clearing user event logs");
            InstabugUserEventLogger.getInstance().clearAll();
        }
        InstabugInternalTrackingDelegate.getInstance().handleActivityDestroyedEvent(activity);
        if (activity instanceof AppCompatActivity && !(activity instanceof _InstabugActivity)) {
            InstabugFragmentLifecycleListener listener = fragmentLifecycleListenerMap.get(activity.hashCode());
            if (listener != null) {
                ((AppCompatActivity) activity).getSupportFragmentManager()
                        .unregisterFragmentLifecycleCallbacks(listener);
            }
            fragmentLifecycleListenerMap.remove(activity.hashCode());
        }
        CoreServiceLocator.getScreenActivityComponentsMonitor().onActivityDestroyed(activity);
        SessionLifecycleHandler.INSTANCE.onActivityDestroyed(activity);
    }

    @Override
    public void onTrimMemory(int level) {
        switch (level) {
            case TRIM_MEMORY_UI_HIDDEN:
                AppStateEventBus.INSTANCE.post(new AppStateEvent.BackgroundAppStateEvent());
                SessionLifecycleHandler.INSTANCE.onTrimMemory(level);
                PoolProvider.postIOTask(() -> {
                    SettingsManager.getInstance().setInBackground(true);
                    SettingsManager.getInstance().setLastForegroundTime(System.currentTimeMillis());
                    SessionManager.getInstance().finishSession();
                });
                if (isPaused) {
                    Instabug.resumeSdk();
                    isPaused = false;
                    return;
                }
                break;

            case TRIM_MEMORY_RUNNING_LOW:
                PoolProvider.getApiExecutor().execute(IBGLowMemoryHandler::pauseSDKIfNeeded);
                break;

            default:
                break;
        }

    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        isConfigChanged = true;
        InstabugInternalTrackingDelegate.getInstance().handleConfigurationChanged(newConfig);
    }

    @Override
    public void onLowMemory() {

    }

}
