package com.instabug.library.networkv2;

import static com.instabug.library.networkv2.request.RequestExtKt.updateDelayedReportedAt;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.networkDiagnostics.manager.NetworkDiagnosticsManager;
import com.instabug.library.networkv2.connection.FileDownloadConnectionManager;
import com.instabug.library.networkv2.connection.InstabugBaseConnectionManager;
import com.instabug.library.networkv2.connection.MultipartConnectionManager;
import com.instabug.library.networkv2.connection.NormalConnectionManager;
import com.instabug.library.networkv2.detectors.IBGNetworkAvailabilityManager;
import com.instabug.library.networkv2.execptions.RequestInterruptedException;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.networkv2.utils.NetworkRequestRetryingHelper;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.security.SecureRandom;
import java.util.Random;

/**
 * Instabug Network Manager that build the connection and perform the request
 */
@Keep
public class NetworkManager implements INetworkManager {
    @Nullable
    private OnDoRequestListener onDoRequestListener;

    public NetworkManager() {

    }

    @Override
    @Nullable
    public OnDoRequestListener getOnDoRequestListener() {
        return onDoRequestListener;
    }

    public NetworkManager(@Nullable OnDoRequestListener onDoRequestListener) {
        this.onDoRequestListener = onDoRequestListener;
    }

    @Override
    public void setOnDoRequestListener(@Nullable OnDoRequestListener onDoRequestListener) {
        this.onDoRequestListener = onDoRequestListener;
    }
    @VisibleForTesting
    public static NetworkManager newInstance() {
        return new NetworkManager();
    }

    public static boolean isOnline() {
        return IBGNetworkAvailabilityManager.INSTANCE.isOnline();
    }

    @WorkerThread
    private void doRequest(@NonNull @IBGNetworkWorker String workerThreadName,
                           @NonNull final InstabugBaseConnectionManager connectionManager,
                           @NonNull Request request,
                           Request.Callbacks<RequestResponse, Throwable> callback) {
        PoolProvider.getNetworkingSingleThreadExecutor(workerThreadName).execute(() -> {
            handleRequest(request, connectionManager, callback);
        });
    }

    @WorkerThread
    private void doRequestOnSameThread(@NonNull final InstabugBaseConnectionManager connectionManager,
                                       @NonNull Request request,
                                       Request.Callbacks<RequestResponse, Throwable> callback) {
        handleRequest(request, connectionManager, callback);
    }

    private void handleRequest(Request request,
                               InstabugBaseConnectionManager connectionManager,
                               Request.Callbacks<RequestResponse, Throwable> callback
    ) {
        NetworkRequestRetryingHelper.resetRequestAttemptCounter(request);
        boolean shouldKeepRetry = false;
        NetworkDiagnosticsManager networkDiagnosticsManager = CoreServiceLocator.getNetworkDiagnosticsManager();
        do {
            try {
                performRequest(request, connectionManager, callback);
                break;
            } catch (IOException e) {
                shouldKeepRetry = NetworkRequestRetryingHelper.shouldRetryRequest(request);
                if (shouldKeepRetry) {
                    request = updateDelayedReportedAt(request);
                    callback.onRetrying(e);
                    try {
                        long timeout = NetworkRequestRetryingHelper.getNextRetryTimeout(request);
                        InstabugSDKLogger.d(
                                Constants.LOG_TAG,
                                "Request " + request.getRequestUrl() +
                                        " failed to connect to network, retrying in " + timeout + " seconds."
                        );
                        Thread.sleep(timeout * 1000);
                    } catch (InterruptedException ex) {
                        throw new RequestInterruptedException(ex, "Thread is interrupted while waiting for the next network request retry!");
                    }
                    NetworkRequestRetryingHelper.incrementRequestAttemptCounter(request);
                } else if (callback != null) {
                    networkDiagnosticsManager.onNetworkRequestFailed();
                    callback.onFailed(e);
                }
            } catch (Exception e) {
                if (callback != null) {
                    callback.onFailed(e);
                }
                networkDiagnosticsManager.onNetworkRequestFailed();
            } catch (OutOfMemoryError error) {
                if (callback != null) {
                    callback.onFailed(error);
                }
                networkDiagnosticsManager.onNetworkRequestFailed();
            }
        } while (shouldKeepRetry);
    }

    private void performRequest(Request request,
                                InstabugBaseConnectionManager connectionManager,
                                Request.Callbacks<RequestResponse, Throwable> callback
    ) throws Exception, OutOfMemoryError {
        if (onDoRequestListener != null) {
            onDoRequestListener.onRequestStarted(request);
        }

        HttpURLConnection connection = null;
        NetworkDiagnosticsManager networkDiagnosticsManager = CoreServiceLocator.getNetworkDiagnosticsManager();
        try {
            connection = connectionManager.create(request);
            if (connection == null) return;
            if (connection.getResponseCode() >= RequestResponse.HttpStatusCode._4xx.BAD_REQUEST) {
                Throwable error = connectionManager.handleServerError(connection);
                if (callback != null)
                    callback.onFailed(error);
                networkDiagnosticsManager.onNetworkRequestFailed();
                return;
            }
            RequestResponse requestResponse = connectionManager.handleResponse(connection, request);

            if (callback != null) {
                callback.onSucceeded(requestResponse);
            }
            networkDiagnosticsManager.onNetworkRequestSucceeded();
            if (onDoRequestListener != null) {
                onDoRequestListener.onComplete();
            }
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
            try {
                if (connection != null && connection.getInputStream() != null) {
                    connection.getInputStream().close();
                }
            } catch (Exception e) {
                try {
                    if (connection.getErrorStream() != null) {
                        connection.getErrorStream().close();
                    }
                } catch (Exception ex) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "failed to close connection input stream for url " + request.getRequestUrl(), e);
                }
            }
        }
    }

    @WorkerThread
    @Override
    public void doRequest(@IBGNetworkWorker String ibgNetworkWorker, @RequestType int requestType, @NonNull Request request, Request.Callbacks<RequestResponse, Throwable> callback) {
        if (isOnline()) {
            switch (requestType) {
                case RequestType.FILE_DOWNLOAD:
                    doRequest(ibgNetworkWorker, new FileDownloadConnectionManager(), request, callback);
                    break;
                case RequestType.MULTI_PART:
                    doRequest(ibgNetworkWorker, new MultipartConnectionManager(), request, callback);
                    break;
                case RequestType.NORMAL:
                    doRequest(ibgNetworkWorker, new NormalConnectionManager(), request, callback);
                    break;
                case RequestType.UNDEFINED:
                default:
                    InstabugSDKLogger.e(Constants.LOG_TAG, "undefined request type for " + request.getRequestUrlForLogging());
                    break;
            }
        } else {
            InstabugSDKLogger.d(com.instabug.library.Constants.LOG_TAG, "Device internet is disabled, can't make request: " + request.getEndpoint());
            callback.onDisconnected();
        }
    }

    @WorkerThread
    @Override
    public void doRequestOnSameThread(
            @RequestType int requestType,
            @NonNull Request request,
            Request.Callbacks<RequestResponse, Throwable> callback
    ) {
        doRequestOnSameThread(requestType, request, false, callback);
    }

    @WorkerThread
    @Override
    public void doRequestOnSameThread(
            int requestType,
            @NonNull Request request,
            boolean onlineOverride,
            Request.Callbacks<RequestResponse, Throwable> callback
    ) {
        if (onlineOverride || isOnline()) {
            switch (requestType) {
                case RequestType.FILE_DOWNLOAD:
                    doRequestOnSameThread(new FileDownloadConnectionManager(), request, callback);
                    break;
                case RequestType.MULTI_PART:
                    doRequestOnSameThread(new MultipartConnectionManager(), request, callback);
                    break;
                case RequestType.NORMAL:
                    doRequestOnSameThread(new NormalConnectionManager(), request, callback);
                    break;
                case RequestType.UNDEFINED:
                default:
                    InstabugSDKLogger.e(Constants.LOG_TAG, "undefined request type for " + request.getRequestUrlForLogging());
                    break;
            }
        } else {
            InstabugSDKLogger.d(com.instabug.library.Constants.LOG_TAG, "Device internet is disabled, can't make request: " + request.getEndpoint());
            callback.onDisconnected();
        }
    }

    public interface OnDoRequestListener {
        void onRequestStarted(Request request);

        void onStart();

        void onComplete();
    }

}
