package com.instabug.library.internal.video;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.internal.InstabugMediaProjectionIntent;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;

import java.io.File;

import io.reactivexport.disposables.Disposable;

public class InternalAutoScreenRecorderHelper implements ScreenRecordingContract, DefaultActivityLifeCycleEventHandler {

    @Nullable
    private static InternalAutoScreenRecorderHelper INSTANCE;
    private ScreenRecordingFileHolder fileHolder;
    private Disposable sessionDisposable;
    @Nullable
    private ActivityLifecycleSubscriber activityLifeCycleSubscriber = null;
    private boolean isCrashOccurred = false;

    public InternalAutoScreenRecorderHelper() {
        fileHolder = new ScreenRecordingFileHolder();
        subscribeToSessionEvents();
        subscribeToActivityLifeCycleEvents();
    }
    private void subscribeToSessionEvents() {
        if (sessionDisposable == null || sessionDisposable.isDisposed()) {
            sessionDisposable = SessionStateEventBus.getInstance()
                    .subscribe(sessionState -> {
                        if (sessionState == SessionState.FINISH && !isCrashOccurred() && isEnabled()) {
                            //Session finishes while manual screen recording, keep the video and go back to report screen
                            AutoScreenRecordingEventBus.getInstance().post(ScreenRecordingServiceAction.CustomeActions.STOP_DELETE);

                            SettingsManager.getInstance().setAutoScreenRecordingDenied(false);
                            unsubscribeFromSessionEvents();
                        }
                    });
        }
    }

    public boolean isCrashOccurred() {
        return isCrashOccurred;
    }
    public void setCrashOccurred(boolean isCrashOccurred) {this.isCrashOccurred = isCrashOccurred;}
    private void unsubscribeFromSessionEvents() {
        if (!sessionDisposable.isDisposed()) {
            sessionDisposable.dispose();
        }
    }
    private void subscribeToActivityLifeCycleEvents() {
        if (activityLifeCycleSubscriber == null) {
            activityLifeCycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
            activityLifeCycleSubscriber.subscribe();
        }
    }

    @Override
    public void handleActivityResumed() {
        if (isEnabled() && !SettingsManager.getInstance().isProcessingForeground()) {
            subscribeToSessionEvents();
            Handler handler = new Handler();
            handler.postDelayed(() -> InternalAutoScreenRecorderHelper.getInstance().start(), 700);
        }
    }

    public static InternalAutoScreenRecorderHelper getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new InternalAutoScreenRecorderHelper();
        }

        return INSTANCE;
    }
    @VisibleForTesting
    static void release() {
        if (INSTANCE != null && INSTANCE.activityLifeCycleSubscriber != null) {
            INSTANCE.activityLifeCycleSubscriber.unsubscribe();
        }
        INSTANCE = null;
    }
    public void start() {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            return;
        }

        if (SettingsManager.getInstance().isScreenCurrentlyRecorded() || SettingsManager.getInstance().isAutoScreenRecordingDenied() || !isEnabled()) {
            return;
        }

        Activity activity = InstabugInternalTrackingDelegate.getInstance().getTargetActivity();

        if (activity != null
                && !(activity instanceof _InstabugActivity)
                && SettingsManager.getInstance().isAppOnForeground()
                && InstabugStateProvider.getInstance().getState() == InstabugState.ENABLED
                && !InstabugCore.isForegroundBusy()) {

            Intent intent = new Intent(activity, RequestPermissionActivity.class);
            activity.startActivity(intent);
            activity.overridePendingTransition(0, 0);
        }
    }

    @Override
    @Nullable
    public Uri getAutoScreenRecordingFileUri() {
        return fileHolder.getAutoScreenRecordingFileUri();
    }

    @Override
    public void delete() {
        fileHolder.delete();
    }
    @Override
    public void clear() {
        fileHolder.clear();
    }
    @Override
    public boolean isEnabled() {
        return SettingsManager.getInstance().isAutoScreenRecordingEnabled();
    }
    public void setAutoScreenRecordingFile(File autoScreenRecordingFile) {
        fileHolder.setAutoScreenRecordingFile(autoScreenRecordingFile);
    }
    static void setStaticIntentData(@Nullable Intent staticIntentData) {
        InstabugMediaProjectionIntent.setMediaProjectionIntent(staticIntentData);
    }
    static void setStaticResultCode(int staticResultCode) {
        InstabugMediaProjectionIntent.setStaticResultCode(staticResultCode);
    }
}
