package com.instabug.library.internal.resolver;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.internal.dataretention.files.FileInspector;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.orchestrator.DisposeDataAction;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * A logging setting resolver which will hold the logging setting to get any settings
 * Also, it start an observer to listen on logging level changes.
 */
public class LoggingSettingResolver {

    private static LoggingSettingResolver loggingSettingResolver;

    private static final String TAG = "LoggingSettingResolver";

    /**
     * A callback to keep track with all changes with the logging level.
     */
    private LoggingLevelChangeListener loggingLevelChangeListener;

    @Nullable
    private volatile LoggingSettings loggingSettings;


    private LoggingSettingResolver() {
        initLoggingLevelCallbacks();
        getLoggingSettingsFromPreferences();
    }

    public static synchronized LoggingSettingResolver getInstance() {
        if (loggingSettingResolver == null) {
            loggingSettingResolver = new LoggingSettingResolver();
        }
        return loggingSettingResolver;
    }


    /**
     * Resolve logging settings from json object to return a logging setting
     * it will return a new object with the new settings.
     *
     * @param jsonObject response of logging settings.
     * @return logging settings.
     * @throws JSONException
     */
    private LoggingSettings resolveLoggingSettings(@NonNull JSONObject jsonObject) throws JSONException {
        LoggingSettings loggingSettings = new LoggingSettings();
        loggingSettings.fromJson(jsonObject);
        return loggingSettings;
    }

    /**
     * start observing on logging level changes
     */
    private void initLoggingLevelCallbacks() {

        loggingLevelChangeListener = new LoggingLevelChangeListener() {
            @Override
            public void onLevelChanged(@LoggingSettings.LogLevels int logLevel) {
                InstabugSDKLogger.onDiskLoggingLevelChanged(logLevel);
                if (logLevel == LoggingSettings.LogLevels.NO_LOGS) {
                    ActionsOrchestrator.obtainOrchestrator()
                            .addWorkerThreadAction(new DisposeDataAction(FileInspector.create()))
                            .orchestrate();
                }
            }
        };
    }

    public void setLoggingFeatureSettings(@Nullable JSONObject sdkLogFeatureJSon) throws JSONException {
        if (sdkLogFeatureJSon == null) {
            loggingSettings = new LoggingSettings();
            notifyStateChange(LoggingSettings.LogLevels.NO_LOGS);
            SettingsManager.getInstance().setLoggingFeatureSettings(null);
            return;
        }
        this.loggingSettings = resolveLoggingSettings(sdkLogFeatureJSon);
        SettingsManager.getInstance().setLoggingFeatureSettings(sdkLogFeatureJSon.toString());
        if (loggingSettings != null)
            notifyStateChange(loggingSettings.getLevel());
    }

    private void notifyStateChange(@LoggingSettings.LogLevels int level) {
        if (loggingLevelChangeListener == null) {
            initLoggingLevelCallbacks();
        }
        loggingLevelChangeListener.onLevelChanged(level);
    }

    /**
     * A method to get logging setting
     *
     * @return logging setting or null if logging setting not found which means that the logging should be disabled
     */
    @Nullable
    public LoggingSettings getLoggingSettings() {
        return loggingSettings;
    }

    private void getLoggingSettingsFromPreferences() {
        try {
            if (loggingSettings == null) {
                String json = SettingsManager.getInstance().getLoggingFeatureSettings();
                if (json != null) {
                    LoggingSettings newLoggingSettings = new LoggingSettings();
                    newLoggingSettings.fromJson(json);
                    loggingSettings = newLoggingSettings;
                }
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.toString(), e);
        }
    }


    interface LoggingLevelChangeListener {
        void onLevelChanged(@LoggingSettings.LogLevels int level);
    }

}
