package com.instabug.library.internal.contentprovider;

import android.app.Application;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.ProviderInfo;
import android.os.Bundle;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Feature;
import com.instabug.library.Instabug;
import com.instabug.library.diagnostics.customtraces.IBGPendingTraceHandler;
import com.instabug.library.internal.stub.ContentProviderStub;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;

@Keep
public class InstabugContentProvider extends ContentProviderStub {

    @Override
    public void attachInfo(Context context, ProviderInfo info) {
        super.attachInfo(context, info);
        try {
            initialize(context);
        } catch (Exception e) {
            InstabugSDKLogger.e("IBG-CORE", "Error in content provider: " + e.getMessage(), e);
        }
    }

    private void initialize(Context context) throws PackageManager.NameNotFoundException {
        IBGPendingTraceHandler.setCoreCPStartTime(System.currentTimeMillis());
        if (context == null) return;

        Application appContext = null;
        if (context instanceof Application) {
            appContext = (Application) context;
        } else {
            Context applicationContext = context.getApplicationContext();
            if (applicationContext instanceof Application) {
                appContext = (Application) applicationContext;
            }
        }
        initSDK(appContext);
        initTrackingDelegate(appContext);
        initApplicationProvider(appContext);
        IBGPendingTraceHandler.setCoreCPEndTime(System.currentTimeMillis());
    }

    @NonNull
    String getInstabugToken(Application context) throws PackageManager.NameNotFoundException {
        if (context == null) return "";
        ApplicationInfo app = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        Bundle bundle = app.metaData;
        if (bundle != null) {
            String token = bundle.getString("com.instabug.APP_TOKEN");
            return token != null ? token : "";
        }
        return "";
    }

    private void initSDK(@Nullable Application appContext) throws PackageManager.NameNotFoundException {
        if (appContext != null) {
            new Instabug.Builder(appContext, getInstabugToken(appContext)).build(Feature.State.DISABLED);
        }
    }

    private void initApplicationProvider(@Nullable Application applicationContext) {
        if (applicationContext == null) return;
        InstabugApplicationProvider.init(applicationContext);
    }

    private void initTrackingDelegate(@Nullable Application applicationContext) {
        if (applicationContext == null) return;
        InstabugInternalTrackingDelegate.init(applicationContext);
    }
}
