package com.instabug.library.diagnostics.sdkEvents

import android.content.SharedPreferences
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher.post
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.diagnostics.configuration.ConfigurationChangedHandler
import com.instabug.library.diagnostics.configuration.ConfigurationChangedHandler.Companion.ENABLED_KEY
import com.instabug.library.diagnostics.sdkEvents.configurations.SDKEventConfigs
import com.instabug.library.diagnostics.sdkEvents.configurations.SDKEventConfigs.Companion.SDK_EVENTS_KEY
import com.instabug.library.diagnostics.sdkEvents.di.SDKEventsServiceLocator
import com.instabug.library.percentagefeatures.IBGPercentageFlagsResolver
import org.json.JSONArray
import org.json.JSONObject

internal class SDKEventsConfigurationChangedHandler : ConfigurationChangedHandler {

    private val sharedPreferences: SharedPreferences?
        get() = SDKEventsServiceLocator.preferences
    private val sdkEventsManager: SDKEventsManager
        get() = SDKEventsServiceLocator.sdkEventsManager
    private val sdkEventsConfigs: SDKEventConfigs
        get() = SDKEventsServiceLocator.sdkEventConfigs

    /*
    * keeps track of old allow list and feature state
    * in case of any changes of the old allow list that are not present at the new list will be removed from the database
    * it also keeps track of the old feature state when it is different from the new a feature updated core event will be sent
    */
    override fun onConfigurationChanged(json: JSONObject?) {
        json?.optJSONObject(SDK_EVENTS_KEY)?.run {
            handleSDKEventsState(optDouble(ENABLED_KEY))
            handleAllowList(optJSONArray(SDKEventConfigs.ALLOW_LIST_KEY))
        }
    }

    private fun handleSDKEventsState(enabledPercentage: Double) {
        val oldFeatureState = sdkEventsConfigs.isEnabled
        IBGPercentageFlagsResolver.resolve(
            SDKEventConfigs.FEATURE_NAME,
            enabledPercentage
        )
        checkFeatureStatusChanges(oldFeatureState)
    }

    private fun checkFeatureStatusChanges(oldFeatureState: Boolean) {
        //todo to be updated
        if (oldFeatureState != sdkEventsConfigs.isEnabled) post(Features.Updated)
    }

    private fun handleAllowList(allowListArray: JSONArray?) {
        val oldAllowList = sdkEventsConfigs.allowList
        parseAllowList(allowListArray).also(::cacheAllowList)
        clearNotAllowedSDKEvents(oldAllowList)
    }

    private fun parseAllowList(allowListArray: JSONArray?): List<String>? = mutableListOf<String>()
        .takeIf { allowListArray != null && allowListArray.length() > 0 }
        ?.apply { for (i in 0 until allowListArray!!.length()) add(allowListArray.getString(i)) }

    private fun cacheAllowList(allowList: List<String>?) = allowList?.let {
        sharedPreferences
            ?.edit()
            ?.putStringSet(SDKEventConfigs.ALLOW_LIST_KEY, it.toSet())
            ?.apply()
    } ?: sharedPreferences?.edit()?.remove(SDKEventConfigs.ALLOW_LIST_KEY)?.apply()

    private fun clearNotAllowedSDKEvents(oldEventList: Set<String>?) = sdkEventsConfigs.allowList
        .let {
            if (it.isNullOrEmpty() && it != oldEventList) sdkEventsManager.clearCache()
            else getNotAllowedSDKEvents(oldEventList, it)?.let(sdkEventsManager::deleteManyEvents)
        }

    private fun getNotAllowedSDKEvents(oldEventList: Set<String>?, newList: Set<String>?) = newList
        ?.let { oldEventList?.filter { oldListItem -> !newList.contains(oldListItem) } }
        ?.takeIf { it.isNotEmpty() }
}