package com.instabug.library.core.ui;

import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.ViewStub;
import android.widget.ImageButton;
import android.widget.TextView;

import com.instabug.library.R;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.util.SystemServiceUtils;
import com.instabug.library.util.ThemeApplier;

import androidx.annotation.CallSuper;
import androidx.annotation.LayoutRes;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;

/**
 * Created by tarek on 9/25/16.
 */
public abstract class ToolbarFragment<P extends BaseContract.Presenter> extends InstabugBaseFragment<P> {

    @Nullable
    protected ImageButton toolbarImageButtonDone, toolbarImageButtonClose;
    @Nullable
    private View instabugToolbar;
    @Nullable
    private TextView titleTextView;

    @CallSuper
    @Override
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        initToolbarViews();

        ViewStub content = findViewById(R.id.instabug_content);
        if (content != null) {
            content.setLayoutResource(getContentLayout());
            content.inflate();
        }

        initContentViews(rootView, savedInstanceState);

        setTitle(getTitle());
        ThemeApplier.setScreenBackground(rootView, getIBGTheme());
    }

    private void initToolbarViews() {
        toolbarImageButtonDone = findViewById(R.id.instabug_btn_toolbar_right);
        instabugToolbar = findViewById(R.id.instabug_toolbar);
        titleTextView = findViewById(R.id.instabug_fragment_title);

        if (toolbarImageButtonDone != null) {
            toolbarImageButtonDone.setOnClickListener(v -> {
                FragmentActivity activity = getActivity();
                if (activity != null) {
                    SystemServiceUtils.hideInputMethod(activity);
                }
                onDoneButtonClicked();
            });
        }
        toolbarImageButtonClose = findViewById(R.id.instabug_btn_toolbar_left);
        if (toolbarImageButtonClose != null) {
            toolbarImageButtonClose.setOnClickListener(v -> onCloseButtonClicked());
        }
        styleToolbar();
    }

    private void styleToolbar() {
        if (instabugToolbar != null)
            instabugToolbar.setBackgroundColor(InstabugCore.getPrimaryColor());
        ThemeApplier.applyTitleStyle(titleTextView, getIBGTheme());
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_fragment_toolbar;
    }

    /**
     * @return content layout resource id.
     */
    protected abstract
    @LayoutRes
    int getContentLayout();

    /**
     * Init fragment content views
     *
     * @param rootView           fragment root view.
     * @param savedInstanceState saved instance state
     */
    protected abstract void initContentViews(View rootView, @Nullable Bundle savedInstanceState);

    /**
     * set fragment title
     *
     * @param title fragment title
     */
    protected void setTitle(String title) {
        if (rootView == null) {
            return;
        }
        TextView titleTextView = findViewById(R.id.instabug_fragment_title);
        if (titleTextView != null)
            titleTextView.setText(title);
    }

    /**
     * @return Fragment title
     */
    protected abstract String getTitle();

    protected abstract void onDoneButtonClicked();

    protected void onCloseButtonClicked() {
        FragmentActivity activity = getActivity();
        if (activity != null) {
            SystemServiceUtils.hideInputMethod(activity);
            activity.onBackPressed();
        } else Log.w("ToolbarFragment",
                "onCloseButtonClicked can't be executed due to null getActivity() reference");
    }

    @Nullable
    protected IBGTheme getIBGTheme() {
        return presenter != null && presenter.getTheme() != null ? presenter.getTheme() : null;
    }

}
