package com.instabug.library.settings;

import static com.instabug.library.diagnostics.customtraces.settings.CustomTracesSettingsResolver.DEFAULT_MAX_COUNT;
import static com.instabug.library.experiments.constants.Constants.EXPERIMENTS_STORE_LIMIT_FALLBACK;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.Uri;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.BuildConfig;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.LogLevel;
import com.instabug.library.OnSdkDismissCallback;
import com.instabug.library.OnSdkInvokedCallback;
import com.instabug.library.Platform;
import com.instabug.library.ReproConfigurations;
import com.instabug.library.diagnostics.DiagnosticsCoreEventHandler;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.InvocationManagerContract;
import com.instabug.library.invocation.OnInvokeCallback;
import com.instabug.library.invocation.util.InstabugVideoRecordingButtonPosition;
import com.instabug.library.model.FeaturesCache;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.model.Report;
import com.instabug.library.model.session.config.SessionsConfig;
import com.instabug.library.model.session.config.SessionsConfigMapper;
import com.instabug.library.percentagefeatures.PercentageFeature;
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils;
import com.instabug.library.util.InstabugAnnotations;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.ThemeApplier;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

/**
 * Man-in-the-middle Settings class to handle different settings usage based on their lifetime
 *
 * @author mSobhy
 * @see PersistableSettings
 * @see PerSessionSettings
 */
public class SettingsManager {
    /**
     * Enables verbose logging, with really annoying log statements
     * <br/>
     * <b>This should be disabled when releasing a build</b>
     */
    public static final boolean VERBOSE = false;
    private static final boolean SHOULD_LOG_EXTRA_REQUEST_DATA = false;

    public static final String INSTABUG_SHARED_PREF_NAME = "instabug";
    public static final String MIGRATION_STATE_SHARED_PREF = "instabug_migration_state";

    public static final int MAX_ASR_DURATION_IN_SECONDS = 180;
    public static final int MIN_ASR_DURATION_IN_SECONDS = 30;
    private static final String ASR_DURATION_ERROR_MSG_BELOW_MIN = "Auto screen recording duration should be " +
            "in a range from " + MIN_ASR_DURATION_IN_SECONDS + " to " + MAX_ASR_DURATION_IN_SECONDS + " seconds. Falling back to %d seconds";

    @Nullable
    private static SettingsManager settingsManager;

    /**
     * Returns the current singleton instance of this class.
     *
     * @return a {@code SettingsManager} instance
     */
    public synchronized static SettingsManager getInstance() {
        if (settingsManager == null) {
            settingsManager = new SettingsManager();
        }
        return settingsManager;
    }

    @Nullable
    public String getAppToken() {
        return PerSessionSettings.getInstance().getInstabugAppToken();
    }

    @Nullable
    public String getEarlyAppToken() {
        String builderToken = getAppToken();
        if (builderToken != null) {
            return builderToken;
        }
        Context localContext = Instabug.getApplicationContext();
        if (localContext != null) {
            return AndroidManifestSettings.getInstance(localContext).getAppToken();
        }
        return null;
    }

    public void setAppToken(String token) {
        PerSessionSettings.getInstance().setInstabugAppToken(token);
    }

    @Nullable
    public OnInvokeCallback getOnInvokeCallback() {
        return PerSessionSettings.getInstance().getOnInvokeCallback();
    }

    public void setOnInvokeCallback(OnInvokeCallback onInvokeCallback) {
        PerSessionSettings.getInstance().setOnInvokeCallback(onInvokeCallback);
    }

    @Nullable
    public OnSdkInvokedCallback getOnSdkInvokedCallback() {
        return PerSessionSettings.getInstance().getOnSdkInvokedCallback();
    }

    @Nullable
    public Report.OnReportCreatedListener getOnReportCreatedListener() {
        return PerSessionSettings.getInstance().getOnReportCreatedListener();
    }

    public void setOnReportCreatedListener(Report.OnReportCreatedListener listener) {
        PerSessionSettings.getInstance().setOnReportCreatedListener(listener);
    }

    /**
     * Set an on SDK invoked runnable. <br/>
     * WARNING: This runs on your application's main UI thread. Please do not include any
     * blocking operations to avoid ANRs.
     *
     * @param onSdkInvokedCallback This callback is run when SDK is invoked
     */
    public void setOnSdkInvokedCallback(OnSdkInvokedCallback onSdkInvokedCallback) {
        PerSessionSettings.getInstance().setOnSdkInvokedCallback(onSdkInvokedCallback);
    }


    public String getUserData() {
        if (InstabugFeaturesManager.getInstance().getFeatureState(IBGFeature.USER_DATA) == Feature
                .State.ENABLED && PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getUserData();
        } else {
            return "";
        }
    }


    public void setUserData(String userData) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setUserData(userData);
        }
    }

    @NonNull
    public Locale getInstabugLocale(@Nullable Context context) {
        return PerSessionSettings.getInstance().getInstabugLocale(context);
    }

    public void setInstabugLocale(Locale instabugLocale) {
        PerSessionSettings.getInstance().setInstabugLocale(instabugLocale);
    }

    /**
     * If you want to attach specific files to all reports, please specify its uri here
     *
     * @param attachmentUri         Uri to desired file
     * @param fileNameWithExtension the file name with extension
     */
    public void addExtraAttachmentFile(Uri attachmentUri, String fileNameWithExtension) {
        PerSessionSettings.getInstance().addExtraAttachmentFile(attachmentUri,
                fileNameWithExtension);
    }

    /**
     * If you want to attach specific data to all reports as a file.
     *
     * @param data                  byte[] of data wil be saved as a file
     * @param fileNameWithExtension the file name with extension
     */
    public void addExtraAttachmentFile(byte[] data, String fileNameWithExtension) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            if (AttachmentsUtility.isValidSize(data.length, AttachmentsUtility.MAX_FILE_SIZE_IN_MB)) {
                Uri uri = AttachmentsUtility.getUriFromBytes(context, data, fileNameWithExtension);
                if (uri != null) {
                    PerSessionSettings.getInstance().addExtraAttachmentFile(uri, fileNameWithExtension);
                }
            }
        } else
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't execute addExtraAttachmentFile(). Null context");
    }

    /**
     * Clears extra attachment files data <fileUri,fileNameWithExtension>.
     */
    public void clearExtraAttachmentFiles() {
        PerSessionSettings.getInstance().clearExtraAttachmentFiles();
        AttachmentsUtility.clearInternalAttachments(Instabug.getApplicationContext());
    }

    /**
     * Gets extra attachment files map contains <fileUri,fileNameWithExtension>.
     *
     * @return the extra attachment files
     */
    @Nullable
    public LinkedHashMap<Uri, String> getExtraAttachmentFiles() {
        return PerSessionSettings.getInstance().getExtraAttachmentFiles();
    }

    @NonNull
    public String getIdentifiedUserEmail() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getIdentifiedUserEmail();
        }
        return "";
    }

    public void setIdentifiedUserEmail(String identifiedUserEmail) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIdentifiedUserEmail(identifiedUserEmail);
        }
    }

    public boolean isDeviceRegistered() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isDeviceRegistered();
        }
        return false;
    }

    public void setIsDeviceRegistered(boolean registered) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsDeviceRegistered(registered);
        }
    }

    public boolean isFirstRun() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isFirstRun();
        }
        return true;
    }

    public void setIsFirstRun(boolean isFirstRun) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsFirstRun(isFirstRun);
        }
    }

    public Date getFirstRunAt() {
        if (PersistableSettings.getInstance() != null) {
            return new Date(PersistableSettings.getInstance().getFirstRunAt());
        }
        return new Date(0);
    }

    public void setFirstRunAt(long time) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setFirstRunAt(time);
        }
    }

    @Deprecated
    public long getLastContactedAt() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastContactedAt();
        }
        return 0L;
    }

    @Deprecated
    public void setLastContactedAt(long lastContactedAt) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setLastContactedAt(lastContactedAt);
        }
    }

    public boolean isAppOnForeground() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isAppOnForeground();
        }
        return true;
    }

    public void setIsAppOnForeground(boolean isAppOnForeground) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsAppOnForeground(isAppOnForeground);
        }
    }

    public int getLastMigrationVersion() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastMigrationVersion();
        }
        return 0;
    }

    public void setLastMigrationVersion(int lastMigrationVersion) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setLastMigrationVersion(lastMigrationVersion);
        }
    }

    public boolean isFirstDismiss() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isFirstDismiss();
        }
        return true;
    }

    public void setIsFirstDismiss(boolean firstDismiss) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsFirstDismiss(firstDismiss);
        }
    }

    public int getPrimaryColor() {
        return PerSessionSettings.getInstance().getPrimaryColor();
    }

    @SuppressLint("NULL_DEREFERENCE")
    public void setPrimaryColor(@ColorInt int primaryColorValue) {
        PerSessionSettings.getInstance().setPrimaryColor(primaryColorValue);
        InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
        if (contract != null) {
            contract.notifyPrimaryColorChanged();
        }
    }

    public InstabugColorTheme getTheme() {
        return PerSessionSettings.getInstance().getTheme();
    }

    public void setTheme(InstabugColorTheme instabugTheme) {
        PerSessionSettings.getInstance().setTheme(instabugTheme);
    }

    public void setIBGTheme(@Nullable IBGTheme theme) {
        if (theme != null) {
            PerSessionSettings.getInstance().setIBGTheme(theme);
            InvocationManagerContract contract = CoreServiceLocator.getInvocationManagerContract();
            if (contract != null && theme.getPrimaryColor() != ThemeApplier.DEFAULT_COLOR) {
                contract.notifyPrimaryColorChanged();
            }
        }
    }

    @Nullable
    public IBGTheme getIBGTheme() {
        return PerSessionSettings.getInstance().getIBGTheme();
    }

    @Nullable
    public Bitmap getLightCustomBrandingLogo() {
        return PerSessionSettings.getInstance().getLightCustomBrandingLogoBitmap();
    }

    public void setLightCustomBrandingLogo(Bitmap customLogoBitmap) {
        PerSessionSettings.getInstance().setLightCustomBrandingLogoBitmap(customLogoBitmap);
    }

    @Nullable
    public Bitmap getDarkCustomBrandingLogo() {
        return PerSessionSettings.getInstance().getDarkCustomBrandingLogoBitmap();
    }

    public void setDarkCustomBrandingLogo(Bitmap customLogoBitmap) {
        PerSessionSettings.getInstance().setDarkCustomBrandingLogoBitmap(customLogoBitmap);
    }

    public boolean isCustomBrandingEnabled() {
        return getLightCustomBrandingLogo() != null && getDarkCustomBrandingLogo() != null;
    }

    public String getIdentifiedUsername() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getIdentifiedUsername();
        }
        return "";
    }

    public void setIdentifiedUsername(@Nullable String username) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIdentifiedUsername(username);
        }
    }

    public void setEnteredUsername(String enteredUsername) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setEnteredUsername(enteredUsername);
        }
    }

    public String getEnteredUsername() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getEnteredUsername();
        }
        return "";
    }

    @Nullable
    public String getUuid() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getUuid();
        }
        return null;
    }

    public void setUuid(@Nullable String uuid) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setUuid(uuid);
        }
    }

    @Nullable
    public String getMD5Uuid() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getMD5Uuid();
        }
        return null;
    }

    public void setMD5Uuid(@Nullable String uuid) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setMD5Uuid(uuid);
        }
    }

    @Nullable
    public ArrayList<String> getTags() {
        return PerSessionSettings.getInstance().getTags();
    }

    public void addTags(String... tags) {
        PerSessionSettings.getInstance().addTags(tags);
    }

    public void resetTags() {
        PerSessionSettings.getInstance().resetTags();
    }

    public String getTagsAsString() {
        // Tags format: {"tags" : "tag1, tag2, tag3"}
        StringBuilder sb = new StringBuilder();
        ArrayList<String> tagsList = PerSessionSettings.getInstance().getTags();
        if (tagsList != null && tagsList.size() > 0) {
            int size = tagsList.size();
            for (int i = 0; i < size; i++) {
                sb.append(tagsList.get(i));
                if (i != size - 1)
                    sb.append(", ");
            }
        }
        return sb.toString();
    }

    public long getSessionStartedAt() {
        return PerSessionSettings.getInstance().getSessionStartedAt();
    }

    public void setSessionStartedAt(long sessionStartedAt) {
        PerSessionSettings.getInstance().setSessionStartedAt(sessionStartedAt);
    }

    @Nullable
    public InstabugCustomTextPlaceHolder getCustomPlaceHolders() {
        return PerSessionSettings.getInstance().getCustomPlaceHolders();
    }

    public void setCustomPlaceHolders(InstabugCustomTextPlaceHolder instabugCustomTextPlaceHolder) {
        PerSessionSettings.getInstance().setCustomPlaceHolders(instabugCustomTextPlaceHolder);
    }

    public boolean isUserLoggedOut() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isUserLoggedOut();
        }
        return true;
    }

    public void setUserLoggedOut(boolean isUserLoggedOut) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setUserLoggedOut(isUserLoggedOut);
        }
    }

    public boolean shouldMakeUUIDMigrationRequest() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().shouldMakeUUIDMigrationRequest();
        }
        return false;
    }

    public void setShouldMakeUUIDMigrationRequest(boolean shouldMigrate) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setShouldMakeUUIDMigrationRequest(shouldMigrate);
        }
    }

    public void setCurrentSDKVersion(String sdkVersion) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setCurrentSDKVersion(sdkVersion);
        }
    }

    public String getLastSDKVersion() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastSDKVersion();
        }
        return BuildConfig.SDK_VERSION;
    }

    public boolean isSDKVersionSet() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isSDKVersionSet();
        }
        return false;
    }

    public void setSavedAppToken(String appToken) {
        if (PersistableSettings.getInstance() == null) return;
        PersistableSettings.getInstance().setAppToken(appToken);
    }

    @Nullable
    public String getSavedAppToken() {
        if (PersistableSettings.getInstance() == null) return null;
        return PersistableSettings.getInstance().getAppToken();
    }

    public void setOSVersion(int osVersion) {
        if (PersistableSettings.getInstance() == null) return;
        PersistableSettings.getInstance().setOsVersion(osVersion);
    }

    public int getOSVersion() {
        if (PersistableSettings.getInstance() == null) return -1;
        return PersistableSettings.getInstance().getOsVersion();
    }

    public boolean isOSVersionSet() {
        return getOSVersion() != -1;
    }

    public int getStatusBarColor() {
        return PerSessionSettings.getInstance().getStatusBarColor();
    }

    public void setStatusBarColor(int statusBarColor) {
        PerSessionSettings.getInstance().setStatusBarColor(statusBarColor);
    }

    public void setRequestedOrientation(@InstabugAnnotations.ScreenOrientation int
                                                requestedOrientation) {
        PerSessionSettings.getInstance().setRequestedOrientation(requestedOrientation);
    }

    public int getRequestedOrientation() {
        return PerSessionSettings.getInstance().getRequestedOrientation();
    }

    public void resetRequestedOrientation() {
        PerSessionSettings.getInstance().resetRequestedOrientation();
    }

    public int getSessionsCount() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getSessionsCount();
        }
        return 0;
    }

    public void updateUserSessionCount(int sessionCount) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setSessionCount(sessionCount);
        }
    }

    public void resetSessionCount() {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().resetSessionCount();
        }
    }

    public void incrementSessionsCount() {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().incrementSessionsCount();
        }
    }

    public boolean isPromptOptionsScreenShown() {
        return PerSessionSettings.getInstance().isPromptOptionsScreenShown();
    }

    public void setPromptOptionsScreenShown(boolean promptOptionsShown) {
        PerSessionSettings.getInstance().setPromptOptionsScreenShown(promptOptionsShown);
    }

    public boolean isRequestPermissionScreenShown() {
        return PerSessionSettings.getInstance().isRequestPermissionScreenShown();
    }

    public void setRequestPermissionScreenShown(boolean isRequestPermissionScreenShown) {
        PerSessionSettings.getInstance().setRequestPermissionScreenShown(isRequestPermissionScreenShown);
    }

    @Nullable
    public OnSdkDismissCallback getOnSdkDismissCallback() {
        return PerSessionSettings.getInstance().getOnSdkDismissCallback();
    }

    public void setOnSdkDismissCallback(OnSdkDismissCallback onSdkDismissedCallback) {
        PerSessionSettings.getInstance().setOnSdkDismissCallback(onSdkDismissedCallback);
    }

    public void setAutoScreenRecordingEnabled(boolean autoScreenRecordingEnabled) {
        PerSessionSettings.getInstance().setAutoScreenRecordingEnabled(autoScreenRecordingEnabled);
    }

    public boolean isAutoScreenRecordingEnabled() {
        return PerSessionSettings.getInstance().isAutoScreenRecordingEnabled();
    }

    public int autoScreenRecordingMaxDuration() {
        return PerSessionSettings.getInstance().autoScreenRecordingMaxDuration();
    }

    public boolean isScreenCurrentlyRecorded() {
        return PerSessionSettings.getInstance().isScreenCurrentlyRecorded();
    }

    public void setScreenCurrentlyRecorded(boolean screenCurrentlyRecorded) {
        PerSessionSettings.getInstance().setScreenCurrentlyRecorded(screenCurrentlyRecorded);
    }

    public boolean isAutoScreenRecordingDenied() {
        return PerSessionSettings.getInstance().isAutoScreenRecordingDenied();
    }

    public void setAutoScreenRecordingDenied(boolean autoScreenRecordingDenied) {
        PerSessionSettings.getInstance().setAutoScreenRecordingDenied(autoScreenRecordingDenied);
    }

    public void setEnteredEmail(String enteredEmail) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setEnteredEmail(enteredEmail);
        }
    }

    @NonNull
    public String getEnteredEmail() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getEnteredEmail();
        }
        return "";
    }

    public void setScreenshotByMediaProjectionEnabled(boolean enabled) {
        PerSessionSettings.getInstance().setScreenshotByMediaProjectionEnabled(enabled);
    }

    public boolean isScreenshotByMediaProjectionEnabled() {
        return PerSessionSettings.getInstance().isScreenshotByMediaProjectionEnabled();
    }

    public void setIsSessionEnabled(boolean enabled) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsSessionEnabled(enabled);
        }
    }

    public boolean isSessionEnabled() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isSessionEnabled();
        }
        return true;
    }

    public void setOnboardingShowing(boolean onboardingShowing) {
        PerSessionSettings.getInstance().setOnboardingShowing(onboardingShowing);
    }

    public boolean isOnboardingShowing() {
        return PerSessionSettings.getInstance().isOnboardingShowing();
    }

    public void setShouldAutoShowOnboarding(boolean shouldShowOnboarding) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setShouldAutoShowOnboarding(shouldShowOnboarding);
        }
    }

    public boolean shouldAutoShowOnboarding() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().shouldAutoShowOnboarding();
        }
        return true;
    }

    public boolean isProcessingForeground() {
        return PerSessionSettings.getInstance().isProcessingForeground();
    }

    @Nullable
    public Locale getAppLocale() {
        return PerSessionSettings.getInstance().getAppLocale();
    }

    public void setAppLocale(Locale appLocale) {
        PerSessionSettings.getInstance().setAppLocale(appLocale);
    }

    /**
     * This is only a trigger that is used by the MediaProjection for screen-shot-capturing
     * to avoid flickering the invocation button.
     *
     * @param isProcessingForeground
     */
    public void setProcessingForeground(boolean isProcessingForeground) {
        PerSessionSettings.getInstance().setProcessingForeground(isProcessingForeground);
    }

    public void setAutoScreenRecordingAudioCapturingState(Feature.State asrAudioCapturingEnabled) {
        PerSessionSettings.getInstance().setAutoScreenRecordingAudioCapturingState(asrAudioCapturingEnabled);
    }

    public Feature.State getAutoScreenRecordingAudioCapturingState() {
        return PerSessionSettings.getInstance().getAutoScreenRecordingAudioCapturingState();
    }

    private String getASRError(int duration) {
        if (duration < MIN_ASR_DURATION_IN_SECONDS) {
            return String.format(Locale.ENGLISH, ASR_DURATION_ERROR_MSG_BELOW_MIN, MIN_ASR_DURATION_IN_SECONDS);
        } else {
            return String.format(Locale.ENGLISH, ASR_DURATION_ERROR_MSG_BELOW_MIN, MAX_ASR_DURATION_IN_SECONDS);
        }
    }

    private int convertSecondToMilliseconds(int seconds) {
        return (int) TimeUnit.MILLISECONDS.convert(seconds, TimeUnit.SECONDS);
    }

    public void setLastSeenTimestamp(long currentTimeMillis) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setLastSeenTimestamp(currentTimeMillis);
        }
    }

    public long getLastSeenTimestamp() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastSeenTimestamp();
        }
        return System.currentTimeMillis();
    }

    public void addPrivateViews(@NonNull View... views) {
        PerSessionSettings.getInstance().addPrivateViews(views);
    }

    public void removePrivateViews(@NonNull View... views) {
        PerSessionSettings.getInstance().removePrivateViews(views);
    }

    public void removeAllPrivateViews() {
        PerSessionSettings.getInstance().removeAllPrivateViews();
    }

    @NonNull
    public Collection<View> getPrivateViews() {
        return PerSessionSettings.getInstance().getPrivateViews();
    }

    // TODO: 6/17/19 this is a workaround util we refactor the setting manager
    public static void setInitialScreenShotAllowed(boolean initialScreenShotAllowed) {
        PerSessionSettings.getInstance().setInitialScreenShotAllowed(initialScreenShotAllowed);
    }

    public static boolean isInitialScreenShotAllowed() {
        return PerSessionSettings.getInstance().isInitialScreenShotAllowed();
    }

    public void release() {
        PerSessionSettings.release();
    }

    public long getFeaturesTTL() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getFeaturesTTL();
        }
        return 0;
    }

    public void setFeaturesTTL(long ttl) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setFeaturesTTL(ttl);
        }
    }

    public String getFeaturesHash() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getFeaturesHash();
        }
        return "";
    }

    public void setFeaturesHash(@Nullable String hash) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setFeaturesHash(hash);
        }
    }

    public void setLoggingFeatureSettings(@Nullable String loggingSettings) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setLoggingFeatureSettings(loggingSettings);
        }
    }

    @Nullable
    public String getLoggingFeatureSettings() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLoggingFeatureSettings();
        }
        return null;
    }

    @Nullable
    public FeaturesCache getFeaturesCache() throws JSONException {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getFeaturesCache();
        }
        return null;
    }

    public void setFeaturesCache(FeaturesCache cache) throws JSONException {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setFeaturesCache(cache);
        }
    }

    public static void setPushNotificationToken(String token) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setPushNotificationToken(token);
        }
    }

    public static String getPushNotificationToken() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getPushNotificationToken();
        }
        return "";
    }

    public static void setPushNotificationTokenSent(boolean isSent) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setPushNotificationTokenSent(isSent);
        }
    }

    public static boolean isPushNotificationTokenSent() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isPushNotificationTokenSent();
        }
        return false;
    }

    public boolean isFirstRunAfterEncryptorUpdate() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isFirstRunAfterEncryptorUpdate();
        }
        return true;
    }

    public void setFirstRunAfterEncryptorUpdate(boolean isFirstRun) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setFirstRunAfterEncryptorUpdate(isFirstRun);
        }
    }

    public void setSessionsSyncConfigurations(String json) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setSessionsSyncConfigurations(json);
        }
    }

    public SessionsConfig getSessionsSyncConfigurations() {
        if (PersistableSettings.getInstance() != null) {
            String json = PersistableSettings.getInstance().getSessionsSyncConfigurations();
            return SessionsConfigMapper.map(json);
        }
        return SessionsConfigMapper.map("{}");
    }

    public static SessionsConfig getSessionsSyncConfigurations(@NonNull Context context) {
        String json = PersistableSettings.getSessionsSyncConfigurations(context);
        return SessionsConfigMapper.map(json);
    }

    public boolean isFirstSession() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isFirstSession();
        }
        return true;
    }

    public void setIsFirstSession(boolean firstSession) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setIsFirstSession(firstSession);
        }
    }

    public void setVersionCode(int versionCode) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setVersionCode(versionCode);
        }
    }

    public int getLastKnownVersionCode() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastKnownVersionCode();
        }
        return -1;
    }

    public void setUsersPageEnabled(boolean enabled) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setUsersPageEnabled(enabled);
        }
    }

    public boolean isUsersPageEnabled() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().isUsersPageEnabled();
        }
        return false;
    }

    public void setCurrentPlatform(@Platform int platform) {
        PerSessionSettings.getInstance().setCurrentPlatform(platform);
    }

    public @Platform
    int getCurrentPlatform() {
        return PerSessionSettings.getInstance().getPlatform();
    }

    public @Platform
    int getEarlyCurrentPlatform(@Nullable Context context) {
        int runtimeCurrentPlatform = getCurrentPlatform();
        return context != null ? AndroidManifestSettings
                .getInstance(context)
                .getCurrentPlatform(runtimeCurrentPlatform)
                : runtimeCurrentPlatform;
    }

    public boolean isInBackground() {
        return PerSessionSettings.getInstance().isInBackground();
    }

    public void setInBackground(boolean inBackground) {
        PerSessionSettings.getInstance().setInBackground(inBackground);
    }

    public Feature.State getFeatureState(@IBGFeature.Companion.IBGFeature String feature, boolean defaultState) {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getBoolean(feature, defaultState) ? Feature.State.ENABLED : Feature.State.DISABLED;
        } else {
            return defaultState ? Feature.State.ENABLED : Feature.State.DISABLED;
        }
    }

    public boolean isFeatureEnabled(String feature, boolean defaultState) {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getBoolean(feature, defaultState);
        } else {
            return defaultState;
        }
    }

    public void setFeatureEnabled(@IBGFeature.Companion.IBGFeature String feature, boolean enabled) {
        if (PersistableSettings.getInstance() != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Saving feature: " + feature + " enabled state to " +
                    enabled);
            PersistableSettings.getInstance().saveBoolean(feature, enabled);
        }
    }

    public long getFirstSeen() {
        if (PersistableSettings.getInstance() == null) return -1;
        return PersistableSettings.getInstance().getFirstSeen();
    }

    public void setFirstSeen(long firstSeen) {
        if (PersistableSettings.getInstance() == null) return;
        PersistableSettings.getInstance().setFirstSeen(firstSeen);
    }

    @Nullable
    public String getLastAppVersion() {
        if (PersistableSettings.getInstance() == null) return null;
        return PersistableSettings.getInstance().getLastAppVersion();
    }

    public void setLastAppVersion(@Nullable String lastAppVersion) {
        if (PersistableSettings.getInstance() == null) return;
        PersistableSettings.getInstance().setLastAppVersion(lastAppVersion);
    }

    public void savePercentageFeature(@Nullable String feature, @Nullable PercentageFeature percentages) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().savePercentageFeature(feature, percentages);
        }
    }

    public PercentageFeature getPercentageFeature(String featureName) {
        PercentageFeature feature = new PercentageFeature();
        if (PersistableSettings.getInstance() != null && featureName != null) {
            feature = PersistableSettings.getInstance().getPercentageFeature(featureName);
        }

        return feature;
    }

    @Nullable
    public String getNonFatalsFeatureSettings() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getNonFatalsFeatureSettings();
        }
        return null;
    }

    public void setNonFatalsFeatureSettings(String nonFatalsFeatureSettings) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setNonFatalsFeatureSettings(nonFatalsFeatureSettings);
        }
    }

    public void setDiagnosticsSyncInterval(int syncInterval) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setDiagnosticsSyncInterval(syncInterval);
        }
    }

    public int getDiagnosticsSyncInterval() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getDiagnosticsSyncInterval();
        }

        return DiagnosticsCoreEventHandler.DEFAULT_SYNC_INTERVAL;
    }

    public long getDiagnosticsLastSyncTime() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getDiagnosticsLastSyncTime();
        }
        return System.currentTimeMillis();
    }

    public void setDiagnosticsLastSyncTime(long timeMillis) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setDiagnosticsLastSyncTime(timeMillis);
        }
    }

    public static boolean shouldLogExtraRequestData() {
        return SHOULD_LOG_EXTRA_REQUEST_DATA;
    }

    public long geLastForegroundTime() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getLastForegroundTime();
        }
        return -1;
    }

    public void setLastForegroundTime(long currentTimeMillis) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setLastForegroundTime(currentTimeMillis);
        }
    }

    public int getSessionStitchingTimeoutInSeconds(int defaultSessionTimeoutInSeconds) {
        if (PersistableSettings.getInstance() == null) return defaultSessionTimeoutInSeconds;
        return PersistableSettings.getInstance().getSessionStitchingTimeout(defaultSessionTimeoutInSeconds);
    }

    public void setSessionStitchingTimeout(int sessionStitchingTimeout) {
        if (PersistableSettings.getInstance() != null) {
            PersistableSettings.getInstance().setSessionStitchingTimeout(sessionStitchingTimeout);
        }
    }

    public boolean isCrashedSession() {
        if (PerSessionSettings.getInstance() != null) {
            return PerSessionSettings.getInstance().isCrashedSession();
        }
        return false;
    }

    public void setCrashedSession(boolean isCrashedSession) {
        if (PerSessionSettings.getInstance() != null) {
            PerSessionSettings.getInstance().setCrashedSession(isCrashedSession);
        }
    }

    public void setLogLevel(@LogLevel int logLevel) {
        PerSessionSettings.getInstance().setLogLevel(logLevel);
    }

    @LogLevel
    public int getLogLevel() {
        return PerSessionSettings.getInstance().getLogLevel();
    }

    public boolean shouldIgnoreFlagSecure() {
        return PerSessionSettings.getInstance().shouldIgnoreFlagSecure();
    }

    public void setIgnoreFlagSecure(boolean shouldIgnore) {
        PerSessionSettings.getInstance().setIgnoreFlagSecure(shouldIgnore);
    }

    public void setExperimentsStoreLimit(int storeLimit) {
        PersistableSettings persistableSettings = PersistableSettings.getInstance();
        if (persistableSettings != null) {
            persistableSettings.setExperimentsStoreLimit(storeLimit);
        }
    }

    public int getExperimentsStoreLimit() {
        PersistableSettings persistableSettings = PersistableSettings.getInstance();
        if (persistableSettings != null) {
            return persistableSettings.getExperimentsStoreLimit();
        }
        return EXPERIMENTS_STORE_LIMIT_FALLBACK;
    }

    /**
     * Only used for migrating old FeaturesManager configuration
     * to Fatal hang's configuration
     *
     * @return the saved FH sensitivity in key ib_fatal_hangs_sensitivity
     */
    public static long getFatalHangsSensitivity() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getFatalHangsSensitivity();
        }
        return 2000;
    }

    public void saveCustomTracesCount(int count) {
        if (PersistableSettings.getInstance() != null)
            PersistableSettings.getInstance().saveCustomTracesCount(count);
    }

    public int getCustomTracesCount() {
        if (PersistableSettings.getInstance() != null) {
            return PersistableSettings.getInstance().getCustomTracesCount();
        }

        return DEFAULT_MAX_COUNT;
    }

    @Nullable
    public ReproConfigurations getReproConfigurations() {
        PerSessionSettings instance = PerSessionSettings.getInstance();
        if (instance != null) {
            return instance.getReproConfigurations();
        }
        return null;
    }

    public void setReproConfigurations(ReproConfigurations reproConfigurations) {
        PerSessionSettings instance = PerSessionSettings.getInstance();
        if (instance != null) {
            instance.setReproConfigurations(reproConfigurations);
        }
    }

    public void setFullScreen(boolean isFullscreen) {
        PerSessionSettings.getInstance().setFullscreen(isFullscreen);
    }

    public boolean isFullscreen() {
        return PerSessionSettings.getInstance().isFullscreen();
    }

    public void setDevicePerformanceClass(int devicePerformanceClass) {
        if (PerSessionSettings.getInstance() != null)
            PerSessionSettings.getInstance().setDevicePerformanceClass(devicePerformanceClass);
    }

    public int getDevicePerformanceClass() {
        if (PerSessionSettings.getInstance() != null) {
            return PerSessionSettings.getInstance().getDevicePerformanceClass();
        }
        return DevicePerformanceClassUtils.PERFORMANCE_CLASS_UNDEFINED;
    }


    public InstabugVideoRecordingButtonPosition getVideoRecordingButtonPosition() {
        return PerSessionSettings.getInstance().getVideoRecordingButtonPosition();
    }

    public void setVideoRecordingButtonPosition(InstabugVideoRecordingButtonPosition
                                                        videoRecordingButtonPosition) {
        PerSessionSettings.getInstance().setVideoRecordingButtonPosition(videoRecordingButtonPosition);
    }

    @Nullable
    public InstabugInvocationEvent[] getInstabugInvocationEvents() {
        return PerSessionSettings.getInstance().getInstabugInvocationEvents();
    }
    public void setShouldCaptureDialog(boolean shouldCapture){
        PerSessionSettings.getInstance().setShouldCaptureDialog(shouldCapture);
    }
    public boolean getShouldCaptureDialog(){
       return PerSessionSettings.getInstance().getShouldCaptureDialog();
    }
}
