package com.instabug.library.sessionprofiler.model.timeline;

import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * Created by tarek on 3/4/18.
 */

public class SessionProfilerTimeline implements Serializable {

    private static final long serialVersionUID = 1105162041950151401L;

    private final static int DURATION_TO_TRIM = 60_000;
   public static final int TIME_INTERVAL_HIGH_FREQUENCY = 500;
    public static final int TIME_INTERVAL_LOW_FREQUENCY = 2000;

    private static final int COUNT_OF_HIGH_FREQUENCY_ELEMENTS_IN_MINUTE = DURATION_TO_TRIM /
            TIME_INTERVAL_HIGH_FREQUENCY;
    private static final int COUNT_OF_LOW_FREQUENCY_ELEMENTS_IN_MINUTE = DURATION_TO_TRIM /
            TIME_INTERVAL_LOW_FREQUENCY;

    private final static int VERSION_NUMBER = 1;
    private final static String KEY_PLATFORM = "platform";
    private final static String KEY_VERSION = "version";
    private final static String KEY_TIMELINE = "timeline";
    private final static String KEY_BATTERY = "battery";
    private final static String KEY_MEMORY = "memory";
    private final static String KEY_STORAGE = "storage";
    private final static String KEY_ORIENTATION = "orientation";
    private final static String KEY_CONNECTIVITY = "connectivity";


    private Queue<TimelinePoint> batteryStates;
    private Queue<TimelinePoint> connectivityStates;
    private Queue<TimelinePoint> orientationModes;
    private Queue<TimelinePoint> memoryUsages;
    private Queue<TimelinePoint> storageUsages;

    private final long totalStorage;

    public SessionProfilerTimeline() {
        batteryStates = new ConcurrentLinkedQueue<>();
        connectivityStates = new ConcurrentLinkedQueue<>();
        memoryUsages = new ConcurrentLinkedQueue<>();
        storageUsages = new ConcurrentLinkedQueue<>();
        orientationModes = new ConcurrentLinkedQueue<>();
        totalStorage = DeviceStateProvider.getTotalStorage();
    }

    public void addBatteryState(float value, boolean plugged) {
        batteryStates.add(new BatteryState(value, plugged));
    }

    public void addConnectivityState(ConnectivityState connectivityState) {
        connectivityStates.add(connectivityState);
    }

    public void addMemoryUsage(MemoryUsage usedMemory) {
        memoryUsages.add(usedMemory);
    }

    public void addStorageUsage(MemoryUsage usedStorage) {
        storageUsages.add(usedStorage);
    }

    public long getTotalStorage() {
        return totalStorage;
    }

    public void addScreenOrientation(ScreenOrientationMode screenOrientation) {
        orientationModes.add(screenOrientation);
    }

    Queue<TimelinePoint> getBatteryStates() {
        return batteryStates;
    }

    Queue<TimelinePoint> getConnectivityStates() {
        return connectivityStates;
    }

    Queue<TimelinePoint> getOrientationModes() {
        return orientationModes;
    }

    Queue<TimelinePoint> getMemoryUsages() {
        return memoryUsages;
    }

    Queue<TimelinePoint> getStorageUsages() {
        return storageUsages;
    }

    public SessionProfilerTimeline trim() {
        return trim(1.0f);
    }

    public SessionProfilerTimeline trim(float percentage) {
        try {
            trimQueue(batteryStates, Math.round(getCountOfLowFrequencyElements() * percentage));

            trimQueue(connectivityStates, Math.round(getCountOfLowFrequencyElements() * percentage));

            trimQueue(orientationModes, Math.round(getCountOfLowFrequencyElements() * percentage));

            trimQueue(memoryUsages, Math.round(getCountOfHighFrequencyElements() * percentage));

            trimQueue(storageUsages, Math.round(getCountOfHighFrequencyElements() * percentage));
        } catch (OutOfMemoryError outOfMemoryError) {
            InstabugCore.reportError(outOfMemoryError, "OOM while trimming session profiler timeline");
            InstabugSDKLogger.e(Constants.LOG_TAG, "OOM while trimming session profiler timeline", outOfMemoryError);
        }
        return this;
    }

    @VisibleForTesting
    public static void adjustTime(Collection<TimelinePoint> timelinePoints, float frequency) {
        ConcurrentLinkedQueue<TimelinePoint> timelinePointsAsQueue =
                timelinePoints instanceof ConcurrentLinkedQueue
                        ? (ConcurrentLinkedQueue<TimelinePoint>) timelinePoints
                        : new ConcurrentLinkedQueue<>(timelinePoints);

        int i = 0;
        for (TimelinePoint point : timelinePointsAsQueue) {
            long timeInMilliSeconds = (long) (i / frequency * getDurationToTrim());
            double timeInSeconds = Math.round(timeInMilliSeconds / 10.0) / 100.0;
            if (point != null) {
                point.setTime(timeInSeconds);
            }
            i++;
        }
    }

    @VisibleForTesting
    static Queue<TimelinePoint> trimQueue(Queue<TimelinePoint> src, int trimCount) {
        while (src.size() > trimCount) {
            // Drop most least inserted items that exceed the trim count limit.
            src.poll();
        }

        return src;
    }

    public JSONObject toJson() {

        //adjusting the timing of the session profiles timeline right before converting it to Json
        adjustTime(batteryStates,
                getCountOfLowFrequencyElements());
        adjustTime(connectivityStates,
                getCountOfLowFrequencyElements());
        adjustTime(orientationModes,
                getCountOfLowFrequencyElements());
        adjustTime(memoryUsages,
                getCountOfHighFrequencyElements());
        adjustTime(storageUsages,
                getCountOfHighFrequencyElements());

        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject
                    .put(KEY_VERSION, VERSION_NUMBER)
                    .put(KEY_PLATFORM, "Android")
                    .put(KEY_BATTERY, getTimelineJSONObject(batteryStates))
                    .put(KEY_ORIENTATION, getTimelineJSONObject(orientationModes))
                    .put(KEY_BATTERY, getTimelineJSONObject(batteryStates))
                    .put(KEY_CONNECTIVITY, getTimelineJSONObject(connectivityStates))
                    .put(KEY_MEMORY, getTimelineJSONObject(memoryUsages))
                    .put(KEY_STORAGE, getTimelineJSONObject(storageUsages).put(MemoryUsage
                            .KEY_TOTAL, getTotalStorage()));
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return jsonObject;
    }

    private JSONObject getTimelineJSONObject(Collection<TimelinePoint> timelinePoints) throws
            JSONException {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(KEY_TIMELINE, TimelinePoint.toJSONArray(timelinePoints));
        return jsonObject;

    }

    public static SessionProfilerTimeline fromJson(JSONObject jsonObject) {
        SessionProfilerTimeline sessionProfilerTimeline = new SessionProfilerTimeline();
        try {
            sessionProfilerTimeline.batteryStates = BatteryState.fromJSONArray(jsonObject
                    .getJSONObject(KEY_BATTERY).getJSONArray(KEY_TIMELINE));

            sessionProfilerTimeline.connectivityStates = ConnectivityState.fromJSONArray(jsonObject
                    .getJSONObject(KEY_CONNECTIVITY).getJSONArray(KEY_TIMELINE));

            sessionProfilerTimeline.orientationModes = ScreenOrientationMode.fromJSONArray
                    (jsonObject
                            .getJSONObject(KEY_ORIENTATION).getJSONArray(KEY_TIMELINE));

            sessionProfilerTimeline.memoryUsages = MemoryUsage.fromJSONArray(jsonObject
                    .getJSONObject(KEY_MEMORY).getJSONArray(KEY_TIMELINE));

            sessionProfilerTimeline.storageUsages = MemoryUsage.fromJSONArray(jsonObject
                    .getJSONObject(KEY_STORAGE).getJSONArray(KEY_TIMELINE));

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return sessionProfilerTimeline;
    }

    private static int getDurationToTrim() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(DURATION_TO_TRIM);
    }

    public static int getTimeIntervalHighFrequency() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(TIME_INTERVAL_HIGH_FREQUENCY);
    }

    public static int getTimeIntervalLowFrequency() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(TIME_INTERVAL_LOW_FREQUENCY);
    }

    static int getCountOfHighFrequencyElements() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(COUNT_OF_HIGH_FREQUENCY_ELEMENTS_IN_MINUTE);
    }

    static int getCountOfLowFrequencyElements() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(COUNT_OF_LOW_FREQUENCY_ELEMENTS_IN_MINUTE);
    }
}
