package com.instabug.library.sessionprofiler.model.timeline;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.util.DeviceStateProvider;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * Created by tarek on 3/4/18.
 */

public class ConnectivityState extends TimelinePoint {

    private final static String KEY_NAME = "name";

    private final static String KEY_WIFI = "WiFi";
    private final static String KEY_CELLULAR = "Cellular";
    private final static String KEY_NO_CONNECTION = "no_connection";

    @Nullable
    private String value;
    @Nullable
    private String name;

    @Nullable
    public String getValue() {
        return value;
    }

    public void setValue(@Nullable String value) {
        this.value = value;
    }

    @Nullable
    public String getName() {
        return name;
    }

    public void setName(@Nullable String name) {
        this.name = name;
    }

    public static ConnectivityState getNetworkState(@NonNull Context context) {
        ConnectivityState connectivityState = new ConnectivityState();

        if (context == null) {
            connectivityState.setValue(KEY_NO_CONNECTION);
            return connectivityState;
        }

        ConnectivityManager connectivityManager = getConnectivityManager(context);

        if (connectivityManager != null) {
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                NetworkCapabilities capabilities = getNetworkCapabilities(connectivityManager);
                return getNetworkState(capabilities, connectivityState);
            } else {
                return getNetworkState(context, connectivityManager, connectivityState);
            }
        } else {
            connectivityState.setValue(KEY_NO_CONNECTION);
        }
        return connectivityState;
    }

    private static ConnectivityState getNetworkState(@NonNull Context context,
                                                     @NonNull ConnectivityManager connectivityManager,
                                                     @NonNull ConnectivityState connectivityState) {
        NetworkInfo activeNetwork = connectivityManager.getActiveNetworkInfo();
        if (activeNetwork != null) { // connected to the internet
            if (activeNetwork.getType() == ConnectivityManager.TYPE_WIFI) {
                // connected to wifi
                connectivityState.setValue(KEY_WIFI);
            } else if (activeNetwork.getType() == ConnectivityManager.TYPE_MOBILE) {
                // connected to the mobile provider's data plan
                connectivityState.setName(DeviceStateProvider.getCarrier(context));
                connectivityState.setValue(activeNetwork.getSubtypeName());
            }
        } else {
            connectivityState.setValue(KEY_NO_CONNECTION);
        }
        return connectivityState;
    }

    @RequiresApi(29)
    @NonNull
    private static ConnectivityState getNetworkState(@Nullable NetworkCapabilities capabilities,
                                                     @NonNull ConnectivityState connectivityState) {
        if (capabilities == null) {
            connectivityState.setValue(KEY_NO_CONNECTION);
            return connectivityState;
        }

        if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
            connectivityState.setValue(KEY_CELLULAR);
        } else if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
            connectivityState.setValue(KEY_WIFI);
        } else {
            connectivityState.setValue(KEY_NO_CONNECTION);
        }
        return connectivityState;
    }

    @Nullable
    @RequiresApi(29)
    private static NetworkCapabilities getNetworkCapabilities(@Nullable ConnectivityManager connectivityManager) {
        if (connectivityManager == null) return null;
        return connectivityManager.getNetworkCapabilities(connectivityManager.getActiveNetwork());
    }

    @Nullable
    private static ConnectivityManager getConnectivityManager(@Nullable Context context) {
        if (context == null) return null;
        return (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
    }

    @Override
    protected JSONObject toJSONObject() throws JSONException {

        JSONObject jsonObject = getTimelinePointJSONObject(value);
        if (name != null) {
            jsonObject.put(KEY_NAME, name);
        }
        return jsonObject;
    }

    private static ConnectivityState fromJSONObject(JSONObject jsonObject) throws JSONException {
        ConnectivityState connectivityState = new ConnectivityState();
        connectivityState.setTime(jsonObject.getDouble(KEY_TIME));
        connectivityState.setValue(jsonObject.getString(KEY_VALUE));
        if (jsonObject.has(KEY_NAME)) {
            connectivityState.setName(jsonObject.getString(KEY_NAME));
        }
        return connectivityState;
    }

    static Queue<TimelinePoint> fromJSONArray(JSONArray jsonArray) throws
            JSONException {
        Queue<TimelinePoint> connectivityStates = new ConcurrentLinkedQueue<>();

        for (int i = 0; i < jsonArray.length(); i++) {
            connectivityStates.add(fromJSONObject(jsonArray.getJSONObject(i)));
        }
        return connectivityStates;
    }
}
