package com.instabug.library.sessionprofiler;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.core.eventbus.SessionStateEventBus;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.model.session.SessionState;
import com.instabug.library.sessionprofiler.model.timeline.ConnectivityState;
import com.instabug.library.sessionprofiler.model.timeline.MemoryUsage;
import com.instabug.library.sessionprofiler.model.timeline.ScreenOrientationMode;
import com.instabug.library.sessionprofiler.model.timeline.SessionProfilerTimeline;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import io.reactivexport.Observable;
import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;
import io.reactivexport.functions.Function;

/**
 * Created by tarek on 3/4/18.
 */

public class SessionProfiler {

    private static SessionProfiler INSTANCE;
    @Nullable
    private Runnable captureRunnable;

    private Boolean isFirstOOMSessionProfiler = false;

    private SessionProfilerTimeline timeline;
    @Nullable
    private Disposable capturingDisposable;
    private long time = 0;

    public synchronized static SessionProfiler getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new SessionProfiler();
        }

        return INSTANCE;

    }

    private SessionProfiler() {
        timeline = new SessionProfilerTimeline();
        subscribeToSessionEvents();
    }





    private void subscribeToSessionEvents() {
        SessionStateEventBus.getInstance()
                .subscribe(new Consumer<SessionState>() {
                    @Override
                    public void accept(SessionState sessionState) {
                        if (sessionState == SessionState.START) {
                            start();
                        } else if (sessionState == SessionState.FINISH) {
                            stop();
                        }
                    }
                });
    }

    public void start() {
        if (isFeatureEnabled()) {
            stop();
            capturingDisposable = Observable.interval(SessionProfilerTimeline.TIME_INTERVAL_HIGH_FREQUENCY, TimeUnit.MILLISECONDS)
                    .map(new Function<Long, Long>() {
                        @Override
                        public Long apply(Long iteration) {
                            // starts from 0
                            return (iteration + 1) * SessionProfilerTimeline.TIME_INTERVAL_HIGH_FREQUENCY;
                        }
                    }).subscribe(new Consumer<Long>() {
                        @Override
                        public void accept(Long time) {
                            capture(time);
                        }
                    }, new Consumer<Throwable>() {
                        @Override
                        public void accept(Throwable throwable) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while starting session profiler", throwable);
                        }
                    });
        }
    }

    public void stop() {
        if (capturingDisposable != null) capturingDisposable.dispose();
    }

    @NonNull
    private Observable<Long> getCaptureObservable(final long time) {

        return Observable.fromCallable(new Callable<Long>() {
            @Override
            public Long call() {
                capture(time);
                return time;
            }
        });
    }

    private boolean isFeatureEnabled() {
        return InstabugFeaturesManager.getInstance()
                .getFeatureState(IBGFeature.SESSION_PROFILER) == Feature.State.ENABLED;
    }

    private void capture(long time) {
        try {
            Context context = Instabug.getApplicationContext();

            if (time % SessionProfilerTimeline.TIME_INTERVAL_LOW_FREQUENCY == 0){

                //BatteryLevel
                if (context != null) {
                    int batteryLevel = DeviceStateProvider.getBatteryLevel(context);
                    String batteryState = DeviceStateProvider.getBatteryState(context);
                    timeline.addBatteryState(batteryLevel, !"Unplugged".equals(batteryState));
                } else
                    InstabugSDKLogger.e(Constants.LOG_TAG, "could attach battery state (Null app context)");

                //OrientationMode
                if (context != null) {
                    String screenOrientation = DeviceStateProvider.getScreenOrientation(context);
                    timeline.addScreenOrientation(new ScreenOrientationMode(screenOrientation));
                } else
                    InstabugSDKLogger.e(Constants.LOG_TAG, "could attach screen orientation (Null app context)");


                //ConnectivityState
                if (context != null) {
                    timeline.addConnectivityState(ConnectivityState.getNetworkState(context));
                } else
                    InstabugSDKLogger.e(Constants.LOG_TAG, "could attach network state (Null app context)");
            }

            //MemoryUsage
            if (context != null) {
                timeline.addMemoryUsage(new MemoryUsage(DeviceStateProvider.getUsedMemory(context),
                        DeviceStateProvider.getTotalMemory(context)));
            } else
                InstabugSDKLogger.e(Constants.LOG_TAG, "could attach used memory (Null app context)");

            //StorageUsage
            timeline.addStorageUsage(new MemoryUsage(DeviceStateProvider.getUsedStorage()));

            timeline.trim();
        } catch (OutOfMemoryError e) {
            if (!isFirstOOMSessionProfiler) {
                IBGDiagnostics.reportNonFatal(e, "Couldn't capture session profiler. Device is low on memory ");
                isFirstOOMSessionProfiler = true;
            }
          }
    }



    public SessionProfilerTimeline fetch() {
        return fetch(1.0f);
    }

    public SessionProfilerTimeline fetch(float percentage) {
        return timeline.trim(percentage);
    }

}
