package com.instabug.library.sessionV3.ratingDialogDetection

import com.instabug.library.model.v3Session.CustomStoreRattingMode.Companion.INVOKED_AND_REDIRECTED_TO_STORE
import com.instabug.library.model.v3Session.CustomStoreRattingMode.Companion.INVOKED_ONLY
import com.instabug.library.model.v3Session.RatingDialogDetection
import com.instabug.library.sessionV3.configurations.IBGSessionConfigurations
import com.instabug.library.sessionV3.configurations.RatingDialogDetectionConfigurations
import com.instabug.library.sessionV3.manager.IBGSessionManager
import com.instabug.library.sessionV3.providers.SessionAppDataProvider
import com.instabug.library.util.extenstions.logVerbose
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject
import java.util.concurrent.Executor


interface ManualRatingDetector {
    fun addCurrentModeToRateDetectionDataJson(
        timestamp: Long,
        ratingDialogDataJson: String? = null
    ): String?

    fun willRedirectToTheStoreAt(timestamp: Long)

    fun reset()
}

class ManualRatingDetectorImpl(
    private val appDataProvider: SessionAppDataProvider,
    private val rateDetectionConfigurations: RatingDialogDetectionConfigurations,
    private val sessionConfigurations: IBGSessionConfigurations,
    private val executor: Executor,
) : ManualRatingDetector {
    private var apiInvocationTimeStamp: Long? = null

    override fun addCurrentModeToRateDetectionDataJson(
        timestamp: Long, ratingDialogDataJson: String?
    ): String? = apiInvocationTimeStamp
        ?.takeIf { isFeatureEnabled() }
        ?.let { currentModeFromTimeStamp(timestamp) }
        ?.let { mode -> putCurrentModeInRatingDialogData(ratingDialogDataJson, mode) }
        ?: ratingDialogDataJson

    private fun putCurrentModeInRatingDialogData(ratingDialogDataJson: String?, mode: Int) =
        runOrLogError(errorMessage = "Something Went Wrong While adding Custom Store Rate Mode in Rating Dialog Data") {
            ratingDialogDataJson
                ?.let { JSONObject(it) }
                .orEmpty()
                .apply { put(RatingDialogDetection.CUSTOM_STORE_RATING_MODE_KEY, mode) }
                .toString()
        }.getOrElse { ratingDialogDataJson }

    private fun JSONObject?.orEmpty() = this ?: JSONObject()
    private fun currentModeFromTimeStamp(timeStamp: Long) =
        if (isAllowedIntervalPassed(timeStamp))
            INVOKED_ONLY
        else
            INVOKED_AND_REDIRECTED_TO_STORE

    private fun isAllowedIntervalPassed(timeStamp: Long) =
        timeStamp - (apiInvocationTimeStamp
            ?: 0) > rateDetectionConfigurations.allowedIntervalBeforeRedirection

    override fun willRedirectToTheStoreAt(timestamp: Long) = executor.execute {
        takeIf { isFeatureEnabled() }
            ?.takeIf { appDataProvider.isInstalledFromPlayStore }
            ?.takeIf { IBGSessionManager.currentSession != null }
            ?.let { apiInvocationTimeStamp = timestamp }
            ?.also { "Custom Ratting Dialog API is invoked at $timestamp".logVerbose() }
    }

    private fun isFeatureEnabled() =
        sessionConfigurations.isV3SessionEnabled &&
                rateDetectionConfigurations.isEnabled &&
                rateDetectionConfigurations.isCustomStoreRateApiEnabled

    override fun reset() = executor.execute {
        apiInvocationTimeStamp = null
    }
}