package com.instabug.library.sessionV3.manager

import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.InstabugState
import com.instabug.library.InstabugStateProvider
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher
import com.instabug.library.model.v3Session.IBGInMemorySession
import com.instabug.library.model.v3Session.IBGSession
import com.instabug.library.model.v3Session.IBGSessionExperiments
import com.instabug.library.model.v3Session.IBGSessionMapper.asForegroundStartEvent
import com.instabug.library.model.v3Session.IBGSessionMapper.toCoreSession
import com.instabug.library.model.v3Session.IBGSessionState
import com.instabug.library.model.v3Session.SessionEvent
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessioncontroller.SessionManualController
import com.instabug.library.sessionreplay.SRStateChangeListener
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.ScheduledFuture

object IBGSessionManager : SRStateChangeListener {

    @Volatile
    var currentSession: IBGInMemorySession? = null
        private set(newSession) {
            field = newSession
            if (newSession != null) latestSession = newSession
        }
    var latestSession: IBGInMemorySession? = null
        private set
        get() {
            if (currentSession == null && !configurations.isV3SessionEnabled) return null
            return field
        }

    @Volatile
    var latestSessionId: String? = null
        private set
        get() {
            if (currentSession == null && !configurations.isV3SessionEnabled) return null
            return field
        }

    private val cacheManger by lazy { IBGSessionServiceLocator.sessionCacheManger }
    private val experimentsCacheManager by lazy { IBGSessionServiceLocator.experimentsCacheManager }
    private val featuresFlagsCacheManager by lazy { IBGSessionServiceLocator.featuresFlagsCacheManager }
    private val backgroundStateProvider by lazy { IBGSessionServiceLocator.backgroundStateProvider }
    private val sessionExecutor by lazy { IBGSessionServiceLocator.sessionExecutor }
    private val stitchingManger by lazy { IBGSessionServiceLocator.stitchingManger }
    private val configurations by lazy { IBGSessionServiceLocator.sessionConfigurations }
    private val sessionStateManager by lazy { IBGSessionServiceLocator.stateManager }
    private val ratingDialogDetectionConfigs
        get() = IBGSessionServiceLocator.ratingDialogDetectionConfigs

    private val sharedPref
        get() = IBGSessionServiceLocator.durationSharedPreferences
    private val editor
        get() = sharedPref?.edit()

    @Volatile
    private var updatingDurationTask: ScheduledFuture<*>? = null

    init {
        IBGSessionServiceLocator.registerSrConfigurationChangedListener(this)
    }

    infix fun emit(sessionEvent: SessionEvent) = emit(sessionEvent, false)

    fun emit(sessionEvent: SessionEvent, handleEventOnSameThread: Boolean) {
        val eventRunnable = Runnable {
            runOrLogError(errorMessage = "Something went wrong while handling $sessionEvent") {
                logEvent(sessionEvent)
                handleSessionEvent(sessionEvent)
            }
        }
        if (handleEventOnSameThread) {
            eventRunnable.run()
        } else {
            sessionExecutor.execute(eventRunnable)
        }
    }

    private fun logEvent(sessionEvent: SessionEvent) =
        sessionEvent.run { "session $this event happen at $eventTimeMicro" }
            .let(::logD)

    private fun handleSessionEvent(sessionEvent: SessionEvent): Unit = synchronized(this) {
        when (sessionEvent) {
            is SessionEvent.Start -> startSession(sessionEvent)
            is SessionEvent.Stop -> stopSession(sessionEvent)
            is SessionEvent.End -> endSession(sessionEvent)
            is SessionEvent.RatingDialogDataReady -> storeRatingDialogData(sessionEvent)
        }
    }

    private fun stopSession(sessionEvent: SessionEvent.Stop) {
        logD("Instabug is disabled during app session, ending current session")
        endSession(sessionEvent)
    }

    private fun startSession(sessionEvent: SessionEvent.Start) {
        if (SessionManualController.isEnabled() && !sessionEvent.startedManually)
            return

        if (configurations.isV3SessionEnabled && InstabugStateProvider.getInstance().state != InstabugState.BUILDING) {
            updateNotEndedSessionDuration()
            currentSession
                ?.let { updateCurrentSession(sessionEvent) }
                ?: createNewSession(sessionEvent)
            startPeriodicDurationUpdate()
        }

    }

    private fun updateNotEndedSessionDuration() =
        runOrLogError("Something went wrong while updating not ended session duration") {
            if (currentSession == null)
                sharedPref?.all
                    ?.filter { it.value != null && it.value is Long }
                    ?.map { it.key to it.value as Long }
                    ?.forEach {
                        cacheManger.updateSessionDuration(it.first, it.second)
                    }
            editor?.clear()?.apply()
        }

    private fun endSession(sessionEvent: SessionEvent) {
        if (currentSession == null) return
        currentSession = null
        changeStateToFinished()
        stitchingManger.lastForegroundMicroTime = sessionEvent.eventTimeMicro
        cacheManger
            .queryLastSession()
            ?.update(sessionEvent)
            ?.let(cacheManger::insertOrUpdate)
            ?.also { sessionSerial ->
                IBGSessionServiceLocator.manualRatingDetector.reset()
                saveExperiments(sessionSerial)
                saveFeaturesFlags(sessionSerial)
                stopUpdatingDuration()
            }
            ?: logD("trying to end session while last session is null")
    }

    private fun changeStateToFinished() {
        sessionStateManager.handleStateChanged(IBGSessionState.Finished)
    }

    private fun saveExperiments(sessionSerial: Long) = sessionExecutor.execute {
        sessionSerial
            .let(IBGSessionExperiments::create)
            ?.let(experimentsCacheManager::insert) ?: logD(
            "experiments weren't saved as Experiments seems to be disabled for your Instabug " +
                    "company account. Please contact support for more information."
        )
    }

    private fun saveFeaturesFlags(sessionSerial: Long) = sessionExecutor.execute {
        featuresFlagsCacheManager.saveFeaturesFlags(sessionSerial)
    }

    private fun createNewSession(sessionEvent: SessionEvent.Start) =
        IBGInMemorySession.create(sessionEvent)
            .also { newSession -> currentSession = newSession }
            .also(::onHandleForegroundStart)
            .also { newSession -> latestSessionId = newSession.id }
            .let(IBGSession::create)
            .also(::changeStateToStarted)
            .also { session ->
                cacheManger.insertOrUpdate(session).let {
                    saveExperiments(it)
                    saveFeaturesFlags(it)
                }
            }

    private fun changeStateToStarted(session: IBGSession) {
        sessionStateManager.handleStateChanged(IBGSessionState.Started(session.toCoreSession()))
    }

    private fun onHandleForegroundStart(session: IBGInMemorySession) {
        if (session.startTime.isBackground) return
        IBGCoreEventPublisher.post(session.asForegroundStartEvent)
    }


    private fun updateCurrentSession(sessionEvent: SessionEvent) {
        val startMicroTime = sessionEvent.eventTimeMicro
        val appInForeground = !backgroundStateProvider.isAppStartedInBackground
        currentSession?.takeIf { it.startTime.isBackground && appInForeground }
            ?.run { copy(startTime = startTime.copy(foregroundMicroStartTime = startMicroTime)) }
            ?.also { updatedSession -> currentSession = updatedSession }
            ?.also(::onHandleForegroundStart)
            ?.let { cacheManger.queryLastSession()?.toForegroundSession(startTime = it.startTime) }
            ?.let(cacheManger::insertOrUpdate)
    }

    private fun startPeriodicDurationUpdate() {
        if (updatingDurationTask != null) return
        if (configurations.isV3SessionEnabled && configurations.periodicDurationCaptureEnabled) {
            updatingDurationTask = PoolProvider.postDelayedTaskAtFixedDelay(
                configurations.periodicDurationCaptureInterval,
                configurations.periodicDurationCaptureInterval,
            ) {
                sessionExecutor.execute {
                    updateCurrentSessionDuration()
                }
            }
        }
    }

    private fun stopUpdatingDuration() {
        runOrLogError(errorMessage = "Something went wrong while stopping session duration update") {
            updatingDurationTask?.cancel(true)
            updatingDurationTask = null
            editor?.remove(latestSessionId)?.apply()
        }
    }

    private fun updateCurrentSessionDuration() =
        runOrLogError(errorMessage = "Something went wrong while updating session duration") {
            currentSession
                ?.run {
                    startTime.value
                        .let { startMicroTime -> TimeUtils.currentTimeStampMicroSeconds() - startMicroTime }
                        .takeUnless { it < 0 }
                        ?.let { duration -> editor?.putLong(id, duration)?.apply() }
                }
        }

    private fun logD(message: String) = InstabugSDKLogger.d(LOG_TAG, message)

    private fun storeRatingDialogData(event: SessionEvent.RatingDialogDataReady) {
        if (configurations.isV3SessionEnabled && ratingDialogDetectionConfigs.isEnabled)
            cacheManger.updateRatingDialogDetection(event.ratingDialogData, latestSessionId)
    }

    /**
     * Sets srEnabled to true for the current running session if @param isEnabled is true
     * or to false for all the cached sessions if @param isEnabled is false
     */
    override fun onSRStateChanged(isEnabled: Boolean) {
        sessionExecutor.execute {
            takeIf { isEnabled }
                ?.cacheManger
                ?.queryLastSession()
                ?.let { session -> session.update(!session.startTime.isBackground) }
                ?.let { updatedSession -> cacheManger.insertOrUpdate(updatedSession) }
                ?: cacheManger.disableSessionsSR()
        }
    }
}
