package com.instabug.library.screenshot.instacapture

import androidx.annotation.IntDef
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.screenshot.instacapture.ViewsFilterTypes.Companion.INPUT
import com.instabug.library.screenshot.instacapture.ViewsFilterTypes.Companion.LABELS
import com.instabug.library.screenshot.instacapture.ViewsFilterTypes.Companion.MEDIA
import com.instabug.library.screenshot.instacapture.ViewsFilterTypes.Companion.PRIVATE

/**
 *
 *an annotation for internal mapping from user's MaskingTypes to
 * ViewsFilterTypes
 *to be used
 * */
@IntDef(value = [PRIVATE, LABELS, INPUT, MEDIA])
@Retention(AnnotationRetention.SOURCE)
annotation class ViewsFilterTypes {
    companion object {
        const val PRIVATE = 2
        const val INPUT = 3
        const val LABELS = 4
        const val MEDIA = 5
    }
}

fun interface ViewsFilter {
    infix fun shouldMask(node: IBGUINode?): Boolean

    companion object Factory {
        fun fromViewsFilterType(@ViewsFilterTypes type: Int): ViewsFilter? = when (type) {
            PRIVATE -> PrivateViewsFilter
            INPUT -> InputViewsFilter
            LABELS -> LabelsViewsFilter
            MEDIA -> MediaViewsFilter
            else -> null
        }
    }

}

internal object PrivateViewsFilter : ViewsFilter {
    override fun shouldMask(node: IBGUINode?): Boolean = node?.isPrivate ?: false
}

internal object LabelsViewsFilter : ViewsFilter {
    override fun shouldMask(node: IBGUINode?): Boolean =
        node?.type?.let { type -> (type and IBGUINode.Type.LABEL) != 0 } ?: false
}

internal object MediaViewsFilter : ViewsFilter {
    override fun shouldMask(node: IBGUINode?): Boolean =
        node?.type?.let { type -> (type and IBGUINode.Type.MEDIA) != 0 } ?: false
}

internal object InputViewsFilter : ViewsFilter {
    override fun shouldMask(node: IBGUINode?): Boolean =
        node?.type?.let { type -> (type and IBGUINode.Type.INPUT_FIELD) != 0 } ?: false
}

internal class ComposedViewsFilter(filters: Collection<ViewsFilter> = listOf()) : ViewsFilter {
    private val filters = filters.toMutableSet()

    override fun shouldMask(node: IBGUINode?): Boolean {
        return filters.any { filter -> filter.shouldMask(node) }
    }
}
