package com.instabug.library.logging.disklogs;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Instabug;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library.internal.resolver.LoggingSettingResolver;
import com.instabug.library.model.LogDescriptor;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.model.LoggingSettings.LogLevels;
import com.instabug.library.util.threading.PoolProvider;

import java.util.concurrent.Executor;

/**
 * A middle ware logger to control the logs based on its settings.
 */
public final class InstabugSDKDiskLogger {

    @Nullable
    private IBGLoggingThread ibgLoggingThread;
    private LoggingSettingResolver loggingSettingResolver;

    private Executor loggingExecutor = PoolProvider.getSingleThreadExecutor("LoggingExecutor");

    public InstabugSDKDiskLogger(Context context) {
        ibgLoggingThread = new IBGLoggingThread(context);
        loggingSettingResolver = LoggingSettingResolver.getInstance();
    }


    /**
     * Log base logs as long as log's level != {@link LogLevels#NO_LOGS}
     * in this level it will log
     * 1. debug
     * 2. warning
     * 3. errors
     * 4. info
     *
     * @param tag
     * @param message
     * @param currentThreadName
     */
    @SuppressLint("THREAD_SAFETY_VIOLATION")
    public void logBaseLevel(String tag, String message, String currentThreadName, long timestamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS
                            && ibgLoggingThread != null) {
                        ibgLoggingThread.addLog(tag, message, currentThreadName, timestamp);
                    }
                }
            }
        });
    }


    /**
     * Log base logs as long as log's level == {@link LogLevels#LEVEL_TWO}
     * in this level it will log
     * 1. debug
     * 2. warning
     * 3. errors
     * 4. info
     * 5. verbose
     *
     * @param tag
     * @param message
     * @param currentThreadName
     */
    public void logVerboseLogs(String tag, String message, String currentThreadName, long timestamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() == LogLevels.LEVEL_TWO
                            && ibgLoggingThread != null) {
                        ibgLoggingThread.addLog(tag, message, currentThreadName, timestamp);
                    }
                }
            }
        });
    }


    public void logSessionDetails(LogDescriptor logDescriptor) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS) {
                        if (ibgLoggingThread != null)
                            ibgLoggingThread.logSessionDescriptor(logDescriptor);
                    }
                }
            }
        });
    }

    @VisibleForTesting
    void setLoggingSettingResolver(LoggingSettingResolver loggingSettingResolver) {
        this.loggingSettingResolver = loggingSettingResolver;
    }

    @VisibleForTesting
    void setibgLoggingThread(IBGLoggingThread ibgLoggingThread) {
        this.ibgLoggingThread = ibgLoggingThread;
    }

    public void logEndSession(long timeStamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS) {
                        if (ibgLoggingThread != null)
                            ibgLoggingThread.logEndSession(timeStamp);
                    }
                }
            }
        });
    }

    public void onDiskLoggingLevelChanged(@LoggingSettings.LogLevels int level) {
        if (level == LoggingSettings.LogLevels.NO_LOGS) {
            if (ibgLoggingThread != null) {
                ibgLoggingThread.interrupt();
                ibgLoggingThread = null;
            }
        } else {
            if (ibgLoggingThread == null && Instabug.getApplicationContext() != null) {
                ibgLoggingThread = new IBGLoggingThread(Instabug.getApplicationContext());
            }
        }
    }


    @VisibleForTesting
    void setLoggingExecutor(Executor loggingExecutor) {
        this.loggingExecutor = loggingExecutor;
    }
}
