package com.instabug.library.internal.storage.cache;

import android.annotation.SuppressLint;

import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @author mSobhy
 */
public class InMemoryCache<K, V> extends Cache<K, V> {

    private final Map<K, V> map;

    public InMemoryCache(String mId) {
        this(mId, 1);
    }

    public InMemoryCache(String mId, int mAppVersion) {
        super(mId, mAppVersion);
        this.map = Collections.synchronizedMap(new LinkedHashMap<K, V>());
    }

    @Override
    @Nullable
    public V get(K key) {
        synchronized (map) {
            return map.get(key);
        }
    }

    @Override
    @Nullable
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    public V put(K key, V value) {
        if (value != null && key != null) {
            final V oldElement;
            synchronized (map) {
                oldElement = map.put(key, value);
            }
            if (oldElement == null) {
                notifyItemAdded(value);
                return value;
            } else {
                notifyItemUpdated(oldElement, value);
                return oldElement;
            }
        }
        return null;
    }

    @Override
    @Nullable
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    public V delete(K key) {
        final V removedElement;
        synchronized (map) {
            removedElement = map.remove(key);
        }
        if (removedElement != null)
            notifyItemRemoved(removedElement);
        return removedElement;
    }

    @Override
    public long size() {
        synchronized (map) {
            return map.size();
        }
    }

    @Override
    public void invalidate() {
        synchronized (map) {
            map.clear();
        }
        notifyCacheInvalidated();
    }

    @Override
    public List<V> getValues() {
        List<V> values = new ArrayList<>();
        synchronized (map) {
            for (K key : map.keySet()) {
                V value = get(key);
                if (value != null) {
                    values.add(value);
                }
            }
        }
        return values;
    }
}
