package com.instabug.library.experiments.cache;

import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.experiments.di.ServiceLocator;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class ExperimentsDBManagerImpl implements ExperimentsDBManager {
    private static final String ARGS_REPLACEMENT = "(?)";

    @Override
    public void insertExperiments(List<String> experiments) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        List<String> localExperiments = getExperiments(1f);
        List<String> validExperiments = filterExperiments(localExperiments, experiments);
        if (validExperiments.isEmpty()) return;
        try {
            String insertQuery = createInsertQuery(validExperiments);
            sqLiteDatabaseWrapper.execSQL(insertQuery, validExperiments.toArray());

        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatalAndLog(
                    ex, "Batch inserting experiments to DB failed: " + ex.getMessage(), Constants.LOG_TAG
            );
        }
    }

    private List<String> filterExperiments(@Nullable List<String> localExperiments, List<String> experiments) {
        List<String> validExperiments = new LinkedList<>();

        for (String experiment : experiments) {
            if (localExperiments == null || !localExperiments.contains(experiment)) {
                validExperiments.add(experiment);
            }
        }

        return validExperiments;
    }

    private String createInsertQuery(List<String> validExperiments) {
        StringBuilder queryBuilder = new StringBuilder();
        queryBuilder.append("INSERT INTO ")
                .append(InstabugDbContract.ExperimentsEntry.TABLE_NAME)
                .append(" (")
                .append(InstabugDbContract.ExperimentsEntry.COLUMN_EXPERIMENT)
                .append(") VALUES ");

        for (int i = 0; i < validExperiments.size(); i++) {
            queryBuilder.append(ARGS_REPLACEMENT);
            if (i < validExperiments.size() - 1) {
                queryBuilder.append(InstabugDbContract.COMMA_SEP);
            }
        }
        return queryBuilder.toString();
    }

    @Override
    public void deleteExperiments(@NonNull List<String> experiments) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        String whereClause = InstabugDbContract.ExperimentsEntry.COLUMN_EXPERIMENT +
                " = ?";
        for (String experiment : experiments) {
            sqLiteDatabaseWrapper.delete(InstabugDbContract.ExperimentsEntry.TABLE_NAME, whereClause, new String[]{experiment});
        }
        sqLiteDatabaseWrapper.close();
    }

    @Override
    public int trimToLimit(int limit) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();

        String whereClause = InstabugDbContract.ExperimentsEntry.COLUMN_ID + " not in ("
                + " select " + InstabugDbContract.ExperimentsEntry.COLUMN_ID
                + " from " + InstabugDbContract.ExperimentsEntry.TABLE_NAME
                + " order by " + InstabugDbContract.ExperimentsEntry.COLUMN_ID + " desc"
                + " limit ?" + " )";
        int deletedCount = sqLiteDatabaseWrapper.delete(InstabugDbContract.ExperimentsEntry.TABLE_NAME, whereClause,
                new String[]{String.valueOf(limit)});
        sqLiteDatabaseWrapper.close();
        return deletedCount;
    }

    @Override
    public void clearAllExperiments() {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        sqLiteDatabaseWrapper.delete(InstabugDbContract.ExperimentsEntry.TABLE_NAME, null, null);
        sqLiteDatabaseWrapper.close();
    }

    @Nullable
    @Override
    public List<String> getExperiments(float percentage) {
        DatabaseManager databaseManager = ServiceLocator.getDatabaseManager();
        SQLiteDatabaseWrapper sqLiteDatabaseWrapper = databaseManager.openDatabase();
        Cursor cursor = null;
        String limit = String.valueOf(Math.round(ServiceLocator.getExperimentsStoreLimit() * percentage));
        try {
            cursor = sqLiteDatabaseWrapper.query(InstabugDbContract.ExperimentsEntry.TABLE_NAME,
                    null,
                    null,
                    null,
                    null,
                    null,
                    null,
                    limit);

            if (cursor != null) {
                List<String> experiments = new ArrayList<>();
                if (cursor.moveToFirst()) {
                    do {
                        experiments.add(cursor.getString(cursor.getColumnIndexOrThrow(InstabugDbContract.ExperimentsEntry.COLUMN_EXPERIMENT)));
                    } while (cursor.moveToNext());
                }
                cursor.close();
                sqLiteDatabaseWrapper.close();
                return experiments;
            }
            sqLiteDatabaseWrapper.close();
        } catch (Exception | OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while getting experiments: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Error while getting experiments: " + e.getMessage());
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return null;
    }
}
