package com.instabug.library.datahub

import com.instabug.library.internal.filestore.DataAggregator
import com.instabug.library.internal.filestore.Directory
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.internal.filestore.NEWLINE_BYTE_VALUE
import com.instabug.library.util.extenstions.createNewFileDefensive
import com.instabug.library.util.extenstions.deleteDefensive
import com.instabug.library.util.extenstions.deleteRecursivelyDefensive
import com.instabug.library.util.extenstions.getOrLogAndReport
import com.instabug.library.util.extenstions.ifNotExists
import com.instabug.library.util.extenstions.logDWithThreadName
import com.instabug.library.util.extenstions.logVWithThreadName
import com.instabug.library.util.extenstions.mkdirDefensive
import com.instabug.library.util.extenstions.readJSONLines
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.extenstions.takeIfExists
import com.instabug.library.util.extenstions.takeUnlessExists
import java.io.File
import java.io.FileOutputStream

val DeleteOldestBatchFileOp = FileOperation<Directory, Unit> { directory ->
    runCatching {
        directory.listFiles()
            ?.sortedByDescending(File::lastModified)
            ?.lastOrNull()?.deleteDefensive()
    }.runOrLogAndReport("[File Op] Failed to deleted oldest batch file (Hub Op).")
}

class CreateNewBatchFile(private val batchName: String) : FileOperation<Directory, Unit> {
    override fun invoke(input: Directory) {
        runCatching {
            input.takeUnlessExists()?.mkdirDefensive()
            File(input, "$batchName.txt").ifNotExists { createNewFileDefensive() }
        }.runOrLogAndReport("[File Op] Failed to create new batch file (Hub Op).")
    }
}

class StoreOnBatchedDirectoryOp(
    private val log: DataHubLog,
    private val batcher: LogsBatcher
) : FileOperation<Directory, Unit> {
    override fun invoke(input: Directory) {
        runCatching {
            "[File Op] Storing on batched directory $input".logDWithThreadName()
            val batchingOps = batcher.getBatchingOps()
            input.takeIfExists()
                ?.also { "[File Op] Executing ${batchingOps.size} batching ops".logVWithThreadName() }
                ?.also { batchingOps.forEach { op -> op.invoke(it) } }
                ?.also { batcher.onBatchingOpsExecuted() }
                ?.let { WriteLogOnFileOp(log, "${batcher.currentBatchName}.txt").invoke(it) }
                ?.let { batcher.onLogStored() }
                ?: "[File Op] Input directory does not exists or writing op yielded null".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Failed to store on batched directory (Hub Op).")
    }
}

class ClearBatchedDirectoryOp(private val batcher: LogsBatcher?) : FileOperation<Directory, Unit> {
    override fun invoke(input: Directory) {
        runCatching {
            "[File Op] Clearing batched directory $input".logDWithThreadName()
            input.takeIfExists()
                ?.listFiles()
                ?.forEach(File::deleteDefensive)
                ?.let { batcher?.reset() }
                ?: "[File Op] Input directory does not exist".logDWithThreadName()
        }.runOrLogAndReport("[File Op] Failed to clear batched directory (Hub Op).")
    }
}

class DeleteBatchedDirectoryOp(private val batcher: LogsBatcher?) : FileOperation<Directory, Unit> {
    override fun invoke(input: Directory) {
        "[File Op] Deleting batched directory in $input".logDWithThreadName()
        runCatching {
            input.takeIfExists()
                ?.deleteRecursivelyDefensive()
                ?.let { batcher?.reset() }
                ?: "[File Op] Directory doesn't exists".logVWithThreadName()
        }.runOrLogAndReport("[File Op] Failed to delete batched directory (Hub Op).")
    }
}

class WriteLogOnFileOp(
    private val log: DataHubLog,
    private val fileName: String
) : FileOperation<Directory, Unit?> {
    override fun invoke(input: Directory): Unit? = runCatching {
        File(input, fileName)
            .also { file -> "[File Op] Writing hub log on file $file".logDWithThreadName() }
            .takeIfExists()
            ?.let { FileOutputStream(it, true) }
            ?.use { os -> os.writeDataHubLog(log) }
            ?: run { "[File Op] File to write on does not exist".logDWithThreadName(); null }
    }.getOrLogAndReport(null, "[File Op] Failed to write log on file (Hub Op).")
}

class ReadBatchedLogs<Out>(
    private val aggregator: DataAggregator<Out>
) : FileOperation<Directory, Out> {
    override fun invoke(input: Directory): Out = runCatching {
        "[File Op] Reading batched logs from directory $input".logDWithThreadName()
        input.takeIfExists()
            ?.listFiles()
            .also { files -> "[File Op] Found ${files?.size ?: 0} batch files".logVWithThreadName() }
            ?.apply { sortBy { file -> file.lastModified() } }
            ?.asSequence()
            ?.map { file -> file.bufferedReader() }
            ?.forEach { reader -> reader.readJSONLines(aggregator::add) }
            ?: "[File Op] Input director does not exist".logDWithThreadName()
        aggregator.aggregate()
    }.getOrLogAndReport(aggregator.aggregate(), "[File Op] Failed to read batched logs (Hub Op).")
}

private fun FileOutputStream.writeDataHubLog(log: DataHubLog) {
    log.dataHubRep?.toString()?.toByteArray()
        ?.let { logBytes -> write(logBytes); write(NEWLINE_BYTE_VALUE) }
}