package com.instabug.library

import androidx.annotation.IntDef
import androidx.annotation.Keep
import com.instabug.library.ReproConfigurations.Helper.defaultConfigurationsMap
import com.instabug.library.ReproConfigurations.Helper.updateModesMap

@Target(AnnotationTarget.PROPERTY, AnnotationTarget.VALUE_PARAMETER, AnnotationTarget.TYPE)
@Retention(AnnotationRetention.SOURCE)
@IntDef(
    IssueType.All, IssueType.Bug, IssueType.Crash, IssueType.SessionReplay, IssueType.AllCrashes,
    IssueType.Fatal, IssueType.NonFatal, IssueType.ANR, IssueType.AppHang, IssueType.ForceRestart
)
annotation class IssueType {
    companion object {
        const val Bug: Int = 0x01
        @Deprecated("Deprecated in favor of IssueType.AllCrashes that controls all types of crashes")
        const val Crash: Int = 0x02
        const val SessionReplay: Int = 0x04
        const val Fatal: Int = 0x08
        const val NonFatal: Int = 0x06
        const val ANR: Int = 0x10
        const val AppHang: Int = 0x20
        const val ForceRestart: Int = 0x40
        const val All: Int = Bug or Crash or SessionReplay
        const val AllCrashes = Fatal or NonFatal or ANR or AppHang or ForceRestart
    }
}

@Target(AnnotationTarget.VALUE_PARAMETER, AnnotationTarget.FUNCTION, AnnotationTarget.TYPE)
@Retention(AnnotationRetention.SOURCE)
@IntDef(ReproMode.EnableWithScreenshots, ReproMode.EnableWithNoScreenshots, ReproMode.Disable)
@Keep
annotation class ReproMode {
    companion object {
        /**
         * Disables both, steps & screenshots
         */
        const val Disable: Int = 0x00

        /**
         * Enables steps without screenshots
         */
        const val EnableWithNoScreenshots: Int = 0x01

        /**
         * Enables both, steps & screenshots
         */
        const val EnableWithScreenshots: Int = 0x02 or EnableWithNoScreenshots
    }
}

class ReproConfigurations private constructor(private val internalModesMap: MutableMap<Int, Int>) {

    /**
     * An immutable map with each [IssueType] mapped to its corresponding [ReproMode]
     */
    val modesMap: Map<Int, Int>
        get() = internalModesMap.toMap()

    /**
     * Amends a specific issue type mode in a non-overriding fashion.
     * @param type the [IssueType] that its mode to be amended.
     * @param mode the [ReproMode] to amend the [type] with.
     */
    fun amendIssueMode(@IssueType type: Int, @ReproMode mode: Int) {
        updateModesMap(internalModesMap, type, mode)
    }

    /**
     * A Builder class aiding [ReproConfigurations] object building.
     */
    class Builder {

        private val modesMap: MutableMap<Int, Int> = defaultConfigurationsMap.toMutableMap()

        /**
         * Sets a [ReproMode] for a specified [IssueType]
         * @param type the [IssueType] that its mode to be set.
         * @param mode the [ReproMode] to set for the [type].
         */
        fun setIssueMode(@IssueType type: Int, @ReproMode mode: Int): Builder {
            updateModesMap(modesMap, type, mode)
            return this
        }

        /**
         * Builds a [ReproConfigurations] object with aggregated set of issue types and modes specified by [setIssueMode].
         * @return a [ReproConfigurations] object.
         */
        fun build(): ReproConfigurations = ReproConfigurations(modesMap)
    }

    object Factory {
        /**
         * Returns a [ReproConfigurations] object with the default SDK mode (with no user intervention).
         * @return a [ReproConfigurations] mapped to the default state of repro features.
         */
        @JvmStatic
        fun newDefaultMode(): ReproConfigurations =
            ReproConfigurations(defaultConfigurationsMap.toMutableMap())
    }

    object Helper {
        private val availableIssueTypes: Set<Int> by lazy {
            setOf(
                IssueType.Bug,
                IssueType.SessionReplay,
                IssueType.Fatal,
                IssueType.NonFatal,
                IssueType.ANR,
                IssueType.AppHang,
                IssueType.ForceRestart
            )
        }

        private val availableCrashTypes: Set<Int> by lazy {
            setOf(
                IssueType.Fatal,
                IssueType.NonFatal,
                IssueType.ANR,
                IssueType.AppHang,
                IssueType.ForceRestart
            )
        }

        val defaultConfigurationsMap: Map<Int, Int> by lazy {
            availableIssueTypes
                .associateWith(this::getDefaultConfigurationsFor)
        }

        /**
         * Updates a modes map given a new [IssueType] and its [ReproMode]
         * @param modesMap The modes map to be updated.
         * @param type The [IssueType] whom state to be updated.
         * @param mode The [ReproMode] to update the given [type] with.
         */
        fun updateModesMap(
            modesMap: MutableMap<Int, Int>,
            @IssueType type: Int,
            @ReproMode mode: Int
        ) {
            issueTypeToSet(type).filter(modesMap::containsKey)
                .associateWith { mode }
                .also(modesMap::putAll)
        }

        private fun issueTypeToSet(@IssueType type: Int): Set<Int> = when (type) {
            IssueType.All -> availableIssueTypes
            IssueType.AllCrashes -> availableCrashTypes
            IssueType.Crash -> availableCrashTypes
            else -> setOf(type)
        }

        @ReproMode
        private fun getDefaultConfigurationsFor(@IssueType type: Int): Int = when (type) {
            IssueType.Bug, IssueType.SessionReplay -> ReproMode.EnableWithScreenshots
            IssueType.Fatal, IssueType.NonFatal, IssueType.ANR, IssueType.AppHang, IssueType.ForceRestart -> ReproMode.EnableWithNoScreenshots
            else -> ReproMode.Disable
        }
    }
}