package com.instabug.bug.cache;

import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_BUG_STATE;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_CATEGORIES_LIST;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_FRUSTRATING_EXPERIENCE_INTERNAL_ID;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_MESSAGE;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_STATE;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_TEMPORARY_SERVER_TOKEN;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.COLUMN_VIEW_HIERARCHY;
import static com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.BugEntry.TABLE_NAME;

import android.content.Context;
import android.database.Cursor;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.model.Bug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract;
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager;
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg;
import com.instabug.library.internal.storage.operation.ReadStateFromFileDiskOperation;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

public class BugReportsDbHelperImpl implements BugReportsDbHelper<IBGContentValues> {

    @Override
    public long insert(Bug bug) throws JSONException {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return NOT_FOUND;
        }
        if (bug.getId() == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't save the bug to DB because its ID is null");
            return NOT_FOUND;
        }
        try {
            // Create a new map of values, where column names are the keys
            IBGContentValues values = new IBGContentValues();
            values.put(BugTable.getCOLUMN_ID().component1(), bug.getId(), BugTable.getCOLUMN_ID().component2());
            values.put(BugTable.getCOLUMN_MESSAGE().component1(), bug.getMessage(), BugTable.getCOLUMN_MESSAGE().component2());
            values.put(BugTable.getCOLUMN_BUG_STATE().component1(), bug.getBugState().name(), BugTable.getCOLUMN_BUG_STATE().component2());
            if (bug.getTemporaryServerToken() != null) {
                values.put(BugTable.getCOLUMN_TEMPORARY_SERVER_TOKEN().component1(), bug.getTemporaryServerToken(), BugTable.getCOLUMN_TEMPORARY_SERVER_TOKEN().component2());
            }
            values.put(BugTable.getCOLUMN_TYPE().component1(), bug.getType(), BugTable.getCOLUMN_TYPE().component2());
            values.put(BugTable.getCOLUMN_CATEGORIES_LIST().component1(), bug.getCategoriesAsJSONArray().toString(), BugTable.getCOLUMN_CATEGORIES_LIST().component2());
            if (bug.getViewHierarchy() != null) {
                values.put(BugTable.getCOLUMN_VIEW_HIERARCHY().component1(), bug.getViewHierarchy(), BugTable.getCOLUMN_VIEW_HIERARCHY().component2());
            }
            if (bug.getState() != null && bug.getState().getUri() != null) {
                values.put(BugTable.getCOLUMN_STATE().component1(), bug.getState().getUri().toString(), BugTable.getCOLUMN_STATE().component2());
            }

            for (Attachment attachment : bug.getAttachments()) {
                long rowId = AttachmentsDbHelper.insert(attachment, bug.getId());
                if (rowId != NOT_FOUND) {
                    attachment.setId(rowId);
                }
            }

            if (bug.getConnectionError() != null) {
                values.put(BugTable.getCOLUMN_CONNECTION_ERROR().component1(), bug.getConnectionError(), BugTable.getCOLUMN_CONNECTION_ERROR().component2());
            }

            if (bug.getActionableConsents() != null) {
                values.put(BugTable.getCOLUMN_ACTIONABLE_CONSENT().component1(), bug.getActionableConsentsAsJsonArray().toString(), BugTable.getCOLUMN_ACTIONABLE_CONSENT().component2());
            }

            values.put(BugTable.getCOLUMN_FRUSTRATING_EXPERIENCE_INTERNAL_ID().component1(), bug.getFrustratingExperienceInternalId(), BugTable.getCOLUMN_FRUSTRATING_EXPERIENCE_INTERNAL_ID().component2());

            if (bug.getFrustratingExperienceExternalId() != null) {
                values.put(BugTable.getCOLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID().component1(), bug.getFrustratingExperienceExternalId(), BugTable.getCOLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID().component2());
            }
            // Insert the new row, returning the primary key value of the new row
            long rowId = databaseManager.insertWithOnConflictReplace(TABLE_NAME, null, values);
            databaseManager.setTransactionSuccessful();
            return rowId;
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while inserting bug", e);
            return NOT_FOUND;
        } finally {
            databaseManager.endTransaction();
        }
    }

    @Override
    @NonNull
    public List<Bug> retrieve(Context context) {
        return queryBugsTable(context, null, null);
    }

    @Override
    public List<Bug> retrieveProactiveReportingBugs(Context context) {
        List<IBGWhereArg> selectionArgs = new ArrayList<>();
        selectionArgs.add(new IBGWhereArg(Constants.ReportType.FRUSTRATING_EXPERIENCE, false));
        selectionArgs.add(new IBGWhereArg("null", false));

        return queryBugsTable(context, IBGDbContract.BugEntry.COLUMN_TYPE + " = ? and " + COLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID + " != ?", selectionArgs);
    }

    @Override
    public List<Bug> retrieveNormalBugs(Context context) {
        List<IBGWhereArg> selectionArgs = new ArrayList<>();
        selectionArgs.add(new IBGWhereArg(Constants.ReportType.FRUSTRATING_EXPERIENCE, false));

        return queryBugsTable(context, IBGDbContract.BugEntry.COLUMN_TYPE + " != ?", selectionArgs);

    }

    @NonNull
    private List<Bug> queryBugsTable(Context context, @Nullable String selection,
                                     @Nullable List<IBGWhereArg> selectionArgs) {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return new ArrayList<Bug>();
        }
        if (context != null) {
            List<Bug> bugs = new ArrayList<>();
            Cursor c = null;
            try {
                c = databaseManager.query(
                        TABLE_NAME,
                        null,
                        selection,
                        selectionArgs,
                        null,
                        null,
                        null,
                        null
                );

                if (c != null && c.moveToFirst()) {
                    do {
                        Bug bug = new Bug();
                        int idColIndex = c.getColumnIndex(COLUMN_ID);
                        bug.setId(c.getString(idColIndex));

                        int messageColIndex = c.getColumnIndex(COLUMN_MESSAGE);
                        bug.setMessage(c.getString(messageColIndex));

                        int bugStateColIndex = c.getColumnIndex(COLUMN_BUG_STATE);
                        bug.setBugState(Enum.valueOf(Bug.BugState.class, c.getString(bugStateColIndex)));

                        int tempServerTokenColIndex = c.getColumnIndex(COLUMN_TEMPORARY_SERVER_TOKEN);
                        bug.setTemporaryServerToken(c.getString(tempServerTokenColIndex));

                        int bugTypeColIndex = c.getColumnIndex(COLUMN_TYPE);
                        bug.setType(c.getString(bugTypeColIndex));

                        int bugCategoriesListColIndex = c.getColumnIndex(COLUMN_CATEGORIES_LIST);
                        bug.setCategoriesFromJSONArray(new JSONArray(c.getString(bugCategoriesListColIndex)));

                        int bugViewHierarchy = c.getColumnIndex(COLUMN_VIEW_HIERARCHY);
                        bug.setViewHierarchy(c.getString(bugViewHierarchy));

                        int bugConnectionError = c.getColumnIndex(IBGDbContract.BugEntry.COLUMN_CONNECTION_ERROR);
                        bug.setConnectionError(c.getString(bugConnectionError));

                        int bugActionableConsentColIndex = c.getColumnIndex(IBGDbContract.BugEntry.COLUMN_ACTIONABLE_CONSENT);
                        String actionableConsents = c.getString(bugActionableConsentColIndex);
                        if (actionableConsents != null) {
                            bug.setActionableConsentsFromJSONArray(new JSONArray(actionableConsents));
                        }

                        int bugFrustratingExperienceInternalIdIndex = c.getColumnIndex(IBGDbContract.BugEntry.COLUMN_FRUSTRATING_EXPERIENCE_INTERNAL_ID);
                        bug.setFrustratingExperienceInternalId(c.getLong(bugFrustratingExperienceInternalIdIndex));

                        int bugFrustratingExperienceExternalIdIndex = c.getColumnIndex(IBGDbContract.BugEntry.COLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID);
                        bug.setFrustratingExperienceExternalId(c.getString(bugFrustratingExperienceExternalIdIndex));


                        int stateColIndex = c.getColumnIndex(COLUMN_STATE);
                        State state = new State();
                        String URIString = c.getString(stateColIndex);
                        setState(context, bug, state, URIString);
                        if (bug.getId() != null) {
                            SQLiteDatabaseWrapper unEncryptedDB = DatabaseManager.getInstance().openDatabase();
                            ArrayList<Attachment> bugAttachments = AttachmentsDbHelper.retrieve(bug.getId(), unEncryptedDB);
                            bug.setAttachments(bugAttachments);
                            bugs.add(bug);
                        }
                    } while (c.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "retrieve bug reports failed: ", e);
                IBGDiagnostics.reportNonFatal(e, "retrieve bug reports failed: " + e.getMessage());
            } finally {
                if (c != null) {
                    c.close();
                }
            }
            return bugs;
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Context was null while retrieving bugs from DB.");
            return new ArrayList<>();
        }
    }

    private void setState(Context context, Bug bug, State state, String URIString) {
        if (URIString != null) {
            Uri uri = Uri.parse(URIString);
            state.setUri(uri);
            try {
                String stateJson = DiskUtils.with(context)
                        .readOperation(new ReadStateFromFileDiskOperation(uri))
                        .execute();
                state.fromJson(stateJson);
                bug.setState(state);
            } catch (OutOfMemoryError | Exception e) {
                InstabugCore.reportError(e, "retrieving bug state throws error");
                InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving bug state throws an exception", e);
            }
        }
    }

    @Override
    public void update(String id, IBGContentValues cv) {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return;
        }
        String whereClause = COLUMN_ID + "=? ";
        List<IBGWhereArg> whereArgs = new ArrayList<>();
        whereArgs.add(new IBGWhereArg(id, true));
        try {
            databaseManager.update(TABLE_NAME, cv, whereClause, whereArgs);
            databaseManager.setTransactionSuccessful();
        } finally {
            databaseManager.endTransaction();
        }
    }

    @Override
    public void updateConnectionError(@Nullable String id, String connectionError) {
        if (id != null) {
            IBGContentValues contentValues = new IBGContentValues();
            contentValues.put(BugTable.getCOLUMN_CONNECTION_ERROR().component1(), connectionError, BugTable.getCOLUMN_CONNECTION_ERROR().component2());
            update(id, contentValues);
        }
    }

    @Override
    public void delete(String id) {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return;
        }
        String whereClause = COLUMN_ID + "=? ";
        List<IBGWhereArg> whereArgs = new ArrayList<>();
        whereArgs.add(new IBGWhereArg(id, true));

        try {
            databaseManager.delete(TABLE_NAME, whereClause, whereArgs);
            databaseManager.setTransactionSuccessful();
        } finally {
            databaseManager.endTransaction();
        }
    }

    /**
     * Delete all bug reports stored in the database
     * i.e. deleting the whole table
     */
    @Override
    public void deleteAll() {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return;
        }
        databaseManager.beginTransaction();
        try {
            databaseManager.delete(TABLE_NAME, null, null);
            databaseManager.setTransactionSuccessful();
        } finally {
            databaseManager.endTransaction();
        }
    }

    @Override
    public void dropTable() {
        IBGDbManager databaseManager = IBGDbManager.getInstance();
        if (databaseManager == null) {
            return;
        }
        databaseManager.beginTransaction();
        try {
            databaseManager.execSQL(IBGDbContract.BugEntry.DROP_TABLE, null);
            databaseManager.setTransactionSuccessful();
        } finally {
            databaseManager.endTransaction();
        }
    }

    @Override
    public boolean updateFrustratingExperienceExternalId(Long internalId, String externalId) {
        IBGDbManager databaseManager = IBGDbManager.getInstance();

        if (databaseManager == null)
            return false;

        String whereClause = " " + COLUMN_FRUSTRATING_EXPERIENCE_INTERNAL_ID + " = ?  ";
        List<IBGWhereArg> whereArgs = new ArrayList<>();
        whereArgs.add(new IBGWhereArg(internalId.toString(), false));

        IBGContentValues contentValues = new IBGContentValues();
        contentValues.put(COLUMN_FRUSTRATING_EXPERIENCE_EXTERNAL_ID, externalId, false);

        try {
            boolean updated = databaseManager.update(TABLE_NAME, contentValues, whereClause, whereArgs) >= 1;
            databaseManager.setTransactionSuccessful();
            return updated;
        } finally {
            databaseManager.endTransaction();
        }
    }
}
