package com.instabug.bug.invocation.invoker

import android.Manifest.permission.DETECT_SCREEN_CAPTURE
import android.app.Activity
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.bug.utils.PermissionsUtils
import com.instabug.library.Constants
import com.instabug.library.invocation.InvocationManagerContract
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.threading.PoolProvider

interface ScreenshotCaptorRegistry {
    fun startSdkCaptureScreenShot(activity: Activity)
    fun stopSdkCaptureScreenShot(activity: Activity)
}

@RequiresApi(34)
class IBGRegistryScreenCaptureCallback(
    private val invocationManager: InvocationManagerContract?
) : Activity.ScreenCaptureCallback {
    override fun onScreenCaptured() {
        // No need to offload. The callback shall be triggered on the ioExecutor.
        invocationManager?.getInvocationRequested()
    }
}

class ScreenshotCaptorRegistryImp(
    private val ibgRegistryScreenCaptureCallback: IBGRegistryScreenCaptureCallback
) : ScreenshotCaptorRegistry {

    @RequiresApi(Build.VERSION_CODES.UPSIDE_DOWN_CAKE)
    override fun startSdkCaptureScreenShot(activity: Activity) {
        runOrLogError(Constants.LOG_TAG, "Error happened while registering ScreenCaptureCallback") {
            if (isPermissionGranted(activity)) {
                // Passing ioExecutor for the callback to be triggered on
                activity.registerScreenCaptureCallback(
                    PoolProvider.getInstance().ioExecutor,
                    ibgRegistryScreenCaptureCallback,
                )
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.UPSIDE_DOWN_CAKE)
    override fun stopSdkCaptureScreenShot(activity: Activity) {
        runOrLogError(
            Constants.LOG_TAG,
            "Error happened while unregistering ScreenCaptureCallback",
        ) {
            if (isPermissionGranted(activity)) {
                activity.unregisterScreenCaptureCallback(ibgRegistryScreenCaptureCallback)
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.UPSIDE_DOWN_CAKE)
    private fun isPermissionGranted(activity: Activity): Boolean {
        return (PermissionsUtils.isPermissionGranted(
            activity.applicationContext,
            DETECT_SCREEN_CAPTURE
        )).also {
            it.takeIf { !it }?.let {
                InstabugSDKLogger.w(
                    Constants.LOG_TAG,
                    "DETECT_SCREEN_CAPTURE permission is not granted",
                )
            }
        }
    }
}
