package com.instabug.bug;

import static com.instabug.bug.BugReporting.ReportType.BUG;
import static com.instabug.bug.BugReporting.ReportType.FEEDBACK;
import static com.instabug.bug.BugReporting.ReportType.QUESTION;
import static com.instabug.bug.Constants.COMMENT_MIN_COUNT_LESS_THAN_2_ERROR_MESSAGE;

import android.content.Context;
import android.os.Build;

import androidx.annotation.IntDef;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.extendedbugreport.ExtendedBugReport;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.bug.invocation.Option;
import com.instabug.bug.onboardingbugreporting.State;
import com.instabug.bug.onboardingbugreporting.utils.WelcomeMessageHelperApiContract;
import com.instabug.bug.onboardingbugreporting.utils.WelcomeMessageHelperApiImp;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.userConsent.ActionType;
import com.instabug.chat.ChatsDelegate;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.OnSdkDismissCallback;
import com.instabug.library.OnUsageExceededReady;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.video.MediaProjectionHelper;
import com.instabug.library.internal.video.customencoding.VideoEncoderConfig;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.OnInvokeCallback;
import com.instabug.library.invocation.util.InstabugFloatingButtonEdge;
import com.instabug.library.invocation.util.InstabugVideoRecordingButtonPosition;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * Created by mohamedzakaria on 6/5/17.
 */

public class BugReporting {

    /**
     * Changes the events used to invoke Instabug SDK
     *
     * @param invocationEvents to be used to invoke SDK
     * @see InstabugInvocationEvent
     * @since 8.0.0
     * Check if Instabug is still building then subscribe to SDK_STATE.BUILT event,
     * if it is already built (e.g. Called at runtime) then it will execute the code normally.
     */
    public static void setInvocationEvents(final InstabugInvocationEvent... invocationEvents) {
        APIChecker.checkAndRunInExecutor("BugReporting.setInvocationEvents", new VoidRunnable() {
            @Override
            public void run() {
                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;
                InvocationManager.getInstance().setInstabugInvocationEvent(invocationEvents);
            }
        });
    }

    /**
     * Set invocation move options like making comment field required in bug/feedback,
     * email field required ,optional or hidden, and disabling the success dialog that
     * appears after sending the bug/feedback.
     *
     * @param options array of options.
     */
    public static void setOptions(@NonNull @Option final int... options) {
        APIChecker.checkAndRunInExecutor("BugReporting.NonNull", new VoidRunnable() {
            @Override
            public void run() {
                BugReportingWrapper.setOptions(options);
            }
        });
    }

    /**
     * Sets the OnInvokeCallback that gets executed just before Instabug SDK invocation<br/>
     * WARNING: This runs on your application's main UI thread. Please do not include
     * any blocking operations to avoid ANRs.
     *
     * @param onInvokeCallback to run on the UI thread before Instabug SDK invocation
     * @since 8.0.0
     */
    public static void setOnInvokeCallback(final OnInvokeCallback onInvokeCallback) {
        APIChecker.checkAndRunInExecutorThenPostOnMain("BugReporting.setOnInvokeCallback", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Setting invoke callback");
                SettingsManager.getInstance().setOnInvokeCallback(onInvokeCallback);
            }
        });
    }

    /**
     * Sets the callback that gets executed when SDK is dismissed<br/>
     * WARNING: This runs on your application's main UI thread. Please do not include
     * any blocking operations to avoid ANRs.
     *
     * @param onSdkDismissedCallback to be called when Instabug SDK is dismissed
     * @since 8.0.0
     */
    public static void setOnDismissCallback(final OnSdkDismissCallback onSdkDismissedCallback) {
        APIChecker.checkAndRunInExecutorThenPostOnMain("BugReporting.setOnDismissCallback", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Setting OnSdkDismissCallback");
                ChatsDelegate.setOnSdkDismissCallback(onSdkDismissedCallback);
                BugSettings.getInstance().setOnSdkDismissCallback(onSdkDismissedCallback);
                SettingsManager.getInstance().setOnSdkDismissCallback(onSdkDismissedCallback);
            }
        });
    }


    /**
     * Sets the threshold value of the shake gesture on the device<br/>
     * Default is 350
     *
     * @param threshold threshold to use for shake detection
     * @since 8.0.0
     * Check if Instabug is still building then subscribe to SDK_STATE.BUILT event,
     * if it is already built (e.g. Called at runtime) then it will execute the code normally.
     */
    public static void setShakingThreshold(final int threshold) {
        APIChecker.checkAndRunInExecutor("BugReporting.setShakingThreshold", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Setting ShakingThreshold to: " + threshold);
                InvocationManager.getInstance().getCurrentInvocationSettings().setShakingThreshold(threshold);
            }
        });
    }

    /**
     * Sets the default edge at which the floating button will be shown
     *
     * @param instabugFloatingButtonEdge edge to stick Floating button to
     * @see InstabugFloatingButtonEdge
     * @since 8.0.0
     * Check if Instabug is still building then subscribe to SDK_STATE.BUILT event,
     * if it is already built (e.g. Called at runtime) then it will execute the code normally.
     */
    public static void setFloatingButtonEdge(@NonNull final InstabugFloatingButtonEdge instabugFloatingButtonEdge) {
        APIChecker.checkAndRunInExecutor("BugReporting.setFloatingButtonEdge", new VoidRunnable() {
            @Override
            public void run() {
                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;

                InstabugSDKLogger.d(Constants.LOG_TAG, "Setting FloatingButtonEdge to: " + instabugFloatingButtonEdge);
                InvocationManager.getInstance().getCurrentInvocationSettings().setFloatingButtonEdge
                        (instabugFloatingButtonEdge);
            }
        });
    }

    /**
     * Sets the default offset of the floating button from the top of the screen
     *
     * @param floatingButtonOffsetFromTop offset to use for floating button
     * @since 8.0.0
     * Check if Instabug is still building then subscribe to SDK_STATE.BUILT event,
     * if it is already built (e.g. Called at runtime) then it will execute the code normally.
     */
    public static void setFloatingButtonOffset(@IntRange(from = 0) final int floatingButtonOffsetFromTop) {
        APIChecker.checkAndRunInExecutor("BugReporting.setFloatingButtonOffset", new VoidRunnable() {
            @Override
            public void run() {
                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;

                InstabugSDKLogger.d(Constants.LOG_TAG, "Seetting FloatingButtonOffset: " + floatingButtonOffsetFromTop);
                InvocationManager.getInstance().getCurrentInvocationSettings()
                        .setFloatingButtonOffsetFromTop(floatingButtonOffsetFromTop);
            }
        });
    }

    /**
     * Sets the default corner at which the video recording floating button will be shown
     *
     * @param instabugViewRecordingButtonPosition corner to stick the video recording floating button to
     * @since 8.0.0
     */
    public static void setVideoRecordingFloatingButtonPosition(@NonNull final InstabugVideoRecordingButtonPosition instabugViewRecordingButtonPosition) {
        APIChecker.checkAndRunInExecutor("BugReporting.setVideoRecordingFloatingButtonPosition", new VoidRunnable() {
            @Override
            public void run() {
                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;

                InstabugSDKLogger.d(Constants.LOG_TAG, "setVideoRecordingFloatingButtonPosition: " +
                        instabugViewRecordingButtonPosition);
                InstabugCore.setVideoRecordingButtonPosition(instabugViewRecordingButtonPosition);
            }
        });
    }

    /**
     * Sets whether the extended bug report mode should be disabled, enabled with required fields
     * or enabled with optional fields.
     * <p>
     * This feature is disabled by default. When enabled, it adds more fields for
     * your reporters to fill in. You can set whether the extra fields are required or optional.
     * 1. Expected Results.
     * 2. Actual Results.
     * 3. Steps to Reproduce.
     *
     * @param state An enum to disable the extended bug report mode,
     *              enable it with required or with optional fields.
     * @since 5.0
     */
    public static void setExtendedBugReportState(@NonNull final com.instabug.library.extendedbugreport.ExtendedBugReport.State state) {
        APIChecker.checkAndRunInExecutor("BugReporting.setExtendedBugReportState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == null) {
                    InstabugSDKLogger.w(Constants.LOG_TAG, "state object passed to BugReporting.setExtendedBugReportState() is null");
                    return;
                }

                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;

                InstabugSDKLogger.d(Constants.LOG_TAG, "setExtendedBugReportState: " + state);

                ExtendedBugReport.State extendedBugReportState;
                switch (state) {

                    case ENABLED_WITH_REQUIRED_FIELDS:
                        extendedBugReportState = ExtendedBugReport.State.ENABLED_WITH_REQUIRED_FIELDS;
                        break;
                    case ENABLED_WITH_OPTIONAL_FIELDS:
                        extendedBugReportState = ExtendedBugReport.State.ENABLED_WITH_OPTIONAL_FIELDS;
                        break;

                    default:
                        extendedBugReportState = ExtendedBugReport.State.DISABLED;
                        break;
                }

                BugSettings.getInstance().setExtendedBugReportState(extendedBugReportState);
            }
        });
    }

    /**
     * Changes the extended bug report input fields default hints.
     * <p>
     * This method accepts a String value and use it as a replacement for the corresponding hint,
     * if the passed value is null then the corresponding extended bug report hint will not be affected.
     *
     * @param hint1 Is a replacement for the first extended bug report hint "Steps to reproduce".
     * @param hint2 Is a replacement for the second extended bug report hint "Actual results".
     * @param hint3 Is a replacement for the third extended bug report hint "Expected results".
     */
    public static void setExtendedBugReportHints(@Nullable String hint1, @Nullable String hint2, @Nullable String hint3) {
        APIChecker.checkAndRunInExecutor("BugReporting.setExtendedBugReportHints", () -> {
            InstabugSDKLogger.v(Constants.LOG_TAG, "setExtendedBugReportHints: Hint1 = "
                    + hint1 + ", Hint2 = " + hint2 + ", Hint3 = " + hint3);

            if (BugSettings.getInstance().isBugReportingStateEnabled()) {
                BugSettings.getInstance().setExtendedBugReportHints(hint1, hint2, hint3);
            }
        });
    }

    /**
     * Enable/Disable auto screen recording in bug reporting
     * You will find the recorded video attached to the bug report and you will be able to view or
     * remove it.
     *
     * @param autoScreenRecordingEnabled boolean for enable/disable auto screen recording
     * @since 8.1.1
     */
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    public static void setAutoScreenRecordingEnabled(final boolean autoScreenRecordingEnabled) {
        APIChecker.checkAndRunInExecutor("BugReporting.setAutoScreenRecordingEnabled", new VoidRunnable() {
            @Override
            public void run() {
                if (!BugSettings.getInstance().isBugReportingStateEnabled()) return;

                InstabugSDKLogger.d(Constants.LOG_TAG, "setAutoScreenRecordingEnabled: " + autoScreenRecordingEnabled);
                BugReportingWrapper.setAutoScreenRecordingEnabled(autoScreenRecordingEnabled);
            }
        });
    }

    // ----------------------------------- Remove --------------------------------------------//

    /**
     * Controls if Instabug Prompt Options should contain "Report a problem"
     * and/or "Suggest an improvement" or not.
     * By default, both options are enabled.
     *
     * @param types indicating the enabled bug report type(s).
     * @see Instabug.Builder#build()
     * <p>
     * Check if Instabug is still building then subscribe to SDK_STATE.BUILT event,
     * if it is already built (e.g. Called at runtime) then it will execute the code normally.
     */
    public static void setReportTypes(@BugReporting.ReportType final int... types) {
        APIChecker.checkAndRunInExecutor("BugReporting.setReportTypes", () ->
                PoolProvider.postIOTaskWithCheck(() -> BugReportingWrapper.setReportTypes(types)));
    }

    /**
     * Shows the compose view of a bug report or a feedback.
     *
     * @param type to specify whether to compose a new bug or feedback.
     * @see Instabug.Builder#build()
     * @see #show(int, int...)
     */
    public static void show(@BugReporting.ReportType final int type) {
        APIChecker.checkAndRunInExecutor("BugReporting.show", new VoidRunnable() {
            @Override
            public void run() {
                BugReportingWrapper.show(type);
            }
        });
    }

    /**
     * Shows the compose view of a bug report or a feedback.
     *
     * @param type    to specify whether to compose a new bug or feedback.
     * @param options array of options.
     * @see Instabug.Builder#build()
     * @see #show(int)
     * @see #setOptions(int...)
     */
    public static void show(@BugReporting.ReportType final int type, @Option final int... options) {
        APIChecker.checkAndRunInExecutor("BugReporting.show", new VoidRunnable() {
            @Override
            public void run() {
                BugReportingWrapper.show(type, options);
            }
        });
    }


    /**
     * Acts as a master switch for the Bug Reporting.
     * It's enabled by default. When disabled, both "Report a problem" and "Suggest an improvement" will be removed from Instabug Prompt Options.
     * In addition, when disabled {@link #show(int)} won’t have an effect.
     *
     * @param state possible states are ENABLED and DISABLED.
     * @see Instabug.Builder#build()
     */
    public static void setState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("BugReporting.setState", new VoidRunnable() {
            @Override
            public void run() {
                PoolProvider.postIOTaskWithCheck(new Runnable() {
                    @Override
                    public void run() {
                        if (state == null) {
                            InstabugSDKLogger.w(Constants.LOG_TAG, "state object passed to BugReporting.setState() is null");
                            return;
                        }
                        BugReportingWrapper.setState(state);
                    }
                });
            }
        });
    }

    /**
     * Enable/Disable view hierarchy
     *
     * @param state desired state of view hierarchy feature
     * @since 8.1.4
     */
    public static void setViewHierarchyState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("BugReporting.setViewHierarchyState", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setViewHierarchyState: " + state);
                InstabugCore.setFeatureState(IBGFeature.VIEW_HIERARCHY_V2, state);
            }
        });
    }

    public static void setDisclaimerText(final String disclaimerText) {
        APIChecker.checkAndRun("BugReporting.setDisclaimerText", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setDisclaimerText: " + disclaimerText);
                BugReportingWrapper.setDisclaimerText(disclaimerText);
            }
        });
    }

    // ----------------------------------- Remove --------------------------------------------//

    /**
     * Enabled/disable the attachment types for Bug
     *
     * @param extraScreenshot  whether extra screenshot is enable or not
     * @param imageFromGallery whether image from gallery is enable or not
     * @param screenRecording  whether screen recording is enable or not
     * @since 5.0
     */
    public static void setAttachmentTypesEnabled(final boolean initialScreenshot, final boolean extraScreenshot, final boolean imageFromGallery, final boolean screenRecording) {
        APIChecker.checkAndRunInExecutor("BugReporting.setAttachmentTypesEnabled", new VoidRunnable() {
            @Override
            public void run() {
                InstabugCore.setInitialScreenShotAllowed(initialScreenshot);
                BugReportingWrapper.setAttachmentTypesEnabled(initialScreenshot, extraScreenshot,
                        imageFromGallery, screenRecording);
            }
        });
    }

    @RequiresApi(21)
    public static void setScreenshotByMediaProjectionEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("BugReporting.setScreenshotByMediaProjectionEnabled", new VoidRunnable() {
            @Override
            public void run() {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setScreenshotByMediaProjectionEnabled: " + enabled);
                Context context = Instabug.getApplicationContext();
                if(context != null && enabled && !MediaProjectionHelper.INSTANCE.isMediaProjectionServiceAvailable(context)) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, Constants.SCREENSHOT_BY_MEDIA_PROJECTION_ENABLED_SERVICE_NOT_AVAILABLE);
                }
                SettingsManager.getInstance().setScreenshotByMediaProjectionEnabled(enabled);
            }
        });
    }

    /**
     * @param screenshotRequired Setting this to true would not let users delete the initial
     *                           screenshot taken with each report.
     * @since 8.0.4
     */
    protected static void setScreenshotRequired(final boolean screenshotRequired) {
        APIChecker.checkAndRunInExecutor("BugReporting.setScreenshotRequired", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                InstabugSDKLogger.d(Constants.LOG_TAG, "setScreenshotRequired: " + screenshotRequired);
                BugSettings.getInstance().setScreenshotRequired(screenshotRequired);
            }
        });
    }

    /**
     * @param limit       is Minimum Character Count that will be applied on the specified report type comment
     * @param reportTypes {@link IBGBugReportingType} is the report type that the limit will be applied on
     *                    you can add multiple report types if the API is called without report type
     *                    the limit will be applied on all report types
     */
    public static void setCommentMinimumCharacterCountForBugReportType(int limit, @IBGBugReportingType int... reportTypes) {
        APIChecker.checkAndRunInExecutor("BugReporting.setCommentMinimumCharacterCountForBugReportType", () -> {
            if (limit < 2) {
                InstabugSDKLogger.e(Constants.LOG_TAG, COMMENT_MIN_COUNT_LESS_THAN_2_ERROR_MESSAGE);
            } else {
                @IBGBugReportingType int[] types = reportTypes.length > 0 ? reportTypes : new int[]{IBGBugReportingType.BUG, IBGBugReportingType.FEEDBACK, IBGBugReportingType.QUESTION,IBGBugReportingType.FRUSTRATING_EXPERIENCE};
                BugReportingWrapper.setCommentMinimumCharacterCount(limit, types);
            }
        });
    }

    /**
     * @param onUsageExceededReady will be called with
     *                             true if Bug Reporting has exceeded the usage limit on your plan.
     *                             Otherwise, will be called with false
     */
    public static void getUsageExceeded(@NonNull OnUsageExceededReady onUsageExceededReady) {
        try {
            APIChecker.checkAndRunOrThrow("BugReporting.getUsageExceeded", () -> {
                final boolean isUsageExceeded = ServiceLocator
                        .getConfigurationsProvider()
                        .isBugReportingUsageExceeded();
                PoolProvider.postMainThreadTaskWithoutCheck(() -> {
                    if (onUsageExceededReady != null)
                        onUsageExceededReady.onReady(isUsageExceeded);
                });

            });

        } catch (Exception exception) {
            if (onUsageExceededReady != null) {
                onUsageExceededReady.onReady(false);
            }
        }
    }
    /**
     * Sets the welcome message mode to live, beta or disabled.
     * <p>
     * By default, the welcome message live mode is enabled. and it appears automatically after
     * 30 seconds from the user's first session. You can change it to the beta mode or disable it.
     * <p>
     * The live mode consists of one step to inform the users how to report a bug or feedback.
     * The beta mode consists of three steps to welcome your testers on board, inform them how to
     * report a bug or feedback and to motivate them to always be on the latest app version.
     * <p>
     * Please note, the into message appears only if the invocation event isn't set to none.
     *
     * @param welcomeMessageState An enum to set the welcome message state to live, beta or disabled.
     * @since 4.14.0
     */
    private static void setWelcomeMessageState(@NonNull final @State int welcomeMessageState) {
        APIChecker.checkAndRunInExecutor("Instabug.setWelcomeMessageState", () -> PoolProvider.postIOTaskWithCheck(() -> {
              BugSettings.getInstance().setWelcomeMessageState(welcomeMessageState);
            InstabugSDKLogger.v(com.instabug.library.Constants.LOG_TAG, "setWelcomeMessageState: " + welcomeMessageState);
        }));
    }

    /**
     * Shows the welcome message in a specific mode.
     * <p>
     * By default, the welcome message live mode is enabled. and it appears automatically after
     * 30 seconds from the user's first session. You can change it to the beta mode or disable it.
     * <p>
     * The live mode consists of one step to inform the users how to report a bug or feedback.
     * The beta mode consists of three steps to welcome your testers on board, inform them how to
     * report a bug or feedback and to motivate them to always be on the latest app version.
     * <p>
     * Please note, the into message appears only if the invocation event isn't set to none.
     *
     * @param welcomeMessageState An enum to set the welcome message state to live, beta or disabled.
     * @since 4.14.0
     */
    private static void showWelcomeMessage(@NonNull final @State int welcomeMessageState) {
        WelcomeMessageHelperApiContract welcomeMessageHelperApiContract = new WelcomeMessageHelperApiImp();
        APIChecker.checkAndRunInExecutor("Instabug.showWelcomeMessage", () -> {
            if (!InstabugCore.isForegroundBusy() && BugSettings.getInstance().isBugReportingStateEnabled()) {
                    welcomeMessageHelperApiContract.showWelcomeMessage(welcomeMessageState);

            }
        });
    }

    /**
     * @param  key must be unique and not null check the docs for maximum character limit
     *             calling the api many times with the same key and different parameters will override the previous values
     * @param  description description that will be displayed to the user
     *                     check the docs for maximum character limit
     *                     in case it's value is null or empty a localized generic content will be displayed to the user
     * @param isMandatory if the consent is mandatory then the bug report won't be submitted unless approved by the user
     * @param isChecked this reflects the default check status of the consent check box when adding a new bug
     * */
    public static void addUserConsent(
            @Nullable String key,
            @Nullable String description,
            boolean isMandatory,
            boolean isChecked
    ) {
        APIChecker.checkAndRunInExecutor("BugReporting.addUserConsent", () ->
                BugReportingWrapper.addUserConsent(key, description, isMandatory, isChecked, null)
        );
    }

    /**
     * @param  key must be unique and not null check the docs for maximum character limit
     *             calling the api many times with the same key and different parameters will override the previous values
     * @param  description description that will be displayed to the user
     *                     check the docs for maximum character limit
     *                     in case it's value is null or empty a localized generic content will be displayed to the user
     * @param isMandatory if the consent is mandatory then the bug report won't be submitted unless approved by the user
     * @param isChecked this reflects the default check status of the consent check box when adding a new bug
     * @param consentType reflects the type of data the consent is asking to collect
     * */
    public static void addUserConsent(
            @Nullable String key,
            @Nullable String description,
            boolean isMandatory,
            boolean isChecked,
            @Nullable @ActionType String consentType
    ) {
        APIChecker.checkAndRunInExecutor("BugReporting.addUserConsent", () ->
                BugReportingWrapper.addUserConsent(key, description, isMandatory, isChecked, consentType)
        );
    }

    /**
     * Sets a {@link ProactiveReportingConfigs} controlling proactive reporting settings
     *
     * @param configurations the configurations object to be set and used by Instabug.
     * @see ProactiveReportingConfigs.Builder
     */
    public static void setProactiveReportingConfigurations(ProactiveReportingConfigs configurations) {
        APIChecker.checkAndRunInExecutor("BugReporting.setProactiveReportingConfigurations", ()-> {
            BugReportingWrapper.setProactiveReportingConfigurations(configurations);
        });
    }

    /**
     * <b>Experimental</b>
     * Use to customize the video encoder configuration.
     *
     * @param config encoder configuration
     */
    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
    public static void setVideoEncoderConfig(@NonNull final VideoEncoderConfig config) {
        BugSettings.setVideoEncoderConfig(config);
    }

    @IntDef({BUG, FEEDBACK, QUESTION})
    @Retention(RetentionPolicy.RUNTIME)
    public @interface ReportType {
        int BUG = 0;
        int FEEDBACK = 1;
        int QUESTION = 2;
    }


    @IntDef({IBGBugReportingType.BUG, IBGBugReportingType.FEEDBACK, IBGBugReportingType.QUESTION, IBGBugReportingType.FRUSTRATING_EXPERIENCE})
    @Retention(RetentionPolicy.RUNTIME)
    public @interface IBGBugReportingType {
        int BUG = 0;
        int FEEDBACK = 1;
        int QUESTION = 2;
        int FRUSTRATING_EXPERIENCE = 3;
    }
}
