package com.instabug.chat.annotation.shape;

import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.graphics.Region;

import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;

import com.instabug.chat.annotation.ControlButton;
import com.instabug.chat.annotation.DirectionRectF;
import com.instabug.library.util.DrawingUtility;

/**
 * Created by tarek on 11/6/16.
 */

public class ArrowShape extends Shape {

    private final Paint paint;
    private static final float arrowLength = 60;
    private PointF tailPoint;
    private PointF headPoint;
    private float recognitionAngle;
    @Nullable
    private String type;

    public ArrowShape(PointF tailPoint, PointF headPoint, @ColorInt int color, float strokeWidth) {
        super(color, strokeWidth);
        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setColor(color);
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(strokeWidth);
        this.tailPoint = tailPoint;
        this.headPoint = headPoint;
    }

    public void setType(@Nullable String type) {
        this.type = type;
    }

    @Override
    public void draw(Canvas canvas, DirectionRectF bounds, DirectionRectF lastBounds) {

        if (bounds.firstX == DirectionRectF.X.RIGHT) {
            tailPoint.x = bounds.right;
        } else if (bounds.firstX == DirectionRectF.X.LEFT) {
            tailPoint.x = bounds.left;
        }

        if (bounds.firstY == DirectionRectF.Y.TOP) {
            tailPoint.y = bounds.top;
        } else if (bounds.firstY == DirectionRectF.Y.BOTTOM) {
            tailPoint.y = bounds.bottom;
        }

        if (bounds.secondX == DirectionRectF.X.RIGHT) {
            headPoint.x = bounds.right;
        } else if (bounds.secondX == DirectionRectF.X.LEFT) {
            headPoint.x = bounds.left;
        }

        if (bounds.secondY == DirectionRectF.Y.TOP) {
            headPoint.y = bounds.top;
        } else if (bounds.secondY == DirectionRectF.Y.BOTTOM) {
            headPoint.y = bounds.bottom;
        }

        canvas.drawPath(getPath(bounds), paint);
    }

    @Override
    public boolean isTouched(PointF touchedPoint, DirectionRectF bounds) {
        reBound(bounds);
        float rotation =
                DrawingUtility.getRotationDegrees(headPoint.x, headPoint.y, tailPoint.x, tailPoint.y);

        PointF p1 = DrawingUtility.pointOnCircle(arrowLength, rotation + 90, tailPoint);
        PointF p2 = DrawingUtility.pointOnCircle(arrowLength, rotation + 270, tailPoint);

        PointF p3 = DrawingUtility.pointOnCircle(arrowLength, rotation + 270, headPoint);
        PointF p4 = DrawingUtility.pointOnCircle(arrowLength, rotation + 90, headPoint);

        Region region = new Region();
        RectF rectF = new RectF();

        Path path = new Path();
        path.moveTo(p1.x, p1.y);
        path.lineTo(p2.x, p2.y);
        path.lineTo(p3.x, p3.y);
        path.lineTo(p4.x, p4.y);
        path.close();

        path.computeBounds(rectF, true);
        region.setPath(path,
                new Region((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom));
        return region.contains((int) touchedPoint.x, (int) touchedPoint.y);
    }

    @Override
    public void adjustBounds(DirectionRectF newBounds, DirectionRectF bounds, boolean isUndo) {
        bounds.set(newBounds);
    }

    public void adjustTailPoint(float x, float y, DirectionRectF bounds) {
        tailPoint.set(x, y);
        reBound(bounds);
    }

    public void adjustHeadPoint(float x, float y, DirectionRectF bounds) {
        headPoint.set(x, y);
        reBound(bounds);
    }

    private void reBound(DirectionRectF bounds) {
        if (tailPoint.x < headPoint.x) {
            bounds.left = tailPoint.x;
            bounds.right = headPoint.x;
            bounds.firstX = DirectionRectF.X.LEFT;
            bounds.secondX = DirectionRectF.X.RIGHT;
        } else {
            bounds.right = tailPoint.x;
            bounds.left = headPoint.x;
            bounds.firstX = DirectionRectF.X.RIGHT;
            bounds.secondX = DirectionRectF.X.LEFT;
        }

        if (tailPoint.y < headPoint.y) {
            bounds.top = tailPoint.y;
            bounds.bottom = headPoint.y;
            bounds.firstY = DirectionRectF.Y.TOP;
            bounds.secondY = DirectionRectF.Y.BOTTOM;
        } else {
            bounds.bottom = tailPoint.y;
            bounds.top = headPoint.y;
            bounds.firstY = DirectionRectF.Y.BOTTOM;
            bounds.secondY = DirectionRectF.Y.TOP;
        }
    }

    public DirectionRectF getBounds() {
        DirectionRectF bounds = new DirectionRectF();
        reBound(bounds);
        return bounds;
    }

    @Override
    public void drawControlButtons(Canvas canvas, DirectionRectF bounds,
                                   ControlButton[] controlButtons) {
        int color = paint.getColor();

        controlButtons[0].setCenterPoint(tailPoint);
        controlButtons[1].setCenterPoint(headPoint);

        for (int i = 0; i < 2; i++) {
            controlButtons[i].setColor(color);
            controlButtons[i].draw(canvas);
        }
    }

    @Override
    public void drawBorder(Canvas canvas, PointF topLeftPoint, PointF topRightPoint,
                           PointF bottomRightPoint, PointF bottomLeftPoint) {

    }

    @Override
    public void translateBy(DirectionRectF newBounds, DirectionRectF bounds, int dx, int dy) {
        newBounds.left = bounds.left + dx;
        newBounds.top = bounds.top + dy;
        newBounds.right = bounds.right + dx;
        newBounds.bottom = bounds.bottom + dy;
    }

    @Override
    public Path getPath(DirectionRectF bounds) {
        Path path = new Path();

        float rotation =
                DrawingUtility.getRotationDegrees(headPoint.x, headPoint.y, tailPoint.x, tailPoint.y);

        PointF headLeftPoint = DrawingUtility.pointOnCircle(arrowLength, rotation + 225, headPoint);

        PointF headRightPoint = DrawingUtility.pointOnCircle(arrowLength, rotation + 135, headPoint);

        path.moveTo(tailPoint.x, tailPoint.y);
        path.lineTo(headPoint.x + 1, headPoint.y + 1);

        if ("arrow".equals(type)) {
            path.moveTo(headLeftPoint.x, headLeftPoint.y);
            path.lineTo(headPoint.x, headPoint.y);
            path.lineTo(headRightPoint.x, headRightPoint.y);
        }

        return path;
    }

    public void updateTailAndHead(DirectionRectF rect) {
        float maxDimen = Math.max(rect.width(), rect.height());

        float tailX = rect.centerX() - maxDimen / 2;
        float headX = rect.centerX() + maxDimen / 2;

        PointF tailPoint = new PointF(tailX, rect.centerY());
        PointF headPoint = new PointF(headX, rect.centerY());

        this.tailPoint = DrawingUtility.rotatePoint(rect.centerX(), rect.centerY(), recognitionAngle, tailPoint);
        this.headPoint = DrawingUtility.rotatePoint(rect.centerX(), rect.centerY(), recognitionAngle, headPoint);
    }

    public void setRecognitionAngle(int angle) {
        this.recognitionAngle = angle;
    }
}
