package com.instabug.bug.invocation.invoker;

import android.content.ContentResolver;
import android.content.Context;
import android.database.ContentObserver;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.provider.MediaStore;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.bug.invocation.InvocationListener;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;

public class ScreenshotObserver extends ContentObserver {
    private final String[] PROJECTION = {
            MediaStore.Images.Media._ID,
            MediaStore.Images.Media.DISPLAY_NAME,
            MediaStore.Images.Media.DATA
    };
    private final String MEDIA_EXTERNAL_URI_STRING = MediaStore.Images.Media.EXTERNAL_CONTENT_URI.toString();
    private final InvocationListener invocationListener;

    private ContentResolver contentResolver;
    int screenshotState = 0x0;

    /**
     * Creates a content observer.
     *
     * @param handler            The handler to run {@link #onChange} on, or null if none.
     * @param contentResolver
     * @param invocationListener
     */
    public ScreenshotObserver(Handler handler,
                              ContentResolver contentResolver,
                              InvocationListener invocationListener) {
        super(handler);
        this.contentResolver = contentResolver;
        this.invocationListener = invocationListener;
    }

    @Override
    public void onChange(boolean selfChange, @Nullable Uri uri, int flags) {
        if (uri != null && uri.toString().matches(MEDIA_EXTERNAL_URI_STRING + "/[0-9]+")) {
            updateScreenshotState(flags);
            if (!isValidScreenshotState()) return;
            resetScreenshotState();
            try (Cursor cursor = contentResolver.query(uri, PROJECTION, null, null, null)) {
                if (cursor != null && cursor.moveToFirst()) {

                    final String fileName = cursor.getString(cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DISPLAY_NAME));
                    final String path = cursor.getString(cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA));

                    if (ScreenshotObserverHelper.isScreenshotDir(path) && ScreenshotObserverHelper.isScreenshotFile(fileName)) {
                        PoolProvider.postIOTask(new Runnable() {
                            @Override
                            public void run() {
                                Context applicationContext = Instabug.getApplicationContext();
                                File file = new File(path);
                                InstabugSDKLogger.d(Constants.LOG_TAG, "Creating new Uri for screenshot: " + file + " {" + file.getPath() + "}");
                                Uri newScreenshotUri = AttachmentsUtility
                                        .getNewFileAttachmentUri(applicationContext,
                                                Uri.fromFile(file));
                                if (invocationListener != null) {
                                    invocationListener.onInvocationRequested(newScreenshotUri);
                                }
                            }
                        });
                    }
                }
            }
        }
    }

    /**
     * Updates the screenshot state if the running SDK version is 30 and above.
     *
     * @param flags The new flags sent in the {@link #onChange}} callback.
     */
    private void updateScreenshotState(int flags) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.R) return;
        if (isUpdateNotification(flags) && !hasReceivedInsertionNotification()) {
            resetScreenshotState();
            return;
        }
        screenshotState |= flags;
    }

    /**
     * Checks whether the screenshot reached an acceptable state or not.
     *
     * @return true if the screenshot is in an acceptable state and false otherwise.
     */
    private boolean isValidScreenshotState() {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.R) return true;
        return (screenshotState & ContentResolver.NOTIFY_INSERT) != 0
                && (screenshotState & ContentResolver.NOTIFY_UPDATE) != 0;
    }

    private void resetScreenshotState() {
        screenshotState = 0x0;
    }

    @RequiresApi(api = Build.VERSION_CODES.R)
    private boolean isUpdateNotification(int flags) {
        return (flags & ContentResolver.NOTIFY_UPDATE) != 0;
    }

    @RequiresApi(api = Build.VERSION_CODES.R)
    private boolean hasReceivedInsertionNotification() {
        return (screenshotState & ContentResolver.NOTIFY_INSERT) != 0;
    }
}
