package com.instabug.bug.network;

import static com.instabug.bug.utils.DeleteBugsUtilKt.deleteAttachment;
import static com.instabug.library.networkv2.request.RequestExtKt.getTokenFromState;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.instabug.bug.Constants;
import com.instabug.bug.model.Bug;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.broadcast.LastContactedChangedBroadcast;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.featuresflags.EnhancementRequestBodyParams;
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator;
import com.instabug.library.frustratingexperience.FrustratingExperienceType;
import com.instabug.library.internal.storage.AttachmentsUtility;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;
import org.json.JSONException;
import org.json.JSONObject;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * @author mesbah
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class BugsService {
    @VisibleForTesting
    static final String VIEW_HIERARCHY = "view_hierarchy";
    private static final String TITLE = "title";
    private static final String CATEGORIES = "categories";
    private static final String ATTACHMENTS_COUNT = "attachments_count";
    private static final String FILE_TYPE = "metadata[file_type]";
    private static final String DURATION = "metadata[duration]";
    private static final String PARAM_REPORTED_AT = "reported_at";
    private static final String PARAM_CONN_DELAY_REPORTED_AT = "connection_delay_reported_at";
    @VisibleForTesting
    public static final String PARAM_FORCE_RESTART_TOKEN = "force_restart_token";
    @Nullable
    private static BugsService INSTANCE;
    @VisibleForTesting
    Request reportingBugRequest;
    private final NetworkManager networkManager;

    @VisibleForTesting
    private BugsService() {
        networkManager = NetworkManager.newInstance();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of BugsService
     */
    public static BugsService getInstance() {
        synchronized (BugsService.class.getName()) {
            if (INSTANCE == null) {
                INSTANCE = new BugsService();
            }
            return INSTANCE;
        }
    }

    public void reportBug(final Context context, final Bug bug, final Request.Callbacks<String,
            Throwable> reportingBugCallbacks) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Reporting bug request started");
        // create bug request
        reportingBugRequest = buildBugReportingRequest(bug);
        // do request with NORMAL request type.
        networkManager.doRequestOnSameThread(RequestType.NORMAL, reportingBugRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "ReportingBugRequest succeeded, Response code: " + requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "Response body: " + requestResponse.getResponseBody());
                try {
                    if (requestResponse.getResponseBody() != null) {
                        reportingBugCallbacks.onSucceeded(new JSONObject((String) requestResponse.getResponseBody()).getString("id"));
                    }
                } catch (JSONException e) {
                    InstabugCore.reportError(e, "Reporting bug got an error: " + e.getMessage());
                    InstabugSDKLogger.e(Constants.LOG_TAG, "reportingBugRequest got error: " + e.getMessage(), e);
                    reportingBugCallbacks.onFailed(e);
                }
                if (requestResponse.getResponseCode() == HttpURLConnection.HTTP_OK) {
                    // initialize current calendar.
                    Calendar calendar = Calendar.getInstance(Locale.ENGLISH);
                    InstabugSDKLogger.v(Constants.LOG_TAG, "Updating last_contacted_at to " + calendar.getTime());
                    BugSettings.getInstance().setLastBugTime(calendar.getTime().getTime());
                    InstabugCore.setLastContactedAt(calendar.getTime().getTime());
                    // send LastContactedChangedBroadcast.
                    Intent lastContactedAtChanged = new Intent();
                    lastContactedAtChanged.setAction(LastContactedChangedBroadcast.LAST_CONTACTED_CHANGED);
                    lastContactedAtChanged.putExtra(LastContactedChangedBroadcast.LAST_CONTACTED_AT, calendar.getTime().getTime());
                    LocalBroadcastManager.getInstance(context).sendBroadcast(lastContactedAtChanged);
                }
            }

            @Override
            public void onFailed(Throwable error) {
                if (error instanceof RateLimitedException) {
                    reportingBugCallbacks.onFailed(error);
                    return;
                }
                InstabugCore.reportError(error, "Reporting bug got an error: " + error.getMessage());
                InstabugSDKLogger.e(Constants.LOG_TAG, "reportingBugRequest got error: " + error.getMessage(), error);
                reportingBugCallbacks.onFailed(error);
            }

            @Override
            public void onDisconnected() {
                reportingBugCallbacks.onDisconnected();
            }

            @Override
            public void onRetrying(Throwable error) {
                reportingBugCallbacks.onRetrying(error);
            }
        });
    }


    @VisibleForTesting
    Request buildBugReportingRequest(Bug bug) {
        Request.Builder builder = new Request.Builder()
                .endpoint(Endpoints.REPORT_BUG)
                .method(RequestMethod.POST);

        getTokenFromState(builder, bug.getState());

        builder.addParameter(new RequestParameter(TITLE, bug.getMessage()));
        builder.addParameter(new RequestParameter(ATTACHMENTS_COUNT, bug.getAttachments().size()));
        builder.addParameter(new RequestParameter(CATEGORIES, bug.getCategoriesString()));

        if (bug.getConnectionError() != null) {
            builder.addParameter(new RequestParameter(PARAM_CONN_DELAY_REPORTED_AT, InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds()));
        }

        if(bug.getActionableConsentsAsJsonArray().length() > 0) {
            builder.addParameter(new RequestParameter(Bug.KEY_ACTIONABLE_CONSENT, bug.getActionableConsentsAsJsonArray()));
        }

        RequestParameter requestParameter = getFrustratingExperienceExternalIdRequestParameter(bug);
        if(requestParameter != null) {
            builder.addParameter(requestParameter);
        }



        // add state items to request as request body parameters
        reportingBugRequest = addStateItemsToRequestBodyParameters(builder, bug);
        return reportingBugRequest;
    }

    @Nullable
    private RequestParameter<String> getFrustratingExperienceExternalIdRequestParameter(Bug bug) {
        if(bug.getFrustratingExperienceExternalId() != null) {
            if(bug.getCategories().contains(FrustratingExperienceType.FORCE_RESTART)) {
                return new RequestParameter<>(PARAM_FORCE_RESTART_TOKEN, bug.getFrustratingExperienceExternalId());
            }
        }
        return null;
    }


    @VisibleForTesting
    Request addStateItemsToRequestBodyParameters(Request.Builder reportingBugRequestBuilder, Bug bug) {
        if (bug.getState() != null) {
            EnhancementRequestBodyParams enhancementRequestBodyParams = new EnhancementRequestBodyParams();
            Map<String, Object> result = enhancementRequestBodyParams.getModifiedStateItemsList(bug.getState().getStateItems(), FeaturesFlagServiceLocator.getFeaturesFlagsConfigsProvider().getMode());
            for (Map.Entry<String, Object> entry : result.entrySet()) {
                reportingBugRequestBuilder.addParameter(new RequestParameter(entry.getKey(), entry.getValue()));
            }
        }
        updateReportedAtIfNeeded(reportingBugRequestBuilder, bug);
        return reportingBugRequestBuilder.build();
    }


    private void updateReportedAtIfNeeded(Request.Builder builder, Bug bug) {
        State state = bug.getState();
        if (state != null && !state.isMinimalState() && state.getReportedAt() != 0) return;
        try {
            //Bug id is basically a UTC time in milliseconds. So,it can be used as the report time.
            long reportedAt = bug.getId() != null ? Long.parseLong(bug.getId()) : 0L;
            if (reportedAt != 0L) {
                builder.addParameter(new RequestParameter<>(PARAM_REPORTED_AT, reportedAt));
            }
        } catch (Exception e) {
            IBGDiagnostics.reportNonFatal(e, "Failed to update reported_at in bug reporting request.");
        }
    }

    public void uploadBugAttachments(final Bug bug, final Request.Callbacks<Boolean, Throwable> uploadBugAttachmentsCallbacks) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "Uploading Bug attachments");
        if (bug.getAttachments().isEmpty()) {
            uploadBugAttachmentsCallbacks.onSucceeded(true);
            return;
        }
        final List<Attachment> synced = new ArrayList<>();
        for (int i = 0; i < bug.getAttachments().size(); i++) {
            // create attachment request.
            final Attachment attachment = bug.getAttachments().get(i);
            // for decryption purpose
            boolean isAttachmentDecrypted = AttachmentsUtility.decryptAttachmentAndUpdateDb(attachment);
            if (attachment.getLocalPath() != null && attachment.getName() != null) {
                File file = new File(attachment.getLocalPath());
                if (isAttachmentDecrypted && file.exists() && file.length() > 0) {

                    Request.Builder bugAttachmentRequestBuilder = new Request.Builder()
                            .endpoint(Endpoints.ADD_BUG_ATTACHMENT)
                            .method(RequestMethod.POST)
                            .type(RequestType.MULTI_PART);

                    getTokenFromState(bugAttachmentRequestBuilder, bug.getState());

                    if (bug.getTemporaryServerToken() != null) {
                        bugAttachmentRequestBuilder.endpoint(
                                Endpoints.ADD_BUG_ATTACHMENT.replaceAll(":bug_token",
                                        bug.getTemporaryServerToken()));
                    }

                    if (attachment.getType() != null) {
                        bugAttachmentRequestBuilder.addParameter(new RequestParameter(FILE_TYPE, attachment.getType()));
                        if (attachment.getType() == Attachment.Type.AUDIO && attachment.getDuration() != null) {
                            bugAttachmentRequestBuilder.addParameter(new RequestParameter(DURATION, attachment.getDuration()));
                        }
                    }
                    attachment.setAttachmentState(Attachment.AttachmentState.SYNCED);
                    bugAttachmentRequestBuilder.fileToUpload(new FileToUpload("file",
                            attachment.getName(),
                            attachment.getLocalPath(),
                            attachment.getFileType()));
                    networkManager.doRequestOnSameThread(RequestType.MULTI_PART, bugAttachmentRequestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
                        @Override
                        public void onSucceeded(RequestResponse requestResponse) {
                            InstabugSDKLogger.d(Constants.LOG_TAG, "uploadingBugAttachmentRequest succeeded, Response code: " + requestResponse.getResponseCode());
                            InstabugSDKLogger.v(Constants.LOG_TAG, "uploadingBugAttachmentRequest succeeded, Response body: " + requestResponse.getResponseCode() + requestResponse.getResponseBody());
                            if (attachment.getLocalPath() != null) {
                                deleteAttachment(attachment, bug.getId());
                                synced.add(attachment);
                            }

                            if (synced.size() == bug.getAttachments().size()) {
                                uploadBugAttachmentsCallbacks.onSucceeded(true);
                            }
                        }

                        @Override
                        public void onFailed(Throwable error) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "uploadingBugAttachmentRequest got error: " + error.getMessage(), error);

                            AttachmentsUtility.encryptAttachmentAndUpdateDb(attachment);

                            uploadBugAttachmentsCallbacks.onFailed(error);

                            if (error instanceof IOException) {
                                //Stop sending the remaining attachments if a network error is occurred.
                                bug.getAttachments().clear();
                            }
                        }
                    });
                } else {
                    if (!isAttachmentDecrypted) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                                + attachment.getType() + " because it was not decrypted successfully");
                    } else if (!file.exists() || !(file.length() > 0)) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                                + attachment.getType() + " because it's either not found or empty file");
                    }
                }
            }
        }
    }

    public void uploadBugLogs(final Bug bug, final Request.Callbacks<Boolean, Throwable> callbacks) {
        // create logs request
        InstabugSDKLogger.v(Constants.LOG_TAG, "Uploading bug logs request started");
        Request bugLogsRequest;
        try {
            bugLogsRequest = buildBugLogsRequest(bug);
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "uploading bug logs got Json error ", e);
            callbacks.onFailed(e);
            return;
        }
        networkManager.doRequestOnSameThread(RequestType.NORMAL, bugLogsRequest, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse requestResponse) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "uploading bug logs succeeded, Response code: " +
                        requestResponse.getResponseCode());
                InstabugSDKLogger.v(Constants.LOG_TAG, "uploading bug logs onNext, Response body: " +
                        requestResponse.getResponseBody());
                callbacks.onSucceeded(true);
            }

            @Override
            public void onFailed(Throwable error) {
                InstabugCore.reportError(error, "uploading bug logs got error: " + error.getMessage());
                InstabugSDKLogger.e(Constants.LOG_TAG, "uploading bug logs got error", error);
                callbacks.onFailed(error);
            }
        });
    }

    @VisibleForTesting
    Request buildBugLogsRequest(Bug bug) {
        Request.Builder logsRequestBuilder = new Request.Builder()
                .endpoint(Endpoints.BUG_LOGS)
                .method(RequestMethod.POST);

        getTokenFromState(logsRequestBuilder, bug.getState());

        if (bug.getTemporaryServerToken() != null) {
            logsRequestBuilder.endpoint(Endpoints.BUG_LOGS.replaceAll(":bug_token", bug.getTemporaryServerToken()));
        }

        ArrayList<State.StateItem> logsItems = bug.getState() != null ? bug.getState().getLogsItems() : null;
        if (logsItems != null) {
            for (State.StateItem logItem : logsItems) {
                if (logItem.getKey() != null && logItem.getValue() != null) {
                    logsRequestBuilder.addParameter(new RequestParameter(logItem.getKey(), logItem.getValue()));
                }
            }
        }
        if (bug.getViewHierarchy() != null) {
            logsRequestBuilder.addParameter(new RequestParameter(VIEW_HIERARCHY, bug.getViewHierarchy()));
        }
        return logsRequestBuilder.build();
    }
    @VisibleForTesting
    public static void reset() {
        INSTANCE = null;
    }
}
