package com.instabug.bug.invocation.invoker;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.view.GestureDetector;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.widget.FrameLayout;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.internal.video.InternalScreenRecordHelper;
import com.instabug.bug.view.floatingactionbutton.MuteFloatingActionButton;
import com.instabug.bug.view.floatingactionbutton.RecordingFloatingActionButton;
import com.instabug.bug.view.floatingactionbutton.StopFloatingActionButton;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.R;
import com.instabug.library.core.eventbus.ActivityLifecycleSubscriber;
import com.instabug.library.core.eventbus.CurrentActivityConfigurationChange;
import com.instabug.library.core.eventbus.DefaultActivityLifeCycleEventHandler;
import com.instabug.library.internal.media.AudioPlayer;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.view.BubbleTextView;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.KeyboardEventListener;
import com.instabug.library.util.KeyboardUtils;
import com.instabug.library.util.MicUtils;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.ScreenUtility;
import com.instabug.library.util.StatusBarUtils;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.ref.WeakReference;

import io.reactivexport.disposables.CompositeDisposable;
import io.reactivexport.functions.Consumer;


public class ScreenRecordingFab implements View.OnClickListener, DefaultActivityLifeCycleEventHandler {

    private static final int MIN_TOP_LOCATION = 50;
    private static final int LANDSCAPE_MODE = 2;
    private static final int NAVIGATION_BAR_SIZE = 48;
    private static final int MAX_DURATION_LIMIT = 30000;
    @Nullable
    private FrameLayout.LayoutParams layoutParams;
    private final CompositeDisposable disposables = new CompositeDisposable();
    @VisibleForTesting
    @Nullable
    ActivityLifecycleSubscriber currentActivityLifeCycleSubscriber = null;
    private int x = 0;
    private int y = 0;
    private int width = 0;
    private int height = 0;
    private int realWidth = 0;
    private int orientation = 0;
    private float densityFactor;
    private int extraFloatingButtonSize;
    private boolean isRecording = false;
    private boolean expanded = false;
    private boolean isHintBubbleShown;
    private boolean isMicMuted = true;
    @Nullable
    private MuteFloatingActionButton muteButton;
    @Nullable
    private StopFloatingActionButton stopButton;
    @Nullable
    private BubbleTextView hintBubble;
    private int buttonSpacing;
    private int leftCorner;
    private int rightCorner;
    private int topCorner;
    private int bottomCorner;
    //Recording timer
    private final Handler handler = new Handler();
    private long startTime;
    @Nullable
    private FrameLayout floatingButtonFrameLayout;
    private int floatingButtonSize;
    @Nullable
    private DraggableRecordingFloatingActionButton draggableRecordingFloatingActionButton;
    private final ScreenRecordingFloatingBtnEventListener listener;
    @Nullable
    private KeyboardEventListener keyboardEventListener;
    @Nullable
    private WeakReference<Activity> mCurrentActivity;
    private boolean isKeyboardOpen = false;


    private int getDurationInSeconds() {
        final long start = startTime;
        long millis = System.currentTimeMillis() - start;
        return TimeUtils.millisToSeconds(millis);
    }

    private void announceDuration() {
        if (draggableRecordingFloatingActionButton == null) return;
        String durationText = getAnnouncedDurationText(getDurationInSeconds());
        AccessibilityUtils.sendTextEvent(durationText);
    }

    private String getAnnouncedDurationText(long seconds) {
        return draggableRecordingFloatingActionButton == null ?
                "" : draggableRecordingFloatingActionButton.getContext()
                .getResources()
                .getString(R.string.ibg_screen_recording_duration_for_accessibility, seconds);
    }

    private final Runnable updateTimeTask = new Runnable() {
        public void run() {
            if (isRecording) {
                final long start = startTime;
                long millis = System.currentTimeMillis() - start;
                if (draggableRecordingFloatingActionButton != null) {
                    String formattedDurationText = AudioPlayer.getFormattedDurationText(millis);
                    draggableRecordingFloatingActionButton.setText(formattedDurationText, true);
                    if (AccessibilityUtils.isTalkbackEnabled()) {
                        int seconds = TimeUtils.millisToSeconds(millis);
                        //announce the duration every 10 seconds
                        if (seconds != 0 && seconds % 10 == 0)
                            announceDuration();
                    }
                }
                if (millis > MAX_DURATION_LIMIT) {
                    listener.stop(getDurationInSeconds());
                }

                handler.postDelayed(this, 1000L);
            }
        }
    };

    public ScreenRecordingFab(ScreenRecordingFloatingBtnEventListener listener) {
        this.listener = listener;
    }

    public void init() {
        subscribeToCurranActivityLifeCycle();
        subscribeToCurrentActivityConfigChange();
    }

    private void subscribeToCurrentActivityConfigChange() {
        disposables.add(CurrentActivityConfigurationChange.getInstance().subscribe(new Consumer<CurrentActivityConfigurationChange>() {
            @Override
            public void accept(CurrentActivityConfigurationChange currentActivityConfigurationChange) throws Exception {
                if (currentActivityConfigurationChange.getNewConfig() != null)
                    handleConfigurationChange(currentActivityConfigurationChange.getNewConfig());
            }
        }));
    }

    public void release() {
//        InvocationManager.getInstance().switchOOnInvocation();
        handleKeyboardClosed();
        if (currentActivityLifeCycleSubscriber != null) {
            currentActivityLifeCycleSubscriber.unsubscribe();
        }
        disposables.clear();
        stop();
    }

    public void stop() {
        isRecording = false;
        isMicMuted = true;
        expanded = false;
        handler.removeCallbacks(updateTimeTask);
        hideFAB();
        muteButton = null;
        stopButton = null;
        hintBubble = null;
    }

    public void stopAndWait() {
        listener.stop(getDurationInSeconds());
    }


    private void showFAB(Activity activity, int width, int height) {
        if (floatingButtonFrameLayout != null) {
            floatingButtonFrameLayout.removeAllViews();
        }
        floatingButtonFrameLayout = new FrameLayout(activity);
        orientation = activity.getResources().getConfiguration().orientation;

        int tWidth, tHeight;
        densityFactor = activity.getResources().getDisplayMetrics().density;
        DisplayMetrics metrics = new DisplayMetrics();

        //adding the old screen dimensions to temp variables
        tWidth = width;
        tHeight = height;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            activity.getWindowManager().getDefaultDisplay().getRealMetrics(metrics);
            realWidth = metrics.widthPixels;
        }

        floatingButtonSize =
                (int) activity.getResources().getDimension(R.dimen.instabug_fab_size_normal);

        extraFloatingButtonSize =
                (int) activity.getResources().getDimension(R.dimen.instabug_fab_size_mini);

        buttonSpacing =
                (int) (activity.getResources().getDimension(R.dimen.instabug_fab_actions_spacing));

        int systemBottomInsets = ScreenUtility.getBottomInsets(activity);

        leftCorner = 0;
        rightCorner = width - (floatingButtonSize + buttonSpacing);
        topCorner = StatusBarUtils.getStatusBarHeight(activity);
        bottomCorner = height - (floatingButtonSize + buttonSpacing + systemBottomInsets);

        hintBubble = new BubbleTextView(activity);
        hintBubble.setText(PlaceHolderUtils.getPlaceHolder(
                Instabug.getApplicationContext(),
                InstabugCustomTextPlaceHolder.Key.VIDEO_RECORDING_FAB_BUBBLE_HINT,
                R.string.instabug_str_video_recording_hint
        ));

        muteButton = new MuteFloatingActionButton(activity);

        if (!MicUtils.isAudioPermissionGranted()) {
            if (muteButton.getVisibility() == View.VISIBLE) {
                muteButton.setVisibility(View.GONE);
            }
        }

        if (isMicMuted) {
            muteButton.disable();
        } else {
            muteButton.enable();
        }

        muteButton.setOnClickListener(view -> {
            Context context = Instabug.getApplicationContext();
            if (context != null) {
                if (muteButton != null && muteButton.toggle()) {
                    MicUtils.unmuteMic(context);
                    isMicMuted = false;
                } else {
                    MicUtils.muteMic(context);
                    isMicMuted = true;
                }
            }
        });
        stopButton = new StopFloatingActionButton(activity);
        disposables.add(InternalScreenRecordHelper.getInstance()
                .getIsStoppableObservable()
                .subscribe(canStop -> {
                    if (stopButton != null) {
                        stopButton.setEnabled(canStop);
                    }
                }));
        if (stopButton != null) {
            stopButton.setOnClickListener(view -> {
                //stop the timer
                if (isRecording) {
                    KeyboardUtils.hide(activity);
                    if (listener != null) {
                        listener.stop(getDurationInSeconds());
                    }
                    isRecording = false;
                    handler.removeCallbacks(updateTimeTask);
                }
            });
        }
        draggableRecordingFloatingActionButton = new DraggableRecordingFloatingActionButton(activity);

        if (layoutParams == null) {
            layoutParams = new FrameLayout.LayoutParams(
                    floatingButtonSize, floatingButtonSize, Gravity.TOP | Gravity.LEFT
            );
            draggableRecordingFloatingActionButton.setLayoutParams(layoutParams);

            switch (SettingsManager.getInstance().getVideoRecordingButtonPosition()) {
                case BOTTOM_LEFT:
                    draggableRecordingFloatingActionButton.setLocation(leftCorner, bottomCorner);
                    break;
                case TOP_LEFT:
                    draggableRecordingFloatingActionButton.setLocation(leftCorner, topCorner);
                    break;
                case TOP_RIGHT:
                    draggableRecordingFloatingActionButton.setLocation(rightCorner, topCorner);
                    break;
                case BOTTOM_RIGHT:
                default:
                    draggableRecordingFloatingActionButton.setLocation(rightCorner, bottomCorner);
                    break;
            }

        } else {
            float tempX, tempY;
            tempX = ((float) (x * width)) / (float) tWidth;
            tempY = ((float) (y * height)) / (float) tHeight;

            //adding new values according to the change in width and height
            //in the event of screen rotation
            x = Math.round(tempX);
            y = Math.round(tempY);

            //changing layout params in according to the new position
            layoutParams.leftMargin = x;
            layoutParams.rightMargin = width - x;
            layoutParams.topMargin = y;
            layoutParams.bottomMargin = height - y;

            draggableRecordingFloatingActionButton.setLayoutParams(layoutParams);

            //help the floating button go to the nearest wall
            draggableRecordingFloatingActionButton.goToWall();
        }

        if (draggableRecordingFloatingActionButton != null) {
            draggableRecordingFloatingActionButton.setOnClickListener(this);
            if (floatingButtonFrameLayout != null)
                floatingButtonFrameLayout.addView(draggableRecordingFloatingActionButton);
        }

        setRecordingButtonState();

        ((FrameLayout) activity.getWindow().getDecorView()).addView(floatingButtonFrameLayout,
                new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.MATCH_PARENT));

        new Handler(Looper.getMainLooper()).postDelayed(() -> {
            if (isKeyboardOpen(activity)) {
                isKeyboardOpen = true;
                if (draggableRecordingFloatingActionButton != null)
                    draggableRecordingFloatingActionButton.goToWall();
            }
            if (!isRecording) {
                showHintBubble();
            }
        }, 100);

        registerKeyboardListener(activity);
    }

    private boolean isKeyboardOpen(Activity activity) {
        InputMethodManager imm = (InputMethodManager) activity.getSystemService(Context.INPUT_METHOD_SERVICE);
        return imm != null && imm.isAcceptingText();
    }

    private void registerKeyboardListener(Activity activity) {
        mCurrentActivity = new WeakReference<>(activity);
        keyboardEventListener = new KeyboardEventListener(activity, isKeyboardOpen -> {
            ScreenRecordingFab.this.isKeyboardOpen = isKeyboardOpen;
            if (isKeyboardOpen)
                handleKeyboardOpened();
            else
                handleKeyboardClosed();
            if (expanded) {
                // force expanding to redraw the buttons in right location
                expand();
            }
        });
    }

    private void handleKeyboardOpened() {
        Activity activity = mCurrentActivity != null ? mCurrentActivity.get() : null;
        if (activity == null) {
            return;
        }
        DraggableRecordingFloatingActionButton floatingActionButton = draggableRecordingFloatingActionButton;
        if (floatingActionButton != null) {
            int bottomLocation = getBottomLocationWhenKeyboardOpened(activity);
            int[] buttonLocation = {0, 0};
            floatingActionButton.getLocationOnScreen(buttonLocation);
            floatingActionButton.setLocation(buttonLocation[0], buttonLocation[1] == topCorner ? topCorner : bottomLocation);
        }

    }

    private int getBottomLocationWhenKeyboardOpened(@NonNull Activity activity) {
        View view = activity.getWindow().getDecorView().getRootView();
        Rect visibleBounds = new Rect();
        view.getWindowVisibleDisplayFrame(visibleBounds);
        return visibleBounds.height() - buttonSpacing - floatingButtonSize;
    }

    private void handleKeyboardClosed() {
        int[] buttonLocation = {0, 0};
        if (draggableRecordingFloatingActionButton != null) {
            draggableRecordingFloatingActionButton.getLocationOnScreen(buttonLocation);
        }
        //Don't move the button to the bottom in case it was dragged to the top
        if (buttonLocation[1] == topCorner || draggableRecordingFloatingActionButton == null) {
            return;
        }
        if (mCurrentActivity != null && mCurrentActivity.get() != null)
            height = mCurrentActivity.get().getResources().getDisplayMetrics().heightPixels;
        if (buttonLocation[0] == rightCorner) {
            bottomCorner = height - (floatingButtonSize + buttonSpacing);
        }
        draggableRecordingFloatingActionButton.setLocation(buttonLocation[0], bottomCorner);

        if (isHintBubbleShown) {
            hideHintBubble();
        }
    }


    private void unregisterKeyboardListener() {
        mCurrentActivity = null;
        if (keyboardEventListener != null) {
            keyboardEventListener.unregisterKeyboardListener();
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void hideFAB() {
        unregisterKeyboardListener();
        if (draggableRecordingFloatingActionButton != null) {
            draggableRecordingFloatingActionButton.setOnClickListener(null);
            draggableRecordingFloatingActionButton = null;
        }
        if (floatingButtonFrameLayout != null) {
            floatingButtonFrameLayout.setOnClickListener(null);
            if (floatingButtonFrameLayout.getParent() instanceof ViewGroup) {
                ViewGroup vg = (ViewGroup) (floatingButtonFrameLayout.getParent());
                vg.removeView(floatingButtonFrameLayout);
            }
            floatingButtonFrameLayout = null;
        }
    }

    @Override
    public void onClick(View v) {
        toggle();
        if (!isRecording) {
            if (draggableRecordingFloatingActionButton != null) {
                draggableRecordingFloatingActionButton.setText("00:00", true);
            }
            //start recording
            isRecording = true;
            if (listener != null) {
                listener.start();
            }
            if (draggableRecordingFloatingActionButton != null) {
                draggableRecordingFloatingActionButton.setRecordingState(RecordingFloatingActionButton.RecordingState.RECORDING);
            }
        }

        hideHintBubble();
    }

    private void subscribeToCurranActivityLifeCycle() {
        if (currentActivityLifeCycleSubscriber == null)
            currentActivityLifeCycleSubscriber = CoreServiceLocator.createActivityLifecycleSubscriber(this);
        currentActivityLifeCycleSubscriber.subscribe();
    }

    @VisibleForTesting
    void handleConfigurationChange(Configuration newConfig) {
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (currentActivity != null) {
            handleActivityPaused();
            layoutParams = null;
            width = (int) dpToPixel(currentActivity.getApplicationContext(), newConfig.screenWidthDp);
            height = (int) dpToPixel(currentActivity.getApplicationContext(), newConfig.screenHeightDp);
            showFAB(currentActivity, width, height);
        }
    }

    @Override
    @MainThread
    public void handleActivityResumed() {
        Activity currentActivity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (currentActivity == null) return;

        if (ScreenUtility.getWindowHeight(currentActivity) > 0) {
            showFab(currentActivity);
        } else {
            Runnable showingFabTask = () -> PoolProvider.postMainThreadTask(() -> {
                Activity currentActivityAfterDelay = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                if (currentActivityAfterDelay != null)
                    showFab(currentActivityAfterDelay);
            });
            PoolProvider.postDelayedTask(showingFabTask, 500);
        }
    }

    void showFab(@NonNull Activity currentActivity) {
        height = ScreenUtility.getWindowHeight(currentActivity);
        width = ScreenUtility.getWindowWidth(currentActivity);
        showFAB(currentActivity, width, height);
    }

    @Override
    @MainThread
    public void handleActivityPaused() {
        hideFAB();
        hideHintBubble();
    }

    private void setRecordingButtonState() {
        RecordingFloatingActionButton.RecordingState recordingState;

        recordingState = isRecording ? RecordingFloatingActionButton.RecordingState.RECORDING : RecordingFloatingActionButton.RecordingState.STOPPED;

        if (draggableRecordingFloatingActionButton != null) {
            draggableRecordingFloatingActionButton.setRecordingState(recordingState);
        }
    }

    public void startTimerOnRecordingButton() {

        //start timer
        startTime = System.currentTimeMillis();
        handler.removeCallbacks(updateTimeTask);
        handler.postDelayed(updateTimeTask, 0);
    }

    private void toggle() {
        if (expanded) {
            collapse();
        } else {
            expand();
        }
    }

    private void expand() {
        //Prevent expanding away from the corners.
        int diff = 20;
        int currentTopCorner = topCorner;
        Activity activity = mCurrentActivity != null ? mCurrentActivity.get() : null;
        int[] buttonLocation = {0, 0};
        if (draggableRecordingFloatingActionButton != null) {
            draggableRecordingFloatingActionButton.getLocationOnScreen(buttonLocation);
        }
        if (isKeyboardOpen && activity != null && buttonLocation[1] != topCorner) {
            currentTopCorner = getBottomLocationWhenKeyboardOpened(activity);
        }
        if (layoutParams != null && ((Math.abs(layoutParams.leftMargin - leftCorner) > diff
                && Math.abs(layoutParams.leftMargin - rightCorner) > diff) || (Math.abs(
                layoutParams.topMargin - currentTopCorner) > diff
                && Math.abs(layoutParams.topMargin - bottomCorner) > diff))) {
            return;
        }

        setExtraButtonsLayoutParams();

        if (muteButton != null && muteButton.getParent() != null) {
            ((ViewGroup) muteButton.getParent()).removeView(muteButton);
        }
        if (floatingButtonFrameLayout != null && muteButton != null) {
            floatingButtonFrameLayout.addView(muteButton);
            floatingButtonFrameLayout.setNextFocusForwardId(R.id.instabug_video_mute_button);
        }
        if (stopButton != null && stopButton.getParent() != null) {
            ((ViewGroup) stopButton.getParent()).removeView(stopButton);
        }
        if (floatingButtonFrameLayout != null && stopButton != null) {
            floatingButtonFrameLayout.addView(stopButton);
        }
        expanded = true;
    }

    private void collapse() {
        if (floatingButtonFrameLayout != null && muteButton != null) {
            floatingButtonFrameLayout.removeView(muteButton);
        }
        if (floatingButtonFrameLayout != null && stopButton != null) {
            floatingButtonFrameLayout.removeView(stopButton);
        }
        expanded = false;
    }

    private void setExtraButtonsLayoutParams() {
        FrameLayout.LayoutParams muteButtonLayoutParams =
                new FrameLayout.LayoutParams(extraFloatingButtonSize, extraFloatingButtonSize);

        if (layoutParams != null) {
            muteButtonLayoutParams.leftMargin =
                    layoutParams.leftMargin + (floatingButtonSize - extraFloatingButtonSize) / 2;
            muteButtonLayoutParams.rightMargin =
                    layoutParams.rightMargin + (floatingButtonSize - extraFloatingButtonSize) / 2;
        }

        FrameLayout.LayoutParams stopButtonLayoutParams =
                null;
        if (stopButton != null && layoutParams != null) {
            stopButtonLayoutParams = new FrameLayout.LayoutParams(stopButton.getWidth(), stopButton.getHeight());
            stopButtonLayoutParams.leftMargin =
                    layoutParams.leftMargin + (floatingButtonSize - extraFloatingButtonSize) / 2;
            stopButtonLayoutParams.rightMargin =
                    layoutParams.rightMargin + (floatingButtonSize - extraFloatingButtonSize) / 2;
        }


        int expandingSpace = 2 * (extraFloatingButtonSize + 2 * buttonSpacing);

        int stopButtonTopMargin;
        int muteButtonTopMargin;

        if (layoutParams != null) {
            if (layoutParams.topMargin > expandingSpace) {
                stopButtonTopMargin = layoutParams.topMargin - (extraFloatingButtonSize +
                        buttonSpacing);
                muteButtonTopMargin = stopButtonTopMargin - (extraFloatingButtonSize + buttonSpacing);
            } else {
                stopButtonTopMargin = layoutParams.topMargin + floatingButtonSize + buttonSpacing;
                muteButtonTopMargin = stopButtonTopMargin + extraFloatingButtonSize + buttonSpacing;
            }
            if (stopButtonLayoutParams != null) {
                stopButtonLayoutParams.topMargin = stopButtonTopMargin;
            }
            muteButtonLayoutParams.topMargin = muteButtonTopMargin;
        }
        if (muteButton != null) {
            muteButton.setLayoutParams(muteButtonLayoutParams);
        }
        if (stopButton != null && stopButtonLayoutParams != null) {
            stopButton.setLayoutParams(stopButtonLayoutParams);
        }
    }

    private void showHintBubble() {
        if (layoutParams != null && !isHintBubbleShown && layoutParams.leftMargin != leftCorner) {
            isHintBubbleShown = true;
            final FrameLayout.LayoutParams hintBubbleLayoutParams =
                    new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                            ViewGroup.LayoutParams.WRAP_CONTENT);
            if (hintBubble != null) {
                hintBubble.setLayoutParams(hintBubbleLayoutParams);
                hintBubble.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (hintBubble != null && layoutParams != null) {
                            hintBubbleLayoutParams.leftMargin = layoutParams.leftMargin - hintBubble
                                    .getWidth();
                            hintBubbleLayoutParams.rightMargin = width - layoutParams.leftMargin;
                            hintBubbleLayoutParams.topMargin = layoutParams.topMargin
                                    + ((layoutParams.height + floatingButtonSize) / 2 - hintBubble
                                    .getHeight()) / 2;
                            hintBubble.setLayoutParams(hintBubbleLayoutParams);
                        }
                    }
                }, 100);
            }


            if (floatingButtonFrameLayout != null && hintBubble != null) {
                floatingButtonFrameLayout.addView(hintBubble);
            }
        }
    }

    private void hideHintBubble() {
        if (isHintBubbleShown) {
            isHintBubbleShown = false;
            if (floatingButtonFrameLayout != null && hintBubble != null) {
                floatingButtonFrameLayout.removeView(hintBubble);
            }
        }
    }

    private boolean isExceedingMinDragLimit(float deltaX, float deltaY) {
        return ((deltaX != 0.0f && deltaY != 0.0f) && (deltaX * deltaY > 1.0f) || deltaX * deltaY < -1.0f);
    }

    public interface ScreenRecordingFloatingBtnEventListener {
        void start();

        void pause();

        void stop(int duration);
    }

    static class FlingListener extends GestureDetector.SimpleOnGestureListener {
        private static final float FLING_THRESHOLD_VELOCITY = 90f;

        @Override
        public boolean onFling(@Nullable MotionEvent e1, @Nullable MotionEvent e2, float velocityX, float velocityY) {
            if (e1 == null || e2 == null)
                return false;
            return (Math.abs(e2.getX() - e1.getX()) < FLING_THRESHOLD_VELOCITY && e2.getY() - e1.getY() > FLING_THRESHOLD_VELOCITY);
        }
    }

    public class DraggableRecordingFloatingActionButton extends RecordingFloatingActionButton {

        private static final int TOUCH_TIME_THRESHOLD = 200;

        @Nullable
        private GestureDetector mGestureDetector;

        private static final boolean shouldFlingAway = true;
        @Nullable
        private MoveAnimator mAnimator;
        private long lastTouchDown;
        private float lastXPose, lastYPose;
        private boolean isBeingDragged = false;

        public DraggableRecordingFloatingActionButton(Activity activity) {
            super(activity);
            mGestureDetector = new GestureDetector(activity, new FlingListener());
            mAnimator = new MoveAnimator();
            setId(R.id.instabug_floating_button);
        }

        public DraggableRecordingFloatingActionButton(Context context, AttributeSet attrs) {
            super(context, attrs);
        }

        public DraggableRecordingFloatingActionButton(Context context, AttributeSet attrs,
                                                      int defStyleAttr) {
            super(context, attrs, defStyleAttr);
        }

        @Override
        public void setLayoutParams(ViewGroup.LayoutParams params) {
            layoutParams = ((FrameLayout.LayoutParams) params);
            super.setLayoutParams(params);
        }

        @Override
        public boolean onTouchEvent(MotionEvent event) {
            boolean fling = false;
            if (shouldFlingAway) {
                if (mGestureDetector != null) {
                    fling = mGestureDetector.onTouchEvent(event);
                }
            }
            if (fling) {
                goToWall();
            } else {
                float x = event.getRawX();
                float y = event.getRawY();
                int action = event.getAction();
                if (action == MotionEvent.ACTION_DOWN) {
                    lastTouchDown = System.currentTimeMillis();
                    if (mAnimator != null) {
                        mAnimator.stop();
                    }
                    isBeingDragged = true;
                } else if (action == MotionEvent.ACTION_UP) {
                    if (System.currentTimeMillis() - lastTouchDown < TOUCH_TIME_THRESHOLD) {
                        performClick();
                    }
                    isBeingDragged = false;
                    goToWall();
                } else if (action == MotionEvent.ACTION_MOVE) {
                    if (isBeingDragged) {
                        move(x - lastXPose, y - lastYPose);
                    }
                }
                lastXPose = x;
                lastYPose = y;
            }
            return true;
        }

        private void goToWall() {
            final int nearestXWall = x >= width / 2 ? rightCorner : leftCorner;
            final int nearestYWall;
            if (isKeyboardOpen && mCurrentActivity != null && mCurrentActivity.get() != null) {
                int keyboardHeight = getBottomLocationWhenKeyboardOpened(mCurrentActivity.get());
                nearestYWall = y >= (height - keyboardHeight) / 2 ? keyboardHeight : topCorner;
            } else {
                nearestYWall = y >= height / 2 ? bottomCorner : topCorner;
            }

            if (mAnimator != null) {
                mAnimator.start(nearestXWall, nearestYWall);
            }
        }

        void setLocation(int pX, int pY) {

            x = pX;
            y = pY;

            if (layoutParams != null) {
                layoutParams.leftMargin = x;
                layoutParams.rightMargin = width - x;
                if (orientation == LANDSCAPE_MODE && realWidth > width) {
                    layoutParams.rightMargin =
                            (int) (layoutParams.rightMargin + (NAVIGATION_BAR_SIZE * densityFactor));
                }
                layoutParams.topMargin = y;
                layoutParams.bottomMargin = height - y;
                setLayoutParams(layoutParams);
            }
        }

        void move(float deltaX, float deltaY) {
            if (y + deltaY > MIN_TOP_LOCATION) {
                setLocation((int) (x + deltaX), (int) (y + deltaY));
                setExtraButtonsLayoutParams();
                if (expanded && isExceedingMinDragLimit(deltaX, deltaY)) {
                    collapse();
                }
                hideHintBubble();
            }
            if (shouldFlingAway
                    && !isBeingDragged
                    && layoutParams != null
                    && Math.abs(layoutParams.rightMargin) < 50
                    && Math.abs(
                    layoutParams.topMargin - (getContext().getResources().getDisplayMetrics()
                            .heightPixels
                            / 2)) < 250) {
                goToWall();
            }
        }

        private class MoveAnimator implements Runnable {
            private final Handler handler = new Handler(Looper.getMainLooper());
            private float destinationX;
            private float destinationY;
            private long startingTime;

            private void start(float x, float y) {
                this.destinationX = x;
                this.destinationY = y;
                startingTime = System.currentTimeMillis();
                handler.post(this);
            }

            @Override
            public void run() {
                if (DraggableRecordingFloatingActionButton.this.getParent() != null) {
                    float progress = Math.min(1, (System.currentTimeMillis() - startingTime) /
                            400f);
                    float deltaX = (destinationX - x) * progress;
                    float deltaY = (destinationY - y) * progress;
                    setLocation((int) (x + deltaX), (int) (y + deltaY));
                    if (progress < 1) {
                        handler.post(this);
                    }
                }
            }

            private void stop() {
                handler.removeCallbacks(this);
            }
        }
    }

    private static float dpToPixel(Context c, float dp) {
        float density = c.getResources().getDisplayMetrics().density;
        return dp * density;
    }

}
