package com.instabug.bug.invocation.invoker

import android.annotation.SuppressLint
import android.content.Context
import android.os.Build
import android.view.GestureDetector
import android.view.MotionEvent
import androidx.annotation.VisibleForTesting
import androidx.core.view.GestureDetectorCompat
import com.instabug.bug.invocation.InvocationListener
import com.instabug.bug.invocation.InvocationManager
import com.instabug.library.Constants
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import kotlin.math.abs

/**
 * @author mesbah.
 */
class ThreeFingerSwipeLeftInvoker @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE") constructor(
    private val context: Context, private val invocationListener: InvocationListener
) :
    AbstractInvoker<MotionEvent?> {
    private var gestureDetectorCompat: GestureDetectorCompat? = null
    private var gestureListener: GestureListener? = null

    @VisibleForTesting
    var goodCycle = false

    @Volatile
    private var active = false

    @Synchronized
    override fun listen() {
        PoolProvider.postMainThreadTask {
            val currentContext: Context? = if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q) {
                context
            } else {
                InstabugInternalTrackingDelegate.getInstance()
                    .getCurrentActivity()
            }
            if (currentContext != null) {
                try {
                    val listener = GestureListener()
                    gestureListener = listener
                    gestureDetectorCompat = GestureDetectorCompat(currentContext, listener)
                    active = true
                } catch (exception: Exception) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG,
                        (if (exception.message != null) exception.message else "Couldn't initialize GestureDetector")!!,
                        exception
                    )
                }
            }
        }
    }

    @Synchronized
    override fun handle(event: MotionEvent?) {
        if (gestureDetectorCompat == null) { // invoker is sleep
            return
        }
        //Filter MOVE actions only, because UP and DOWN usually happen with one finger
        // (you can't accurately start the gesture with all three fingers at the exact moment)
        event?.let { event ->
            if (event.action and MotionEvent.ACTION_MASK == MotionEvent.ACTION_MOVE) {
                goodCycle = if (event.pointerCount < NEEDED_CONTAINED_POINTER) {
                    return
                } else {
                    true
                }
            }
            gestureDetectorCompat?.onTouchEvent(event)
        }
    }

    @Synchronized
    override fun sleep() {
        gestureListener = null
        gestureDetectorCompat = null
        active = false
    }

    override fun isActive(): Boolean {
        return active
    }

    internal inner class GestureListener : GestureDetector.SimpleOnGestureListener() {
        override fun onFling(
            e1: MotionEvent?,
            e2: MotionEvent,
            velocityX: Float,
            velocityY: Float
        ): Boolean {
            if (isDirectionValid(e1, e2) && goodCycle) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Three fingers swiped left, invoking SDK")
                InvocationManager.getInstance().lastUsedInvoker = this@ThreeFingerSwipeLeftInvoker
                invocationListener.onInvocationRequested()
            }
            goodCycle = false
            return false
        }
    }

    private fun isDirectionValid(e1: MotionEvent?, e2: MotionEvent?): Boolean {
        return !(e1 == null || e2 == null) && e1.x > e2.x && e1.x - e2.x >= abs((e1.y - e2.y).toDouble())
    }

    companion object {
        /**
         * The needed number of pointers of data contained in this swipe event.
         */
        private const val NEEDED_CONTAINED_POINTER = 3
    }
}
