package com.instabug.chat.ui;

import static androidx.core.view.ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_NO_HIDE_DESCENDANTS;
import static androidx.core.view.ViewCompat.IMPORTANT_FOR_ACCESSIBILITY_YES;
import static com.instabug.chat.ui.ChatActivityLauncher.ChatProcess;
import static com.instabug.chat.ui.ChatActivityLauncher.EXTRA_ATTACHMENT;
import static com.instabug.chat.ui.ChatActivityLauncher.EXTRA_CHAT_NUMBER;
import static com.instabug.chat.ui.ChatActivityLauncher.EXTRA_COMPOSE;
import static com.instabug.chat.ui.ChatActivityLauncher.EXTRA_PROCESS;
import static com.instabug.chat.ui.ChatActivityLauncher.PROCESS_CHAT;
import static com.instabug.chat.ui.ChatActivityLauncher.PROCESS_CHATS;
import static com.instabug.chat.ui.ChatActivityLauncher.PROCESS_HANGING_CHAT;
import static com.instabug.chat.ui.ChatActivityLauncher.PROCESS_NEW_CHAT;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.accessibility.AccessibilityEvent;

import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import com.instabug.chat.ChatPlugin;
import com.instabug.chat.Constants;
import com.instabug.bug.R;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.ui.chat.ChatFragment;
import com.instabug.chat.ui.chats.ChatsFragment;
import com.instabug.chat.util.ChatThemeResolver;
import com.instabug.library._InstabugActivity;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BaseFragmentActivity;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.OrientationUtils;
import com.instabug.library.util.StatusBarUtils;

import java.util.ArrayList;
import java.util.List;

public class ChatActivity extends BaseFragmentActivity<ChatContract.Presenter> implements
        _InstabugActivity, ChatContract.View, ChatsFragment.Callbacks {

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        switch (getChatProcess(intent)) {
            case PROCESS_CHAT:
                String chatNumber = intent.getStringExtra(EXTRA_CHAT_NUMBER);
                if (chatNumber != null)
                    openChat(chatNumber);
                break;
            default:
                break;
        }
    }

    @SuppressLint("STARVATION")
    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null) {
            plugin.setState(Plugin.STATE_FOREGROUND);
        }
        super.onCreate(savedInstanceState);
        OrientationUtils.handelOrientation(this);
        if (SettingsManager.getInstance().getTheme() != null) {
            setTheme(ChatThemeResolver.resolveTheme(SettingsManager.getInstance().getTheme()));
        }
        presenter = new ChatPresenter(this);
        presenter.handleChatProcess(getChatProcess(getIntent()));

        // Handle Accessibility fragment focus changes
        getSupportFragmentManager().addOnBackStackChangedListener(new FragmentManager.OnBackStackChangedListener() {

            @Override
            public void onBackStackChanged() {
                changeAccessibilityImportance();
            }
        });

        // Avoid Talkback to say the AppName when launching the screen
        setTitle("");

        if (!InstabugCore.isFullScreen()) {
            StatusBarUtils.darkenStatusBarColor(this, InstabugCore.getPrimaryColor());
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        if (presenter != null) {
            presenter.dismissSystemNotification();
        }
    }

    public void changeAccessibilityImportance() {
        List<Fragment> fragmentList = getSupportFragmentManager().getFragments();
        List<Fragment> fragmentsWithNonNullViews = new ArrayList<>();
        for (Fragment fragment : fragmentList) {
            if (fragment.getView() != null) {
                fragmentsWithNonNullViews.add(fragment);
            }
        }
        for (int index = 0; index < fragmentsWithNonNullViews.size(); index++) {
        View fragmentView = fragmentsWithNonNullViews.get(index).getView();
        if (fragmentView != null) {
            if (index == fragmentsWithNonNullViews.size() - 1) {
                // This is the visible fragment
                ViewCompat.setImportantForAccessibility(fragmentView, IMPORTANT_FOR_ACCESSIBILITY_YES);
                if (Build.VERSION.SDK_INT < 16) {
                    fragmentView.sendAccessibilityEvent(AccessibilityEvent.TYPE_VIEW_FOCUSED);
                } else {
                    fragmentView.sendAccessibilityEvent(AccessibilityEvent.TYPE_VIEW_ACCESSIBILITY_FOCUSED);
                }

            } else {
                ViewCompat.setImportantForAccessibility(fragmentView, IMPORTANT_FOR_ACCESSIBILITY_NO_HIDE_DESCENDANTS);
            }
        }
    }
}

    @Override
    protected int getLayout() {
        return R.layout.instabug_activity;
    }

    @Override
    protected void initViews() {

    }

    @Override
    public void onStop() {
        super.onStop();
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null && plugin.getState() != Plugin.STATE_PROCESSING_ATTACHMENT) {
            plugin.setState(Plugin.STATE_BACKGROUND);
        }
    }

    @Override
    protected void onDestroy() {
        OrientationUtils.unlockOrientation(this);
        IBGCoreEventPublisher.post(IBGSdkCoreEvent.ForegroundAvailable.INSTANCE);
        super.onDestroy();
    }

    @Override
    public void showChatsFragment() {

        if (!isFinishing()) {

            FragmentManager fragmentManager = getSupportFragmentManager();
            Fragment chatsFragment = fragmentManager.findFragmentByTag(ChatsFragment.TAG);
            if (chatsFragment instanceof ChatsFragment && chatsFragment.isResumed()) {
            } else {
                getSupportFragmentManager().beginTransaction()
                        .replace(R.id.instabug_fragment_container, ChatsFragment.newInstance(
                                shouldEnabledComposeButton()), ChatsFragment.TAG).commit();
            }
        }
    }

    @Override
    public void showChatFragment(String chatNumber) {
        if (!isFinishing()) {
            try {
                getSupportFragmentManager().executePendingTransactions();
                FragmentTransaction fragmentTransaction = getSupportFragmentManager()
                        .beginTransaction()

                        .add(R.id.instabug_fragment_container, ChatFragment.newInstance(chatNumber),
                                ChatFragment.TAG);
                if (getSupportFragmentManager().findFragmentById(R.id.instabug_fragment_container)
                        != null) {
                    fragmentTransaction.addToBackStack(ChatFragment.TAG);
                }
                fragmentTransaction.commitAllowingStateLoss();
            } catch (IllegalStateException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        "Couldn't show Chat fragment due to " + e.getMessage());
            }
        }
    }

    @Override
    public void showChatFragment(String chatNumber, Attachment attachment) {
        if (!isFinishing()) {

            getSupportFragmentManager().executePendingTransactions();
            FragmentTransaction fragmentTransaction = getSupportFragmentManager().beginTransaction()
                    .add(R.id.instabug_fragment_container, ChatFragment.newInstance(chatNumber,
                            attachment), ChatFragment.TAG);
            if (getSupportFragmentManager().findFragmentById(R.id.instabug_fragment_container)
                    != null) {
                fragmentTransaction.addToBackStack(ChatFragment.TAG);
            }
            fragmentTransaction.commit();
        }
    }

    @Override
    @ChatProcess
    public int getChatProcess(Intent intent) {
        int currentProcess = intent.getIntExtra(EXTRA_PROCESS, -1);
        switch (currentProcess) {
            case PROCESS_CHATS:
                return PROCESS_CHATS;
            case PROCESS_CHAT:
                return PROCESS_CHAT;
            case PROCESS_NEW_CHAT:
                return PROCESS_NEW_CHAT;
            case PROCESS_HANGING_CHAT:
                return PROCESS_HANGING_CHAT;
            default:
                return PROCESS_CHATS;
        }
    }

    @Override
    @Nullable
    public String getChatProcessChatNumber() {
        return getIntent().getStringExtra(EXTRA_CHAT_NUMBER);
    }

    @Override
    @Nullable
    public Attachment getChatProcessChatAttachment() {
        return (Attachment) getIntent().getSerializableExtra(EXTRA_ATTACHMENT);
    }

    @Override
    public boolean shouldEnabledComposeButton() {
        Bundle bundle = getIntent().getExtras();
        return bundle != null && bundle.getBoolean(EXTRA_COMPOSE);
    }

    @Override
    public void openChat(String chatNumber) {
        if (presenter != null) {
            presenter.handleOpenChatRequest(chatNumber);
        }
    }


    @Override
    public void finish() {
        if (presenter != null) {
            presenter.handleOnSdkDismissed();
        }
        super.finish();
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        for (Fragment fragment : getSupportFragmentManager().getFragments()) {
            fragment.onActivityResult(requestCode, resultCode, data);
        }
    }
}