package com.instabug.chat.notification;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.media.AudioAttributes;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;

import androidx.annotation.DrawableRes;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.core.app.NotificationCompat;

import com.instabug.bug.R;
import com.instabug.chat.ChatPlugin;
import com.instabug.chat.RepliesWrapper;
import com.instabug.chat.cache.ReadQueueCacheManager;
import com.instabug.chat.model.Message;
import com.instabug.chat.model.NotificationMessage;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.chat.synchronization.SynchronizationManager;
import com.instabug.chat.ui.ChatActivityLauncher;
import com.instabug.chat.util.PlaceHolderResolver;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.PresentationManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.util.InstabugAppData;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.PlaceHolderUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Created by vezikon on 12/21/15.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class NotificationManager {

    private static final int CODE_NOTIFICATION = 0;
    private static final int SINGLE_NOTIFICATION = 0;
    private static final int MULTIPLE_NOTIFICATION = 1;
    private static final String KEY_MESSAGE = "message";
    private static final String KEY_IBG_HOST = "IBGHost";
    private int type;
    private static NotificationManager INSTANCE;
    private final NotificationBarInvoker notificationBarInvoker;
    private InstabugAppData appData;
    private List<Message> messages;

    private NotificationManager() {
        notificationBarInvoker = new NotificationBarInvoker();
    }

    public static synchronized NotificationManager getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new NotificationManager();
        }
        return INSTANCE;
    }

    public void showSystemNotification(final Context context, String message) {
        appData = new InstabugAppData(context);
        Intent intent = ChatActivityLauncher.chatsProcessIntent(context);
        showSystemNotification(context, intent, message);
    }

    public void showNotification(final Context context, List<Message> messages) {
        appData = new InstabugAppData(context);
        String notificationMessage = "";
        Intent intent = null;
        type = getNotificationType(messages);
        this.messages = messages;

        switch (type) {
            case SINGLE_NOTIFICATION:
                Message message = messages.get(messages.size() - 1);
                notificationMessage = generateNotificationBody(context, SINGLE_NOTIFICATION,
                        messages);
                intent = ChatActivityLauncher.chatProcessIntent(context, message.getChatId());
                break;
            case MULTIPLE_NOTIFICATION:
                notificationMessage = generateNotificationBody(context, MULTIPLE_NOTIFICATION,
                        messages);
                intent = ChatActivityLauncher.chatsProcessIntent(context);
                break;
            default:
                break;
        }

        if (!isAppOnForeground() && intent != null) {
            showSystemNotification(context, intent, notificationMessage);
        } else {
            handleAppOnForeground(context, messages, intent, notificationMessage);

        }
    }

    private void handleAppOnForeground(Context context, List<Message> messages, @Nullable Intent intent, @Nullable String notificationMessage) {
        Activity activity = (context instanceof Activity) ? (Activity) context
                : InstabugCore.getTargetActivity();

        if (InstabugCore.isForegroundNotBusy()) {
            if (activity != null) sendInAppNotification(activity, messages);

        } else {
            ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
            if (plugin != null && plugin.getState() == Plugin.STATE_FOREGROUND && activity != null) {
                sendInAppNotification(activity, messages);
            } else {
                if (intent != null) {
                    showSystemNotification(context, intent, notificationMessage);
                }
            }
        }
    }

    private boolean isAppOnForeground() {
        return InstabugCore.getStartedActivitiesCount() > 0;
    }

    public void dismissSystemNotification(@Nullable Context context) {
        if (context != null) {
            android.app.NotificationManager notificationManager =
                    (android.app.NotificationManager) context.getSystemService(Context
                            .NOTIFICATION_SERVICE);
            notificationManager.cancel(CODE_NOTIFICATION);
        }
    }

    public void playNotificationSound(@Nullable Context context) {
        if (context == null || !InstabugDeviceProperties.checkRingerIsOn(context)) {
            return;
        }

        final MediaPlayer mPlayer = MediaPlayer.create(context, com.instabug.library.R.raw
                .ib_core_sound_new_message);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            mPlayer.setAudioAttributes(new AudioAttributes.Builder()
                    .setUsage(AudioAttributes.USAGE_NOTIFICATION)
                    .setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION)
                    .build());
        } else {
            mPlayer.setAudioStreamType(AudioManager.STREAM_NOTIFICATION);
        }

        mPlayer.start();
        mPlayer.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
                mPlayer.release();
            }
        });
    }

    private int getNotificationType(List<Message> messages) {
        int issuesCount = 1;
        List<Message> tempList = new ArrayList<>(messages);
        String temp = messages.get(0).getChatId();
        Collections.sort(tempList, new Message.Comparator(Message.Comparator
                .COMPARE_BY_CONVERSATION_ID));
        for (Message message : tempList) {
            String issueNumber = message.getChatId();
            if (issueNumber != null && !issueNumber.equals(temp)) {
                ++issuesCount;
                temp = issueNumber;
            }
        }
        return issuesCount == 1 ? SINGLE_NOTIFICATION : MULTIPLE_NOTIFICATION;
    }

    @Nullable
    private String generateNotificationBody(@Nullable final Context context, int type, List<Message>
            messages) {
        switch (type) {
            case SINGLE_NOTIFICATION:
                return messages.get(messages.size() - 1).getBody();
            case MULTIPLE_NOTIFICATION:
                if (context != null) {
                    String senderName = messages.get(messages.size() - 1).getSenderName();
                    if (senderName != null) {
                        String[] fullName = senderName.split(" ");
                        return String.format(
                                PlaceHolderUtils.getPlaceHolder(InstabugCustomTextPlaceHolder.Key.CHATS_MULTIPLE_MESSAGE_NOTIFICATION,
                                        LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(context),
                                                R.string.instabug_str_notifications_body,
                                                context)),
                                messages.size(), fullName[0]);
                    }
                }
            default:
                return "";
        }
    }

    private String generateNotificationTitle(int type, @Nullable String name) {
        //Single message ex. Ahmed (Instabug Team)
        //Multiple messages ex. Instabug Team
        if (name == null || name.equals("null")) {
            return PlaceHolderResolver.getTeamTitleResolved();
        }
        switch (type) {
            case SINGLE_NOTIFICATION:
                return name + " (" + PlaceHolderResolver.getTeamTitleResolved() + ")";
            case MULTIPLE_NOTIFICATION:
                return PlaceHolderResolver.getTeamTitleResolved();
        }
        return "";
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void showSystemNotification(Context context, Intent intent, @Nullable CharSequence contentText) {
        if (!RepliesWrapper.isMessagingServiceAvailable()) return;

        @DrawableRes int notificationIcon = ChatSettings.getNotificationIcon();

        if (notificationIcon == -1 || notificationIcon == 0) {
            notificationIcon = appData.getAppIcon();
        }

        String channelId;
        if (ChatSettings.getPushNotificationChannelId() != null) {
            channelId = ChatSettings.getPushNotificationChannelId();
        } else {
            channelId = "ibg-replies-channel";
        }
        // we will need to differentiate between the silent and sounded notifications by channels
        // https://stackoverflow.com/a/55405084/2188269
        if (!ChatSettings.isSystemNotificationSoundEnabled()) {
            channelId = channelId + "-silent";
        }

        int flags;
        if ((android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M)) {
            flags = PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE;
        } else {
            flags = PendingIntent.FLAG_UPDATE_CURRENT;
        }


        PendingIntent pendingIntent = PendingIntent.getActivity(context, CODE_NOTIFICATION, intent,
                flags);

        Uri defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
        NotificationCompat.Builder notificationBuilder = new NotificationCompat.Builder(context,
                channelId)
                .setSmallIcon(notificationIcon)
                .setContentTitle(appData.getAppName())
                .setContentText(contentText)
                .setAutoCancel(true)
                .setContentIntent(pendingIntent);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN)
            notificationBuilder.setPriority(Notification.PRIORITY_HIGH);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
            notificationBuilder.setVibrate(new long[0]);
        if (ChatSettings.isSystemNotificationSoundEnabled()) {
            notificationBuilder.setSound(defaultSoundUri);
        }

        android.app.NotificationManager notificationManager =
                (android.app.NotificationManager) context.getSystemService(Context
                        .NOTIFICATION_SERVICE);

        if (notificationManager != null) {
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                NotificationChannel notificationChannel = buildNotificationChannel(
                        channelId, appData.getAppName(), defaultSoundUri);

                notificationManager.createNotificationChannel(notificationChannel);
            }
            notificationManager.notify(CODE_NOTIFICATION, notificationBuilder.build());
        }

    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private NotificationChannel buildNotificationChannel(String channelId, String appName, Uri defaultSoundUri) {
        NotificationChannel notificationChannel = new NotificationChannel(channelId,
                appName, android.app.NotificationManager.IMPORTANCE_HIGH);
        if (ChatSettings.isSystemNotificationSoundEnabled()) {
            notificationChannel.setSound(defaultSoundUri, null);
        } else {
            notificationChannel.setSound(null, null);
        }
        return notificationChannel;
    }

    public void showNotification(Bundle data) {
        if (InstabugCore.getFeatureState(IBGFeature.PUSH_NOTIFICATION) == Feature.State.ENABLED) {
            if (isInstabugNotification(data) && SynchronizationManager.getInstance() != null) {
                SynchronizationManager.getInstance().sync(false);
            }
        }
    }

    public void showNotification(Map<String, String> data) {
        if (InstabugCore.getFeatureState(IBGFeature.PUSH_NOTIFICATION) == Feature.State.ENABLED) {
            if (isInstabugNotification(data) && SynchronizationManager.getInstance() != null) {
                SynchronizationManager.getInstance().sync(false);
            }
        }
    }

    public boolean isInstabugNotification(Bundle data) {
        if(data.containsKey(KEY_MESSAGE)) {
            try {
                String message = data.getString(KEY_MESSAGE);
                if (message != null) {
                    JSONObject object = new JSONObject(message);
                    String origin = object.getString(KEY_IBG_HOST);
                    InstabugSDKLogger.d(Constants.LOG_TAG, "IBGHost: " + origin);
                    if (origin != null && Boolean.parseBoolean(origin))
                        return true;
                }
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Parsing GCM response failed", e);
            } catch (NullPointerException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while showing notification", e);
            }
        } else if (data.containsKey(KEY_IBG_HOST)) {
            String message = data.getString(KEY_IBG_HOST);
            return message != null && Boolean.parseBoolean(message);
        }
        return false;
    }

    public boolean isInstabugNotification(Map<String, String> data) {
        if (data.containsKey(KEY_MESSAGE)) {
            try {
                String message = data.get(KEY_MESSAGE);
                JSONObject object = new JSONObject(message);
                String origin = object.getString(KEY_IBG_HOST);
                if (origin != null && Boolean.parseBoolean(origin))
                    return true;
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Parsing GCM response failed", e);
            } catch (NullPointerException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while showing notification", e);
            }
        } else if(data.containsKey(KEY_IBG_HOST)) {
            String ibgHost = data.get(KEY_IBG_HOST);
            return ibgHost != null && Boolean.parseBoolean(ibgHost);
        }
        return false;
    }

    private void sendInAppNotification(final Activity activity, final List<Message> messages) {
        if (InstabugCore.isFeatureEnabled(IBGFeature.REPLIES)) {
            WeakReference<Activity> activityWeakReference = new WeakReference<>(activity);
            final Message lastMessage = messages.get(messages.size() - 1);
            Context applicationContext = Instabug.getApplicationContext();
            NotificationMessage message = createNotificationMessage(applicationContext, type, lastMessage);
            notificationBarInvoker.show(activityWeakReference, message,
                    new NotificationBarInvoker.OnButtonsClickListener() {
                        @Override
                        public void onReply() {
                            if (!InstabugCore.isFeatureEnabled(IBGFeature.REPLIES)) return;
                            handleInAppNotificationClickEvent();
                            PresentationManager.getInstance().setNotificationShowing(false);
                        }

                        @Override
                        public void onDismiss() {
                            ReadQueueCacheManager.getInstance().markAsRead(lastMessage);
                            if (SynchronizationManager.getInstance() != null) {
                                SynchronizationManager.getInstance().sync(false);
                            }
                            notifyNotificationDismissed();
                        }
                    });
            PresentationManager.getInstance().setNotificationShowing(true);
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private NotificationMessage createNotificationMessage(@Nullable Context context, int type, Message lastMessage) {
        NotificationMessage message;
        switch (type) {
            default:
            case SINGLE_NOTIFICATION:
                message = new NotificationMessage();
                message.setBody(generateNotificationBody(context, SINGLE_NOTIFICATION,
                        messages));
                message.setFrom(generateNotificationTitle(SINGLE_NOTIFICATION,
                        lastMessage.getSenderName()));
                message.setAvatarURL(lastMessage.getSenderAvatarUrl());
                break;
            case MULTIPLE_NOTIFICATION:
                message = new NotificationMessage();
                message.setBody(generateNotificationBody(context, MULTIPLE_NOTIFICATION,
                        messages));
                message.setFrom(generateNotificationTitle(MULTIPLE_NOTIFICATION,
                        lastMessage.getSenderName()));
                message.setAvatarURL(lastMessage.getSenderAvatarUrl());
                break;
        }
        return message;
    }

    private void handleInAppNotificationClickEvent() {
        Intent intent;
        Context context = Instabug.getApplicationContext();
        switch (type) {
            default:
            case SINGLE_NOTIFICATION:
                Message message = messages.get(messages.size() - 1);
                if (context != null) {
                    intent = ChatActivityLauncher.chatProcessIntent(context, message.getChatId());
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    context.startActivity(intent);
                }
                break;
            case MULTIPLE_NOTIFICATION:
                if (context != null) {
                    intent = ChatActivityLauncher.chatsProcessIntent(context);
                    context.startActivity(intent);
                }
                break;
        }
    }

    private void notifyNotificationDismissed() {
        PresentationManager.getInstance().setNotificationShowing(false);
        PresentationManager.getInstance().notifyActivityChanged();
    }
}