package com.instabug.chat.network.service;

import android.annotation.SuppressLint;

import androidx.annotation.Nullable;

import com.instabug.chat.Constants;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.network.util.MessagingServiceUtil;
import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.model.State;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.FileToUpload;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Created by mesbah on 12/27/2015.
 */
public class MessagingService {


    private static final String MESSAGE_PARAM = "message";
    private static final String BODY_PARAM = "body";
    private static final String MESSAGED_AT_PARAM = "messaged_at";
    private static final String USER_EMAIL_PARAM = "email";
    private static final String USER_NAME_PARAM = "name";
    private static final String FILE_TYPE = "metadata[file_type]";
    private static final String DURATION = "metadata[duration]";
    private static final String PUSH_TOKEN = "push_token";

    private static MessagingService INSTANCE;
    private NetworkManager networkManager;

    private MessagingService() {
        networkManager = new NetworkManager();
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return singleton instance of MessagingService
     */
    public static MessagingService getInstance() {
        synchronized (MessagingService.class.getName()) {
            if (INSTANCE == null) {
                INSTANCE = new MessagingService();
            }
            return INSTANCE;
        }
    }

    public void triggerChat(@Nullable State state, final Request.Callbacks<String,
            Throwable> triggeringChatCallbacks) throws JSONException {
        if (state != null && triggeringChatCallbacks != null) {

            Request.Builder triggeringChatRequestBuilder = new Request.Builder()
                    .endpoint(Endpoints.TRIGGER_CHAT)
                    .method(RequestMethod.POST);

            List<State.StateItem<?>> stateItems = state.getStateItems();
            List userDataKeys = Arrays.asList(State.getUserDataKeys());
            for (int i = 0; i < state.getStateItems().size(); i++) {
                String key = stateItems.get(i).getKey();
                Object stateItemValue = stateItems.get(i).getValue();
                if (key != null && stateItemValue != null) {
                    triggeringChatRequestBuilder.addParameter(new RequestParameter<>(key, stateItemValue));
                }
            }

            // do request with default connection timeout.
            networkManager.doRequest(IBGNetworkWorker.CHATS, RequestType.NORMAL, triggeringChatRequestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    if (requestResponse != null) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "triggeringChatRequest Succeeded, Response code: " +
                                requestResponse.getResponseCode());
                        InstabugSDKLogger.v(Constants.LOG_TAG, "triggeringChatRequest Succeeded, Response body: " + requestResponse.getResponseBody());
                        if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                            try {
                                if (requestResponse.getResponseBody() != null) {
                                    triggeringChatCallbacks.onSucceeded(new JSONObject((String)
                                            requestResponse.getResponseBody()).getString("chat_number"));
                                }
                            } catch (JSONException e) {
                            }
                        } else {
                            triggeringChatCallbacks.onFailed(new Throwable("Triggering chat got error " +
                                    "with response code:" + requestResponse.getResponseCode()));
                        }
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "triggeringChatRequest got error: " + error.getMessage());
                    triggeringChatCallbacks.onFailed(error);
                }
            });
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void sendMessage(final Message message, final Request
            .Callbacks<String, Throwable> sendingMessageCallbacks) throws JSONException {
        if (message != null && sendingMessageCallbacks != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Sending message");

            // create sendMessage request
            Request.Builder sendMessageRequestBuilder = new Request.Builder()
                    .endpoint(Endpoints.SEND_MESSAGE.replaceAll(":chat_number", message.getChatId()))
                    .method(RequestMethod.POST)
                    .tokenProvider(() -> message.getAppToken() == null ? TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken() : message.getAppToken());

            sendMessageRequestBuilder.addParameter(new RequestParameter<>(MESSAGE_PARAM, new JSONObject()
                            .put(BODY_PARAM, message.getBody())
                            .put(MESSAGED_AT_PARAM, message.getMessagedAt())
                            .put(USER_EMAIL_PARAM, message.getRequesterEmail())
                            .put(USER_NAME_PARAM, message.getRequesterName())
                            .put(PUSH_TOKEN, message.getDeviceToken())
                    )
            );

            Request sendMessageRequest = sendMessageRequestBuilder.build();

            // do request with default connection timeout.
            networkManager.doRequest(IBGNetworkWorker.CHATS, RequestType.NORMAL, sendMessageRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    if (requestResponse != null) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "sendMessage request Succeeded, Response code: " + requestResponse.getResponseCode());
                        InstabugSDKLogger.v(Constants.LOG_TAG, "sendMessage request Succeeded, Response body: " + requestResponse.getResponseBody());
                        if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                            try {
                                Object response = requestResponse.getResponseBody();
                                if (response instanceof String) {
                                    sendingMessageCallbacks.onSucceeded(new JSONObject((String) response).getString("message_id"));
                                }
                            } catch (JSONException e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Sending message got error: " + e.getMessage());
                            }
                        } else {
                            sendingMessageCallbacks.onFailed(new Throwable("Sending message got error " +
                                    "with response code:" + requestResponse.getResponseCode()));
                        }
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "sendMessage request got error: " + error.getMessage());
                    sendingMessageCallbacks.onFailed(error);
                }
            });
        }
    }


    public synchronized void uploadMessageAttachments(final Message message, final Request.Callbacks<Boolean, Message> uploadMessageAttachmentsCallbacks) throws JSONException {
        if (message != null && uploadMessageAttachmentsCallbacks != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Uploading message attachments");
            final List<Attachment> synced = new ArrayList<>();
            for (int i = 0; i < message.getAttachments().size(); i++) {
                final Attachment attachment = message.getAttachments().get(i);
                InstabugSDKLogger.v(Constants.LOG_TAG, "Uploading attachment with type: " + attachment.getType());

                if (attachment.getType() != null && attachment.getName() != null
                        && attachment.getLocalPath() != null && attachment.getFileType() != null
                        && message.getChatId() != null) {

                    String endPoint = Endpoints.ADD_MESSAGE_ATTACHMENT.replaceAll(":chat_number", message.getChatId())
                            .replaceAll(":message_id", String.valueOf(message.getId()));

                    // create attachment request.
                    Request.Builder requestBuilder = new Request.Builder()
                            .method(RequestMethod.POST)
                            .type(RequestType.MULTI_PART)
                            .endpoint(endPoint)
                            .tokenProvider(() -> message.getAppToken() == null ? TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken() : message.getAppToken());


                    requestBuilder.addParameter(new RequestParameter<>(FILE_TYPE, attachment.getType()));

                    if (attachment.getType().equals(Attachment.AttachmentType.TYPE_AUDIO) && attachment.getDuration() != null)
                        requestBuilder.addParameter(new RequestParameter<>(DURATION, attachment.getDuration()));

                    requestBuilder.fileToUpload(new FileToUpload("file",
                            attachment.getName(), attachment.getLocalPath(), attachment.getFileType()));

                    InstabugSDKLogger.v(Constants.LOG_TAG, "Uploading attachment with name: " + attachment.getName()
                            + " path: " + attachment.getLocalPath() + " file type: " + attachment
                            .getFileType());

                    File file = new File(attachment.getLocalPath());
                    if (file.exists() && file.length() > 0) {
                        attachment.setState(Attachment.AttachmentState.STATE_SYNCED);

                        networkManager.doRequest(IBGNetworkWorker.CHATS, RequestType.MULTI_PART, requestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
                            @Override
                            public void onSucceeded(RequestResponse requestResponse) {
                                InstabugSDKLogger.d(Constants.LOG_TAG, "uploadingMessageAttachmentRequest succeeded, Response code: " + requestResponse.getResponseCode());
                                InstabugSDKLogger.v(Constants.LOG_TAG, "uploadingMessageAttachmentRequest succeeded, Response body: " + requestResponse.getResponseBody());

                                synced.add(attachment);

                                if (synced.size() == message.getAttachments().size()) {
                                    uploadMessageAttachmentsCallbacks.onSucceeded(true);
                                }
                            }

                            @Override
                            public void onFailed(Throwable error) {
                                InstabugSDKLogger.v(Constants.LOG_TAG, "uploadingMessageAttachmentRequest got error: "
                                        + error
                                        .getMessage());

                                synced.add(attachment);

                                if (synced.size() == message.getAttachments().size()) {
                                    uploadMessageAttachmentsCallbacks.onFailed(message);
                                }
                            }
                        });
                    } else {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Skipping attachment file of type "
                                + attachment.getType() + " because it's either not found or empty file");
                    }
                }
            }
        }
    }


    public void syncMessages(long lastMessageMessagedAtUTCDate, final int
            messagesCount, JSONArray readMessages,
                             final Request.Callbacks<RequestResponse, Throwable>
                                     syncMessageCallbacks) throws JSONException {

        if (readMessages != null && syncMessageCallbacks != null) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Syncing messages with server");

            // create syncMessages request
            Request syncMessagesRequest = MessagingServiceUtil.buildSyncMessagesRequest(lastMessageMessagedAtUTCDate, messagesCount, readMessages);

            // do request with NORMAL request type.
            networkManager.doRequest(IBGNetworkWorker.CHATS, RequestType.NORMAL, syncMessagesRequest, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "syncMessages request Succeeded, Response code: " + requestResponse.getResponseCode());
                    InstabugSDKLogger.v(Constants.LOG_TAG, "syncMessages request Succeeded, Response body: " + requestResponse.getResponseBody());
                    syncMessageCallbacks.onSucceeded(requestResponse);
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "syncMessages request got error: " + error.getMessage());
                    syncMessageCallbacks.onFailed(error);
                }
            });
        }
    }

    public void uploadChatLogs(final Chat chat, final Request.Callbacks<Boolean, Chat> callbacks) {
        if (chat != null && callbacks != null) {
            // create logs request
            Request.Builder requestBuilder = new Request.Builder()
                    .method(RequestMethod.POST)
                    .endpoint(Endpoints.CHAT_LOGS.replaceAll(":chat_token",
                            chat.getId()));


            if (chat.getState() != null) {
                ArrayList<State.StateItem> logsItems = chat.getState().getLogsItems();
                for (State.StateItem logItem : logsItems) {
                    //TODO (split state logs) https://instabug.atlassian.net/browse/IBGBUGCHAT-2170
                    if (logItem.getKey() != null && !logItem.getKey().equals(State.KEY_VISUAL_USER_STEPS) &&
                            !logItem.getKey().equals(State.KEY_SESSIONS_PROFILER) && logItem.getValue() != null) {
                        requestBuilder.addParameter(new RequestParameter<>(logItem.getKey(),
                                logItem.getValue()));
                    }
                }
            }
            networkManager.doRequest(IBGNetworkWorker.CHATS, RequestType.NORMAL, requestBuilder.build(), new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "uploading chat logs onNext, Response code: " + requestResponse.getResponseCode());
                    InstabugSDKLogger.v(Constants.LOG_TAG, "uploading chat logs onNext, Response body: " + requestResponse.getResponseBody());

                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "uploading chat logs got error: " + error.getMessage());
                    callbacks.onFailed(chat);
                }
            });
        }
    }

    public synchronized void sendPushNotificationToken(final String token, final Request.Callbacks<Boolean, Throwable> requestCallbacks) {

        if (token != null && requestCallbacks != null) {
            Request request = new Request.Builder()
                    .endpoint(Endpoints.PUSH_TOKEN)
                    .method(RequestMethod.POST)
                    .addParameter(new RequestParameter<>(PUSH_TOKEN, token))
                    .build();

            networkManager.doRequestOnSameThread(RequestType.NORMAL, request, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse requestResponse) {
                    if (requestResponse != null) {
                        InstabugSDKLogger.d(Constants.LOG_TAG, "Sending push notification request Succeeded");
                        if (requestResponse.getResponseCode() == RequestResponse.HttpStatusCode._2xx.OK) {
                            requestCallbacks.onSucceeded(true);
                        } else {
                            InstabugSDKLogger.e(Constants.LOG_TAG, "sending push notification token got error" +
                                    " with response code: " + requestResponse.getResponseCode());
                        }

                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "sending push notification token got error: " + error.getMessage());
                }
            });
        }
    }
}
