package com.instabug.chat.annotation;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.ColorInt;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.Size;
import androidx.core.view.MotionEventCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import androidx.customview.widget.ExploreByTouchHelper;

import com.instabug.bug.R;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.view.ViewUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class ColorPickerPopUpView extends View {

    public static final int DEFAULT_COLOR = 0xffff0000;
    public static final String COLOR_PICKER = "instabug_color_picker";
    public static final String SELECTED_COLOR = "selected_color";
    private static final int DEFAULT_SELECTED_CIRCLE = 0;
    private static final int DEFAULT_BACKGROUND_COLOR = Color.argb(255, 247, 247, 247);
    @Nullable
    private RectF bounds;
    private Orientation orientation;
    private int popUpBackgroundColor;
    @Nullable
    private OnColorSelectionListener onColorSelectionListener;
    private final InternalViewsAccessibilityHelper accessHelper = new InternalViewsAccessibilityHelper(this);
    @Size(7)
    private int[] colors = {
            DEFAULT_COLOR, 0xff0cff01, 0xffff01fc, 0xff0d7aff, 0xffff7e00, 0xfffffc00, 0xffb7b7b7
    };

    private int selectedColor;
    private List<CheckCircleColor> checkCircleColorList;

    public ColorPickerPopUpView(Context context) {
        this(context, null);
    }

    public ColorPickerPopUpView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public ColorPickerPopUpView(Context context, @Nullable AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(attrs, defStyle);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void init(@Nullable AttributeSet attrs, int defStyle) {

        ViewCompat.setAccessibilityDelegate(this, accessHelper);

        // Load attributes
        final TypedArray array = getContext().obtainStyledAttributes(
                attrs, R.styleable.ColorPickerPopUpView, defStyle, 0
        );

        orientation = getScreenOrientation() == Orientation.PORTRAIT ? Orientation.PORTRAIT :
                Orientation.LANDSCAPE;

        array.recycle();

        popUpBackgroundColor = DEFAULT_BACKGROUND_COLOR;

        checkCircleColorList = new ArrayList<>();

        for (int color : colors) {
            checkCircleColorList.add(new CheckCircleColor(color));
        }

        setSelectedColor(DEFAULT_SELECTED_CIRCLE);
    }

    private Orientation getScreenOrientation() {
        if (getResources().getConfiguration().orientation == Configuration
                .ORIENTATION_LANDSCAPE)
            return Orientation.LANDSCAPE;
        else
            return Orientation.PORTRAIT;
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        int width;
        int height;
        width = getMeasuredWidth();
        height = width * 55 / 426;
        setMeasuredDimension(width, height);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {

        int action = MotionEventCompat.getActionMasked(event);

        float eventX = event.getX();
        float eventY = event.getY();

        if (action == MotionEvent.ACTION_UP) {
            for (int i = 0; i < checkCircleColorList.size(); i++) {
                if (checkCircleColorList.get(i).getHitArea().contains(eventX, eventY)) {
                    setSelectedColor(i);
                    break;
                }
            }
        }

        return true;
    }

    private static int find(int[] array, int target) {
        for (int i = 0; i < array.length; i++) {
            if (array[i] == target) {
                return i;
            }
        }
        return DEFAULT_SELECTED_CIRCLE;
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);

        orientation = getScreenOrientation() == Orientation.PORTRAIT ? Orientation.PORTRAIT :
                Orientation.LANDSCAPE;
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        if (orientation == Orientation.LANDSCAPE) {
            float height = w * 55 / 406f;
            if (InstabugDeviceProperties.isTablet(getContext())) {
                bounds = new RectF(0, ViewUtils.convertDpToPx(getContext(), 110), w, height);
            } else {
                bounds = new RectF(0, ViewUtils.convertDpToPx(getContext(), 50), w, height);
            }
        } else {
            float height = w * 55 / (float) 420;
            if (InstabugDeviceProperties.isTablet(getContext())) {
                bounds = new RectF(0, ViewUtils.convertDpToPx(getContext(), 60), w, height);
            } else {
                bounds = new RectF(0, ViewUtils.convertDpToPx(getContext(), 16), w, height);
            }
        }
        // Circle 1
        checkCircleColorList.get(0).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.04429f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.09857f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 2
        checkCircleColorList.get(1).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.18714f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.24143f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 3
        checkCircleColorList.get(2).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.33f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.38429f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 4
        checkCircleColorList.get(3).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.47286f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.52714f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 5
        checkCircleColorList.get(4).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.61571f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.67f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 6
        checkCircleColorList.get(5).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.75857f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.81286f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

        // Circle 7
        checkCircleColorList.get(6).rect = new RectF(bounds.left + (float) Math.
                floor(bounds.width() * 0.90143f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.2f + 0.5f),
                bounds.left + (float) Math.floor(bounds.width() * 0.95571f + 0.5f),
                bounds.top + (float) Math.floor(bounds.height() * 0.66504f + 0.5f));

    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        drawHorizontalBubbleView(canvas);
    }

    private void drawCircleCheckColor(Canvas canvas, CheckCircleColor checkCircleColor) {

        RectF frame = checkCircleColor.rect;
        int circleColor = checkCircleColor.color;

        Paint paint;

        // OvalColor
        RectF ovalColorRect =
                new RectF(frame.left, frame.top, frame.left + (float) Math.floor(frame.height() +
                        0.5f),
                        frame.top + (float) Math.floor(frame.height() + 0.5f));
        Path ovalColorPath = new Path();
        ovalColorPath.addOval(ovalColorRect, Path.Direction.CW);

        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setStyle(Paint.Style.FILL);
        paint.setColor(circleColor);
        canvas.drawPath(ovalColorPath, paint);

        // Check
        if (checkCircleColor.checkVisibility) {
            Path checkPath = new Path();
            checkPath.moveTo(frame.left + frame.height() * 0.20313f,
                    frame.top + frame.height() * 0.51758f);
            checkPath.lineTo(frame.left + frame.height() * 0.39844f,
                    frame.top + frame.height() * 0.71875f);
            checkPath.lineTo(frame.left + frame.height() * 0.79492f,
                    frame.top + frame.height() * 0.33008f);
            checkPath.lineTo(frame.left + frame.height() * 0.74805f,
                    frame.top + frame.height() * 0.28125f);
            checkPath.lineTo(frame.left + frame.height() * 0.39844f, frame.top + frame.height() *
                    0.625f);
            checkPath.lineTo(frame.left + frame.height() * 0.25f, frame.top + frame.height() *
                    0.47266f);
            checkPath.lineTo(frame.left + frame.height() * 0.20313f,
                    frame.top + frame.height() * 0.51758f);
            checkPath.close();

            paint = new Paint(Paint.ANTI_ALIAS_FLAG);
            paint.setStyle(Paint.Style.FILL);
            paint.setColor(Color.WHITE);
            canvas.drawPath(checkPath, paint);
        }
    }

    private void drawHorizontalBubbleView(Canvas canvas) {
        // General Declarations
        Paint paint;
        // Bubble
        Path bubblePath = new Path();
        if (bounds != null) {
            bubblePath.moveTo(bounds.left, bounds.top);
            bubblePath.lineTo(bounds.width(), bounds.top);
            bubblePath.lineTo(bounds.width(), bounds.height() + 200);
            bubblePath.lineTo(bounds.left, bounds.height() + 200);
            bubblePath.lineTo(bounds.left, bounds.top);
            bubblePath.close();
        }

        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setStyle(Paint.Style.FILL);
        paint.setColor(popUpBackgroundColor);
        canvas.drawPath(bubblePath, paint);

        canvas.save();
        canvas.restore();

        for (CheckCircleColor checkCircleColor : checkCircleColorList) {

            drawCircleCheckColor(canvas, checkCircleColor);
        }
    }

    public void setColors(@Size(7) int[] colors) {
        this.colors = Arrays.copyOf(colors, colors.length);
    }

    public void setPopUpBackgroundColor(int popUpBackgroundColor) {
        this.popUpBackgroundColor = popUpBackgroundColor;
        invalidate();
    }

    public void setOrientation(Orientation orientation) {
        this.orientation = orientation;
    }

    public int getSelectedColor() {
        return selectedColor;
    }

    private void setSelectedColor(int index) {
        selectedColor = colors[index];
        for (int i = 0; i < checkCircleColorList.size(); i++) {
            checkCircleColorList.get(i).checkVisibility = i == index;
        }

        invalidate();

        if (onColorSelectionListener != null) {
            onColorSelectionListener.onColorSelected(selectedColor, index);
        }
    }

    public void setOnColorSelectionListener(@Nullable OnColorSelectionListener onColorSelectionListener) {
        this.onColorSelectionListener = onColorSelectionListener;
    }

    public enum Orientation {
        PORTRAIT,
        LANDSCAPE
    }

    public interface OnColorSelectionListener {
        void onColorSelected(@ColorInt int color, int index);
    }

    static class CheckCircleColor {

        RectF rect = new RectF();

        int color;

        boolean checkVisibility;

        public CheckCircleColor(int color) {
            this.color = color;
        }

        public RectF getHitArea() {
            RectF rect = new RectF();
            rect.set(this.rect);
            rect.inset(-this.rect.width() / 3, -this.rect.height() / 3);
            return rect;
        }
    }

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putParcelable(COLOR_PICKER, super.onSaveInstanceState());
        bundle.putInt(SELECTED_COLOR, selectedColor);
        return bundle;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected void onRestoreInstanceState(Parcelable state) {
        if (state instanceof Bundle) {
            Bundle bundle = (Bundle) state;
            selectedColor = bundle.getInt(SELECTED_COLOR);
            state = bundle.getParcelable(COLOR_PICKER);
            setSelectedColor(find(colors, selectedColor));
        }
        super.onRestoreInstanceState(state);
    }

    public List<CheckCircleColor> getCheckCircleColorList() {
        return checkCircleColorList;
    }

    @Override
    protected boolean dispatchHoverEvent(MotionEvent event) {
        if (accessHelper.dispatchHoverEvent(event)) {
            return true;
        }
        return super.dispatchHoverEvent(event);
    }

    private static class InternalViewsAccessibilityHelper extends ExploreByTouchHelper {

        private final ColorPickerPopUpView host;
        private final String[] colorsNames;

        /**
         * Constructs a new helper that can expose a virtual view hierarchy for the
         * specified host view.
         *
         * @param host view whose virtual view hierarchy is exposed by this helper
         */
        public InternalViewsAccessibilityHelper(@NonNull View host) {
            super(host);
            this.host = (ColorPickerPopUpView) host;
            this.colorsNames = host.getContext().getResources().getStringArray(R.array.ibg_bug_annotation_colors_names_content_description);
        }

        @Override
        protected void getVisibleVirtualViews(List<Integer> virtualViewIds) {
            int colorsSize = host.getCheckCircleColorList().size();
            for (int index = 0; index < colorsSize; index++) {
                virtualViewIds.add(index);
            }
        }

        @Override
        protected int getVirtualViewAt(float x, float y) {
            float intervalWidth = ((float) host.getWidth()) / host.getCheckCircleColorList().size();
            return Math.min((int) (x / intervalWidth), host.getCheckCircleColorList().size());
        }

        @Override
        protected void onPopulateNodeForVirtualView(int virtualViewId, @NonNull AccessibilityNodeInfoCompat node) {
            node.setClassName(ColorPickerPopUpView.class.getSimpleName());
            node.setContentDescription(colorsNames[virtualViewId]);
            node.setRoleDescription("Button");
            node.addAction(AccessibilityNodeInfoCompat.ACTION_CLICK);
            RectF rectF = host.getCheckCircleColorList().get(virtualViewId).rect;
            Rect colorRect = new Rect();
            rectF.round(colorRect);
            node.setBoundsInParent(colorRect);
        }

        @Override
        protected boolean onPerformActionForVirtualView(int virtualViewId, int action, @Nullable Bundle arguments) {
            return false;
        }
    }
}