package com.instabug.chat.annotation;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PathMeasure;
import android.graphics.PointF;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.ViewGroup;

import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageView;

import com.instabug.chat.annotation.recognition.PathRecognizer;
import com.instabug.chat.annotation.recognition.PathRecognizer.Recognition;
import com.instabug.chat.annotation.recognition.ShapeSpecs;
import com.instabug.chat.annotation.shape.ArrowShape;
import com.instabug.chat.annotation.shape.BlurredRectShape;
import com.instabug.chat.annotation.shape.OvalShape;
import com.instabug.chat.annotation.shape.PathShape;
import com.instabug.chat.annotation.shape.RectShape;
import com.instabug.chat.annotation.shape.Shape;
import com.instabug.chat.annotation.shape.ZoomedShape;
import com.instabug.chat.annotation.utility.AspectRatioCalculator;
import com.instabug.library.util.DrawingUtility;
import com.instabug.library.util.OrientationUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;


@SuppressLint({"LI_LAZY_INIT_UPDATE_STATIC", "ST_WRITE_TO_STATIC_FROM_INSTANCE_METHOD"})
public class AnnotationView extends AppCompatImageView {

    private static final float TOUCH_TOLERANCE = 8;
    private static final String SUPER_STATE = "superState";
    private static final String ASPECT_RATIO_CALCULATOR = "aspectRatioCalculator";
    private static final String DRAWING_MODE = "drawingMode";
    private static final String DRAWING_LEVEL = "drawingLevel";
    private static final String MAGNIFIERS_COUNT = "magnifiersCount";
    private final GestureDetector gestureDetector;
    @Nullable
    private Path mPath;
    @Nullable
    private List<PointF> freeHandDrawingPoints;
    private Paint pathPaint;
    private int pathColor;
    private final LinkedHashMap<Path, Integer> paths = new LinkedHashMap<>();
    private float mX, mY;
    private boolean moved;
    @Nullable
    private volatile Drawable screenshot;
    private final PointF[] lastDrawing = new PointF[5];
    @Nullable
    private Bitmap originalBitmap;
    @Nullable
    private Bitmap scaledBitmap;
    private int drawingLevel;
    private volatile boolean capturing;
    private final ControlButton topLeftButton;
    private final ControlButton bottomRightButton;
    private final ControlButton bottomLeftButton;
    private final ControlButton topRightButton;
    private final PointF touchedPoint = new PointF();
    private volatile ActionMode actionMode = ActionMode.NONE;
    private DrawingMode drawingMode = DrawingMode.NONE;
    @Nullable
    private volatile MarkUpStack currentMarkUpStack;
    private AspectRatioCalculator aspectRatioCalculator = new AspectRatioCalculator();
    @Nullable
    private volatile static MarkUpDrawable selectedMarkUpDrawable;
    @Nullable
    private volatile OnActionDownListener onActionDownListener;
    @Nullable
    private OnNewMagnifierAddingAbilityChangedListener onNewMagnifierAddingAbilityChangedListener;
    @Nullable
    private OnPathRecognizedListener onPathRecognizedListener;
    private boolean isTouching;
    @Nullable
    private Shape recognizedShape;
    @Nullable
    private DirectionRectF recognizedShapeRect;
    private volatile boolean orientationChanged = false;

    private final String ANNOTATION_TASK_KEY = "IBG-ANNOTATION-TASK";

    public AnnotationView(Context context) {
        this(context, null);
    }

    public AnnotationView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public AnnotationView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        currentMarkUpStack = new MarkUpStack();

        gestureDetector = new GestureDetector(context, new GestureListener());
        Paint dotPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        dotPaint.setColor(Color.MAGENTA);

        topLeftButton = new ControlButton();
        bottomRightButton = new ControlButton();
        bottomLeftButton = new ControlButton();
        topRightButton = new ControlButton();

        initDrawing();

        for (int i = 0; i < lastDrawing.length; i++) {
            lastDrawing[i] = new PointF();
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        OrientationUtils.lockScreenOrientation(getContext());
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        int width = getMeasuredWidth();
        int height = getMeasuredHeight();
        int widthWithoutPadding = width - getPaddingLeft() - getPaddingRight();
        int heightWithoutPadding = height - getPaddingTop() - getPaddingBottom();
        setMeasuredDimension(widthWithoutPadding + getPaddingLeft() + getPaddingRight(),
                heightWithoutPadding + getPaddingTop() + getPaddingBottom());
    }

    public void setDrawingColor(int drawingColor) {
        this.pathColor = drawingColor;
        pathPaint.setColor(pathColor);
    }

    @Override
    protected synchronized void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        getScaledDrawables();
    }

    private MarkUpStack getScaledDrawables() {
        aspectRatioCalculator.setNewHeight(getHeight());
        aspectRatioCalculator.setNewWidth(getWidth());
        MarkUpStack newMarkUpDrawables = currentMarkUpStack == null ? new MarkUpStack() : currentMarkUpStack;
        if (newMarkUpDrawables != null)
            for (MarkUpDrawable markUpDrawable : newMarkUpDrawables.getAll()) {
                DirectionRectF rect = new DirectionRectF();
                float left = markUpDrawable.bounds.left * aspectRatioCalculator.getFactorWidth();
                float top = markUpDrawable.bounds.top * aspectRatioCalculator.getFactorHeight();
                float right = markUpDrawable.bounds.right * aspectRatioCalculator.getFactorWidth();
                float bottom = markUpDrawable.bounds.bottom * aspectRatioCalculator
                        .getFactorHeight();
                rect.set(left, top, right, bottom);
                if (markUpDrawable.getShape() instanceof ArrowShape) {
                    ((ArrowShape) markUpDrawable.getShape()).updateTailAndHead(rect);
                }
                rect.setVisibility(markUpDrawable.bounds.isVisible());
                markUpDrawable.setBounds(new DirectionRectF(rect));
            }
        currentMarkUpStack = newMarkUpDrawables;
        return currentMarkUpStack;
    }

    private void initDrawing() {
        pathPaint = new Paint();
        pathPaint.setAntiAlias(true);
        pathPaint.setDither(true);
        pathColor = 0XFFFF0000;
        pathPaint.setColor(pathColor);
        pathPaint.setStyle(Paint.Style.STROKE);
        pathPaint.setStrokeJoin(Paint.Join.ROUND);
        pathPaint.setStrokeCap(Paint.Cap.ROUND);
        pathPaint.setStrokeWidth(4 * getContext().getResources().getDisplayMetrics().density);
    }

    private void touch_start(float x, float y) {
        mPath = new Path();
        freeHandDrawingPoints = new ArrayList<>();
        paths.put(mPath, pathColor);
        mPath.reset();
        mPath.moveTo(x, y);
        freeHandDrawingPoints.add(new PointF(x, y));
        mX = x;
        mY = y;

        resetToCurrentPoint(x, y);
    }

    private void resetToCurrentPoint(float x, float y) {
        for (PointF point : lastDrawing) {
            point.x = x;
            point.y = y;
        }
    }

    private void touch_move(float x, float y) {
        float dx = Math.abs(x - mX);
        float dy = Math.abs(y - mY);
        if (dx >= TOUCH_TOLERANCE || dy >= TOUCH_TOLERANCE) {
            if (mPath != null) {
                mPath.quadTo(mX, mY, (x + mX) / 2, (y + mY) / 2);
            }
            mX = x;
            mY = y;
            if (freeHandDrawingPoints != null) {
                freeHandDrawingPoints.add(new PointF(x, y));
            }
        }
    }

    private void touch_up() {
        Path path = mPath;
        if (path != null && freeHandDrawingPoints != null) {
            path.lineTo(mX, mY);

            // Ignore any path has length < 20
            PathMeasure pm = new PathMeasure(path, false);
            if (pm.getLength() < 20) {
                paths.remove(path);
                return;
            }
            MarkUpStack markUpStack = currentMarkUpStack;
            selectedMarkUpDrawable = new MarkUpDrawable(
                    new PathShape(path, pathPaint.getStrokeWidth(), pathPaint, freeHandDrawingPoints));
            MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
            DirectionRectF rect = new DirectionRectF();
            path.computeBounds(rect, true);
            if (markUpDrawable != null)
                markUpDrawable.setBounds(new DirectionRectF(rect));
            if (markUpStack != null) {
                markUpStack.addOnTop(selectedMarkUpDrawable);
            }
            paths.remove(path);

            invalidate();

            recognizeShape(rect);
        }
    }

    private synchronized void recognizeShape(DirectionRectF rect) {
        if (mPath == null) return;
        PathRecognizer pathRecognizer = new PathRecognizer();
        Recognition recognition = pathRecognizer.recognize(mPath);
        Shape shape = null;
        if (recognition.shapeType == ShapeSpecs.Type.ARROW
                || recognition.shapeType == ShapeSpecs.Type.LINE) {

            float maxDimen = Math.max(rect.width(), rect.height());

            float tailX = rect.centerX() - maxDimen / 2;
            float headX = rect.centerX() + maxDimen / 2;

            PointF tailPoint = new PointF(tailX, rect.centerY());
            PointF headPoint = new PointF(headX, rect.centerY());

            DrawingUtility.rotatePoint(rect.centerX(), rect.centerY(), recognition.angle, tailPoint);
            DrawingUtility.rotatePoint(rect.centerX(), rect.centerY(), recognition.angle, headPoint);

            shape = new ArrowShape(tailPoint, headPoint, pathColor,
                    pathPaint.getStrokeWidth());

            ((ArrowShape) shape).setRecognitionAngle(recognition.angle);
            if (recognition.shapeType == ShapeSpecs.Type.ARROW) {
                ((ArrowShape) shape).setType("arrow");
            }

            float left = Math.min(tailPoint.x, headPoint.x);
            float right = Math.max(tailPoint.x, headPoint.x);
            float top = Math.min(tailPoint.y, headPoint.y);
            float bottom = Math.max(tailPoint.y, headPoint.y);

            rect.set(left, top, right, bottom);
        } else if (recognition.shapeType == ShapeSpecs.Type.RECT) {

            float maxDimen = Math.max(rect.width(), rect.height());

            float left = rect.centerX() - maxDimen / 2;
            float right = rect.centerX() + maxDimen / 2;
            float top = rect.centerY() - maxDimen / 2;
            float bottom = rect.centerY() + maxDimen / 2;
            rect.set(left, top, right, bottom);

            float padding = recognition.paddingPercent * rect.width();

            int ang = recognition.angle;
            if (ang <= 20) {
                ang = 0;
            } else if (ang >= 70 && ang <= 110) {
                ang = 90;
            } else if (ang >= 160) {
                ang = 180;
            }

            if (ang == 0 || ang == 180) {
                rect.left += padding;
                rect.right -= padding;
            } else if (ang == 90) {
                rect.top += padding;
                rect.bottom -= padding;
            } else if (ang > 90 && ang < 180) {
                ang = ang - 90;
                rect.top += padding;
                rect.bottom -= padding;
            } else {
                rect.left += padding;
                rect.right -= padding;
            }

            if ((recognition.angle >= 20 && recognition.angle <= 70)
                    || (recognition.angle >= 110 && recognition.angle <= 160)) {
                float extraWidthPadding = rect.width() * 0.1f;
                float extraHeightPadding = rect.height() * 0.1f;
                rect.left += extraWidthPadding;
                rect.right -= extraWidthPadding;
                rect.top += extraHeightPadding;
                rect.bottom -= extraHeightPadding;
            }

            shape = new RectShape(pathColor, pathPaint.getStrokeWidth(), ang);
        } else if (recognition.shapeType == ShapeSpecs.Type.OVAL) {

            float maxDimen = Math.max(rect.width(), rect.height());

            float left = rect.centerX() - maxDimen / 2;
            float right = rect.centerX() + maxDimen / 2;
            float top = rect.centerY() - maxDimen / 2;
            float bottom = rect.centerY() + maxDimen / 2;
            rect.set(left, top, right, bottom);
            float padding = recognition.paddingPercent * rect.width();

            int ang = recognition.angle;
            if (ang <= 20) {
                ang = 0;
            } else if (ang >= 70 && ang <= 110) {
                ang = 90;
            }

            if (ang >= 90) {
                ang = ang - 90;
                rect.top += padding;
                rect.bottom -= padding;
            } else {
                rect.left += padding;
                rect.right -= padding;
            }

            shape = new OvalShape(pathColor, pathPaint.getStrokeWidth(), ang);
        }

        recognizedShape = shape;
        recognizedShapeRect = rect;
        if (recognizedShape != null) {
            onPathRecognized(mPath, recognizedShape.getPath(recognizedShapeRect));
        }
    }

    @Override
    public void setImageBitmap(Bitmap bm) {
        originalBitmap = bm;
        super.setImageBitmap(bm);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void setShape(Shape shape, DirectionRectF bounds) {
        MarkUpStack markUpStack = currentMarkUpStack;
        MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
        if (markUpDrawable != null && markUpStack != null && markUpDrawable.shape != null) {
            markUpDrawable.setShape(shape, bounds);
            markUpDrawable.shape.setRecognized(true);
            markUpStack.onStateChangedMarkUpDrawable(selectedMarkUpDrawable);
        }
    }

    synchronized void setRecognizedShape() {
        if (selectedMarkUpDrawable != null && recognizedShape != null && recognizedShapeRect != null) {
            setShape(recognizedShape, recognizedShapeRect);
            invalidate();
        }
    }

    @Override
    protected synchronized void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        Drawable currentScreenshot = screenshot;
        if (currentScreenshot != null) {
            currentScreenshot.draw(canvas);
        }

        final MarkUpStack markUpStack = currentMarkUpStack;
        if (markUpStack != null) {
            if (!capturing) {
                drawingLevel = markUpStack.getAll().size();
            }
            final List<MarkUpDrawable> markUpDrawables = markUpStack.getAll();
            for (int i = 0; i < markUpDrawables.size(); i++) {
                final MarkUpDrawable markUpDrawable = markUpDrawables.get(i);
                refreshMarkupBackgroundIfPossible(markUpDrawable);
                markUpDrawable.draw(canvas);
            }
        }

        final MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
        if (!capturing && markUpDrawable != null) {
            if (isTouching) {
                markUpDrawable.drawBorder(canvas);
            }
            markUpDrawable.drawControlButtons(
                    canvas, topLeftButton, topRightButton, bottomRightButton, bottomLeftButton
            );
        }

        if (!paths.isEmpty()) {
            Iterator<Map.Entry<Path, Integer>> it = paths.entrySet().iterator();
            do {
                Map.Entry<Path, Integer> pairs = it.next();
                pathPaint.setColor(pairs.getValue());
                canvas.drawPath(pairs.getKey(), pathPaint);
            } while (it.hasNext());
        }
        if (orientationChanged && markUpDrawable != null) {
            orientationChanged = false;
            if (!markUpDrawable.shape.isRecognized())
                recognizeShape(markUpDrawable.bounds);
        }
    }

    @Override
    public synchronized boolean onTouchEvent(MotionEvent event) {
        if (gestureDetector.onTouchEvent(event)) {
            return true;
        }

        int action = event.getActionMasked();

        float eventX = event.getX();
        float eventY = event.getY();

        switch (action) {
            case MotionEvent.ACTION_DOWN:
                isTouching = true;

                // Save the original bitmap before any addition
                getOriginalBitmap();

                OnActionDownListener currentOnActionDownListener = onActionDownListener;
                if (currentOnActionDownListener != null) {
                    currentOnActionDownListener.onActionDown();
                }
                touchedPoint.set(eventX, eventY);

                if (bottomRightButton.isTouched(touchedPoint) && selectedMarkUpDrawable != null) {
                    actionMode = ActionMode.RESIZE_BY_BOTTOM_RIGHT_BUTTON;
                } else if (bottomLeftButton.isTouched(touchedPoint) && selectedMarkUpDrawable != null) {
                    actionMode = ActionMode.RESIZE_BY_BOTTOM_LEFT_BUTTON;
                } else if (topLeftButton.isTouched(touchedPoint) && selectedMarkUpDrawable != null) {
                    actionMode = ActionMode.RESIZE_BY_TOP_LEFT_BUTTON;
                } else if (topRightButton.isTouched(touchedPoint) && selectedMarkUpDrawable != null) {
                    actionMode = ActionMode.RESIZE_BY_TOP_RIGHT_BUTTON;
                } else {

                    selectedMarkUpDrawable = getSelectedMarkUpDrawable();

                    MarkUpStack markUpStack = currentMarkUpStack;
                    if (selectedMarkUpDrawable == null && markUpStack != null) {
                        switch (drawingMode) {
                            case DRAW_RECT:
                                PoolProvider.postOrderedIOTask(ANNOTATION_TASK_KEY, () -> {
                                    selectedMarkUpDrawable = new MarkUpDrawable(
                                            new RectShape(pathColor, pathPaint.getStrokeWidth(), 0));
                                    markUpStack.addOnTop(selectedMarkUpDrawable);
                                    invalidate();
                                });
                                break;
                            case DRAW_CIRCLE:
                                PoolProvider.postOrderedIOTask(ANNOTATION_TASK_KEY, () -> {
                                    selectedMarkUpDrawable = new MarkUpDrawable(
                                            new OvalShape(pathColor, pathPaint.getStrokeWidth(), 0));
                                    markUpStack.addOnTop(selectedMarkUpDrawable);
                                    invalidate();
                                });
                                break;

                            case DRAW_BLUR:
                                PoolProvider.postOrderedIOTask(ANNOTATION_TASK_KEY, () -> {
                                    selectedMarkUpDrawable = new MarkUpDrawable(
                                            new BlurredRectShape(getOriginalBitmap(), getContext().getApplicationContext()));
                                    markUpStack.addOnBottom(selectedMarkUpDrawable);
                                    invalidate();
                                });
                                break;
                            default:
                                break;
                        }

                        actionMode = ActionMode.DRAW;
                    } else {
                        actionMode = ActionMode.DRAG;
                    }
                }

                updateCroppedMarkUpDrawableBackground();

                invalidate();
                break;

            case MotionEvent.ACTION_MOVE:

                handleActionMode(event);
                updateCroppedMarkUpDrawableBackground();

                invalidate();
                break;

            case MotionEvent.ACTION_UP:

                isTouching = false;
                MarkUpStack markUpStack = currentMarkUpStack;
                MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
                if (actionMode == ActionMode.DRAG || actionMode == ActionMode.RESIZE_BY_TOP_LEFT_BUTTON ||
                        actionMode == ActionMode.RESIZE_BY_TOP_RIGHT_BUTTON ||
                        actionMode == ActionMode.RESIZE_BY_BOTTOM_RIGHT_BUTTON ||
                        actionMode == ActionMode.RESIZE_BY_BOTTOM_LEFT_BUTTON) {
                    if (markUpDrawable != null && markUpStack != null) {
                        markUpStack.onStateChangedMarkUpDrawable(selectedMarkUpDrawable);
                        markUpDrawable.onActionUp();
                    }
                }

                touchedPoint.set(eventX, eventY);

                if (drawingMode != DrawingMode.DRAW_PATH) {
                    actionMode = ActionMode.NONE;
                    invalidate();
                }
                break;
            default:
                //do nothing
                break;
        }

        if (actionMode != ActionMode.RESIZE_BY_TOP_LEFT_BUTTON &&
                actionMode != ActionMode.RESIZE_BY_TOP_RIGHT_BUTTON &&
                actionMode != ActionMode.RESIZE_BY_BOTTOM_RIGHT_BUTTON &&
                actionMode != ActionMode.RESIZE_BY_BOTTOM_LEFT_BUTTON &&
                actionMode != ActionMode.DRAG && actionMode == ActionMode.DRAW &&
                drawingMode == DrawingMode.DRAW_PATH) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    moved = false;
                    touch_start(eventX, eventY);
                    break;
                case MotionEvent.ACTION_MOVE:
                    moved = true;
                    touch_move(eventX, eventY);
                    invalidate();
                    break;
                case MotionEvent.ACTION_UP:
                    touch_up();
                    if (!moved) {
                        this.performClick();
                    }
                    invalidate();
                    break;
            }
        }

        return true;
    }

    private void updateCroppedMarkUpDrawableBackground() {
        // Prevent updating during drawing path because path is not a markup object yet.
        MarkUpStack markUpStack = currentMarkUpStack;
        MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
        if (actionMode != ActionMode.DRAW && markUpStack != null && markUpDrawable != null) {
            for (int i = 1; i < markUpStack.size(); i++) {
                MarkUpDrawable drawable = markUpStack.get(i);
                if (markUpStack.indexOf(markUpDrawable) <= i) {
                    if (drawable.getShape() instanceof ZoomedShape
                            && drawable.isVisible()) {
                        ZoomedShape zoomedShape = (ZoomedShape) drawable.getShape();
                        zoomedShape.setBackground(getScaledBitmap());
                    }
                }
            }
        }
    }

    private synchronized void handleActionMode(MotionEvent event) {

        float eventX = event.getX();
        float eventY = event.getY();
        MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
        switch (actionMode) {
            case DRAG:

                if (markUpDrawable != null) {
                    markUpDrawable.translateBy((int) (eventX - touchedPoint.x),
                            (int) (eventY - touchedPoint.y));
                }
                break;

            case RESIZE_BY_BOTTOM_RIGHT_BUTTON:

                if (markUpDrawable != null) {

                    DirectionRectF rect = new DirectionRectF();

                    if (eventX < markUpDrawable.lastBounds.left) {
                        rect.left = markUpDrawable.lastBounds.right + (int) (eventX - touchedPoint.x);
                        rect.right = markUpDrawable.lastBounds.left;
                    } else {
                        rect.left = markUpDrawable.lastBounds.left;
                        rect.right = markUpDrawable.lastBounds.right + (int) (eventX - touchedPoint.x);
                    }

                    if (eventY < markUpDrawable.lastBounds.top) {
                        rect.top = markUpDrawable.lastBounds.bottom + (int) (eventY - touchedPoint.y);
                        rect.bottom = markUpDrawable.lastBounds.top;
                    } else {
                        rect.top = markUpDrawable.lastBounds.top;
                        rect.bottom =
                                markUpDrawable.lastBounds.bottom + (int) (eventY - touchedPoint.y);
                    }

                    markUpDrawable.adjustBounds(rect);
                    if (markUpDrawable.getShape() instanceof RectShape) {
                        RectShape rectShape = (RectShape) markUpDrawable.getShape();
                        rectShape.adjustPoint2(eventX, eventY, markUpDrawable.bounds);
                    }
                }
                break;

            case RESIZE_BY_BOTTOM_LEFT_BUTTON:

                if (markUpDrawable != null) {

                    DirectionRectF rect = new DirectionRectF();

                    if (eventX > markUpDrawable.lastBounds.right) {
                        rect.left = markUpDrawable.lastBounds.right;
                        rect.right = markUpDrawable.lastBounds.left + (int) (eventX - touchedPoint.x);
                    } else {
                        rect.left = markUpDrawable.lastBounds.left + (int) (eventX - touchedPoint.x);
                        rect.right = markUpDrawable.lastBounds.right;
                    }

                    if (eventY < markUpDrawable.lastBounds.top) {
                        rect.top = markUpDrawable.lastBounds.bottom + (int) (eventY - touchedPoint.y);
                        rect.bottom = markUpDrawable.lastBounds.top;
                    } else {
                        rect.top = markUpDrawable.lastBounds.top;
                        rect.bottom =
                                markUpDrawable.lastBounds.bottom + (int) (eventY - touchedPoint.y);
                    }

                    markUpDrawable.adjustBounds(rect);
                    if (markUpDrawable.getShape() instanceof RectShape) {
                        RectShape rectShape = (RectShape) markUpDrawable.getShape();
                        rectShape.adjustPoint3(eventX, eventY, markUpDrawable.bounds);
                    }
                }

                break;

            case RESIZE_BY_TOP_LEFT_BUTTON:

                if (markUpDrawable != null) {

                    if (markUpDrawable.getShape() instanceof ArrowShape) {
                        ArrowShape arrowShape = (ArrowShape) markUpDrawable.getShape();
                        arrowShape.adjustTailPoint(eventX, eventY, markUpDrawable.bounds);
                    } else {
                        DirectionRectF rect = new DirectionRectF();

                        if (eventX > markUpDrawable.lastBounds.right) {
                            rect.left = markUpDrawable.lastBounds.right;
                            rect.right = markUpDrawable.lastBounds.left + (int) (eventX - touchedPoint.x);
                        } else {
                            rect.left = markUpDrawable.lastBounds.left + (int) (eventX - touchedPoint.x);
                            rect.right = markUpDrawable.lastBounds.right;
                        }

                        if (eventY > markUpDrawable.lastBounds.bottom) {
                            rect.top = markUpDrawable.lastBounds.bottom;
                            rect.bottom = markUpDrawable.lastBounds.top + (int) (eventY - touchedPoint.y);
                        } else {
                            rect.top = markUpDrawable.lastBounds.top + (int) (eventY - touchedPoint.y);
                            rect.bottom = markUpDrawable.lastBounds.bottom;
                        }

                        markUpDrawable.adjustBounds(rect);
                        if (markUpDrawable.getShape() instanceof RectShape) {
                            RectShape rectShape = (RectShape) markUpDrawable.getShape();
                            rectShape.adjustPoint0(eventX, eventY, markUpDrawable.bounds);
                        }
                    }
                }

                break;

            case RESIZE_BY_TOP_RIGHT_BUTTON:

                if (markUpDrawable != null) {

                    if (markUpDrawable.getShape() instanceof ArrowShape) {
                        ArrowShape arrowShape = (ArrowShape) markUpDrawable.getShape();
                        arrowShape.adjustHeadPoint(eventX, eventY, markUpDrawable.bounds);
                    } else {
                        DirectionRectF rect = new DirectionRectF();

                        if (eventX < markUpDrawable.lastBounds.left) {
                            rect.left = markUpDrawable.lastBounds.right + (int) (eventX - touchedPoint.x);
                            rect.right = markUpDrawable.lastBounds.left;
                        } else {
                            rect.left = markUpDrawable.lastBounds.left;
                            rect.right =
                                    markUpDrawable.lastBounds.right + (int) (eventX - touchedPoint.x);
                        }

                        if (eventY > markUpDrawable.lastBounds.bottom) {
                            rect.top = markUpDrawable.lastBounds.bottom;
                            rect.bottom = markUpDrawable.lastBounds.top + (int) (eventY - touchedPoint.y);
                        } else {
                            rect.top = markUpDrawable.lastBounds.top + (int) (eventY - touchedPoint.y);
                            rect.bottom = markUpDrawable.lastBounds.bottom;
                        }
                        markUpDrawable.adjustBounds(rect);
                        if (markUpDrawable.getShape() instanceof RectShape) {
                            RectShape rectShape = (RectShape) markUpDrawable.getShape();
                            rectShape.adjustPoint1(eventX, eventY, markUpDrawable.bounds);
                        }
                    }
                }

                break;

            case DRAW:
                if (markUpDrawable != null) {

                    DirectionRectF rect = new DirectionRectF();

                    if (eventX < touchedPoint.x) {
                        rect.left = (int) eventX;
                        rect.right = (int) touchedPoint.x;
                    } else {
                        rect.left = (int) touchedPoint.x;
                        rect.right = (int) eventX;
                    }

                    if (eventY < touchedPoint.y) {
                        rect.top = (int) eventY;
                        rect.bottom = (int) touchedPoint.y;
                    } else {
                        rect.top = (int) touchedPoint.y;
                        rect.bottom = (int) eventY;
                    }

                    markUpDrawable.setBounds(rect);
                }
                break;
            default:
                break;
        }
    }

    @Nullable
    private MarkUpDrawable getSelectedMarkUpDrawable() {
        MarkUpStack markUpStack = currentMarkUpStack;
        if (markUpStack != null) {
            for (int i = markUpStack.size() - 1; i >= 0; i--) {
                MarkUpDrawable markUpDrawable = markUpStack.get(i);
                if (markUpDrawable.isTouched(touchedPoint)) {
                    return markUpDrawable;
                }
            }
        }
        return null;
    }

    public void addMarkUp(Shape shape) {

        int markupDimension = Math.min(getWidth(), getHeight()) / 2;
        int left = (getWidth() - markupDimension) / 2;
        int top = (getHeight() - markupDimension) / 2;

        DirectionRectF bounds =
                new DirectionRectF(left, top - 30, markupDimension + left, markupDimension + top + 30);
        addMarkUp(shape, bounds, Level.HIGH);
    }

    public void addMarkUp(Shape shape, DirectionRectF bounds) {
        addMarkUp(shape, bounds, Level.HIGH);
    }

    private void addMarkUp(Shape shape, DirectionRectF bounds, Level level) {
        MarkUpDrawable markUpDrawable = new MarkUpDrawable(shape);
        markUpDrawable.setBounds(bounds);
        addMarkUp(markUpDrawable, level);
    }

    private void addMarkUp(MarkUpDrawable markUpDrawable, Level level) {
        // Save the original bitmap before any addition
        getOriginalBitmap();
        selectedMarkUpDrawable = markUpDrawable;
        MarkUpStack currentMarkUpStack = this.currentMarkUpStack;
        if (currentMarkUpStack != null) {
            if (level == Level.LOW) {
                currentMarkUpStack.addOnBottom(markUpDrawable);
            } else {
                currentMarkUpStack.addOnTop(markUpDrawable);
            }
            invalidate();
        }
    }

    @Nullable
    private Bitmap getOriginalBitmap() {
        if (originalBitmap == null) {
            originalBitmap = toBitmap();
        }
        return originalBitmap;
    }

    int magnifiersCount;
    private final static int MAGNIFIERS_COUNT_LIMIT = 5;

    public void addMagnifier() {
        if (magnifiersCount < MAGNIFIERS_COUNT_LIMIT) {
            addMarkUp(new ZoomedShape(getScaledBitmap()));
            magnifiersCount++;
        }

        if (magnifiersCount == MAGNIFIERS_COUNT_LIMIT
                && onNewMagnifierAddingAbilityChangedListener != null) {
            onNewMagnifierAddingAbilityChangedListener.onAbilityChanged(false);
        }
    }

    public void setScreenshot(@Nullable Drawable screenshot) {
        this.screenshot = screenshot;
    }

    @Nullable
    public Bitmap toBitmap() {
        if (getWidth() <= 0 || getHeight() <= 0 || currentMarkUpStack == null) {
            return null;
        }
        return toBitmap(currentMarkUpStack.size());
    }

    private Bitmap toBitmap(int drawingLevel) {

        this.drawingLevel = drawingLevel;

        Bitmap bitmap = Bitmap.createBitmap(getWidth(), getHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);

        capturing = true;
        invalidate();

        this.draw(canvas);

        capturing = false;
        invalidate();

        return bitmap;
    }

    public void setDrawingMode(DrawingMode drawingMode) {
        this.drawingMode = drawingMode;
    }

    public DrawingMode getDrawingMode() {
        return drawingMode;
    }

    public void undo() {
        if (currentMarkUpStack != null) {
            MarkUpDrawable removedMarkUpDrawable = currentMarkUpStack.undo();
            if (removedMarkUpDrawable != null && removedMarkUpDrawable.getShape() instanceof ZoomedShape) {
                magnifiersCount--;
                checkNewMagnifierAddingAbility();
            }
            setSelectedMarkUpDrawable(null);
            updateCroppedMarkUpDrawableBackground();
            invalidate();
        }
    }

    public static void setSelectedMarkUpDrawable(@Nullable MarkUpDrawable markUpDrawable) {
        selectedMarkUpDrawable = markUpDrawable;
    }

    public void setOnActionDownListener(@Nullable OnActionDownListener onActionDownListener) {
        this.onActionDownListener = onActionDownListener;
    }

    public void setWidthAndHeightIfHasNot() {
        if (getWidth() <= 0) {
            ViewGroup.LayoutParams lp = getLayoutParams();
            lp.width = ViewGroup.LayoutParams.MATCH_PARENT;
            lp.height = ViewGroup.LayoutParams.MATCH_PARENT;
            setLayoutParams(lp);
        }
    }

    private enum ActionMode {
        NONE,
        DRAG,
        RESIZE_BY_TOP_LEFT_BUTTON,
        RESIZE_BY_TOP_RIGHT_BUTTON,
        RESIZE_BY_BOTTOM_RIGHT_BUTTON,
        RESIZE_BY_BOTTOM_LEFT_BUTTON,
        DRAW
    }

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        Bundle bundle = new Bundle();
        bundle.putParcelable(SUPER_STATE, super.onSaveInstanceState());
        bundle.putSerializable(ASPECT_RATIO_CALCULATOR, aspectRatioCalculator);
        bundle.putSerializable(DRAWING_MODE, getDrawingMode()); // ... save stuff
        bundle.putInt(DRAWING_LEVEL, drawingLevel); // ... save stuff
        bundle.putInt(MAGNIFIERS_COUNT, magnifiersCount); // ... save stuff
        return bundle;
    }

    enum Level {
        HIGH,
        LOW
    }

    public interface OnActionDownListener {
        void onActionDown();
    }

    public void setOnNewMagnifierAddingAbilityChangedListener(OnNewMagnifierAddingAbilityChangedListener listener) {
        this.onNewMagnifierAddingAbilityChangedListener = listener;
    }

    public interface OnNewMagnifierAddingAbilityChangedListener {
        void onAbilityChanged(boolean ability);
    }

    private void checkNewMagnifierAddingAbility() {
        if (onNewMagnifierAddingAbilityChangedListener != null) {
            if (magnifiersCount == MAGNIFIERS_COUNT_LIMIT) {
                onNewMagnifierAddingAbilityChangedListener.onAbilityChanged(false);
            }
            if (magnifiersCount == MAGNIFIERS_COUNT_LIMIT - 1) {
                onNewMagnifierAddingAbilityChangedListener.onAbilityChanged(true);
            }
        }
    }

    @Override
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    protected void onRestoreInstanceState(Parcelable state) {
        // implicit null check
        if (state instanceof Bundle) {
            Bundle bundle = (Bundle) state;
            this.aspectRatioCalculator = (AspectRatioCalculator) bundle.getSerializable
                    (ASPECT_RATIO_CALCULATOR);
            this.drawingLevel = bundle.getInt(DRAWING_LEVEL);
            this.magnifiersCount = bundle.getInt(MAGNIFIERS_COUNT);
            this.drawingMode = (DrawingMode) bundle.getSerializable(DRAWING_MODE);
            state = bundle.getParcelable(SUPER_STATE);
        }
        if (state != null) {
            super.onRestoreInstanceState(state);
        }
    }

    private void refreshMarkupBackgroundIfPossible(MarkUpDrawable markUpDrawable) {
        if (markUpDrawable.getShape() instanceof ZoomedShape) {
            ((ZoomedShape) markUpDrawable.getShape()).setBackground(getScaledBitmap());
        } else if (markUpDrawable.getShape() instanceof BlurredRectShape) {
            PoolProvider.postOrderedIOTask(ANNOTATION_TASK_KEY, () ->
                    ((BlurredRectShape) markUpDrawable.getShape()).setBackground(getScaledBitmap()));
        }
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        scaledBitmap = null;
        orientationChanged = true;
    }

    @Nullable
    private Bitmap getScaledBitmap() {
        if (getWidth() <= 0 || getHeight() <= 0) {
            return null;
        }
        if (scaledBitmap == null && originalBitmap != null) {
            scaledBitmap = Bitmap.createScaledBitmap(originalBitmap, getWidth(), getHeight(), true);
        }
        return scaledBitmap;
    }

    public enum DrawingMode implements Serializable {
        NONE,
        DRAW_PATH,
        DRAW_RECT,
        DRAW_CIRCLE,
        DRAW_BLUR,
        DRAW_ZOOM
    }

    private class GestureListener extends GestureDetector.SimpleOnGestureListener {

        @Override
        public boolean onDoubleTap(MotionEvent e) {
            MarkUpStack markUpStack = currentMarkUpStack;
            MarkUpDrawable markUpDrawable = selectedMarkUpDrawable;
            if (markUpDrawable != null && markUpStack != null) {
                markUpStack.onStateChangedMarkUpDrawable(selectedMarkUpDrawable);
                markUpDrawable.setVisibility(false);
                if (markUpDrawable.getShape() instanceof ZoomedShape) {
                    magnifiersCount--;
                    checkNewMagnifierAddingAbility();
                }
                selectedMarkUpDrawable = null;
                updateCroppedMarkUpDrawableBackground();
                invalidate();
            }
            return true;
        }
    }

    private void onPathRecognized(Path original, Path recognized) {
        if (onPathRecognizedListener != null) {
            onPathRecognizedListener.onPathRecognized(original, recognized);
        }
    }

    public void setOnPathRecognizedListener(@Nullable OnPathRecognizedListener onPathRecognizedListener) {
        this.onPathRecognizedListener = onPathRecognizedListener;
    }

    public interface OnPathRecognizedListener {
        void onPathRecognized(Path original, Path recognized);
    }

    @Override
    synchronized protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        currentMarkUpStack = null;
        selectedMarkUpDrawable = null;
        OrientationUtils.unlockOrientation(getContext());
    }
}
