package com.instabug.chat;

import android.annotation.SuppressLint;
import android.os.Build;
import android.os.Bundle;

import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import com.instabug.library.Feature;
import com.instabug.library.Instabug;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.Map;

public final class Replies {

    private static final String TAG = "Replies";

    /**
     Shows the chats list only if the user has a chats history.
     Use {@link Replies#hasChats()} to know if a user has chats history or not.

     @see Instabug.Builder#build()
     */
    public static void show() {
        APIChecker.checkAndRunInExecutor("Replies.show", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.show();
            }
        });
    }

    /**
     To know if a user already has a chats history or not.
     Use it before calling  {@link #show()}.

     @see Instabug.Builder#build()
     */
    public static boolean hasChats() {
        Boolean hasChats = APIChecker.checkAndGet("Replies.hasChats", new ReturnableRunnable<Boolean>() {
            @Override
            public Boolean run() {
                return RepliesWrapper.hasChats();
            }
        }, false);
        return hasChats != null && hasChats;
    }

    /**
     Sets the runnable that gets executed once receive new message<br/>
     WARNING: This runs on your application's main UI thread. Please do not include
     any blocking operations to avoid ANRs.

     @param runnable to run on the UI thread once receive new message
     */
    public static void setOnNewReplyReceivedCallback(final Runnable runnable) {
        APIChecker.checkAndRunInExecutorThenPostOnMain("Replies.setOnNewReplyReceivedCallback", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.setNewMessageHandler(runnable);
            }
        });
    }

    /**
     Enabled/disable chat in-app notification

     @param enabled whether chat in-app notification is enabled or not
     */
    public static void setInAppNotificationEnabled(final boolean enabled) {
        APIChecker.checkAndRunInExecutor("Replies.setInAppNotificationEnabled", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.enableNotification(enabled);
            }
        });
    }

    /**
     Get current unread count of messages for this user

     @return number of messages that are unread for this user
     */
    public static int getUnreadRepliesCount() {
        Integer count = APIChecker.checkAndGet("Replies.getUnreadRepliesCount", new ReturnableRunnable<Integer>() {
            @Override
            public Integer run() {
                return RepliesWrapper.getUnreadMessagesCount();
            }
        }, 0);
        return count != null ? count : 0;
    }

    /**
     Enable/disable push notifications feature

     @param state desired state of push notifications feature

     @see com.instabug.library.Feature.State
     */
    public static void setPushNotificationState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("Replies.setPushNotificationState", new VoidRunnable() {
            @Override
            public void run() {
                InstabugCore.setPushNotificationState(state);
            }
        });
    }

    /**
     Set the GCM registration token to Instabug

     @param token the GCM registration token
     */
    public static void setPushNotificationRegistrationToken(@NonNull final String token) {
        APIChecker.checkAndRunInExecutor("Replies.setPushNotificationRegistrationToken", () -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "New FCM push token received");
            RepliesWrapper.setPushNotificationRegistrationToken(token);
        });
    }

    /**
     Show in-app Messaging's notifications

     @param data the data bundle related to Instabug

     @see #setPushNotificationRegistrationToken(String)
     */
    public static void showNotification(final Bundle data) {
        APIChecker.checkAndRunInExecutor("Replies.showNotification", () -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "New FCM message received (Bundle), showing notification...");
            RepliesWrapper.showNotification(data);
        });
    }

    /**
     Show in-app Messaging's notifications

     @param data the data bundle related to Instabug

     @see #setPushNotificationRegistrationToken(String)
     */
    public static void showNotification(final Map<String, String> data) {
        APIChecker.checkAndRunInExecutor("Replies.showNotification", () -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "New FCM message received (Map), showing notification...");
            RepliesWrapper.showNotification(data);
        });
    }

    /**
     Set whether new in app notification received will play a small sound notification
     or not (Default is {@code false})

     @param shouldPlaySound desired state of conversation sounds
     */
    public static void setInAppNotificationSound(final boolean shouldPlaySound) {
        APIChecker.checkAndRunInExecutor("Replies.setInAppNotificationSound", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.enableInAppNotificationSound(shouldPlaySound);
            }
        });
    }

    /**
     Set whether new messages received will trigger a small sound notification or not (Default
     is {@code false})

     @param shouldPlaySounds desired state of conversation sounds

     @since 8.1.0
     */
    public static void setShouldPlayConversationSounds(final boolean shouldPlaySounds) {
        APIChecker.checkAndRunInExecutor("Replies.setShouldPlayConversationSounds", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.enableConversationSound(shouldPlaySounds);
            }
        });
    }

    /**
     Acts as a master switch for the In-app Replies.
     It's enabled by default. When disabled, the user can’t reach the chats history from the SDK. The chats list button is removed from Instabug Prompt Options.
     In addition, when disabled the in-app notification as well as the push notifications are disabled.
     And, {@link #show()} won’t have an effect.

     @param state possible states are ENABLED and DISABLED.

     @see Instabug.Builder#build()
     */
    public static void setState(@NonNull final Feature.State state) {
        APIChecker.checkAndRunInExecutor("Replies.setState", new VoidRunnable() {
            @Override
            public void run() {
                if (state == null) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "state object passed to Replies.setState() is null");
                    return;
                }
                RepliesWrapper.setState(state);
                IBGCoreEventPublisher.post(Features.Updated.INSTANCE);
            }
        });
    }

    /**
     Set the push notification's icon that will be shown with Instabug notifications

     @param notificationIcon the notification icon resource ID

     @since 8.1.0
     */
    public static void setNotificationIcon(@DrawableRes final int notificationIcon) {
        APIChecker.checkAndRunInExecutor("Replies.setNotificationIcon", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.setNotificationIcon(notificationIcon);
            }
        });
    }

    /**
     Check if the received push notification is related to instabug or not

     @param data the push notification data bundle

     @since 8.1.0
     */
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static boolean isInstabugNotification(final Bundle data) {
        return APIChecker.checkAndGet("Replies.isInstabugNotification", () -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Checking if a notification (Bundle) is IBG's");
            boolean isIBGNotification = RepliesWrapper.isInstabugNotification(data);
            InstabugSDKLogger.d(Constants.LOG_TAG, "Is an IBG notification: " + isIBGNotification);
            return isIBGNotification;
        }, false);
    }

    /**
     Check if the received push notification is related to instabug or not

     @param data the push notification data bundle, false if the In-App chat feature
     disabled

     @since 8.1.0
     */
    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public static boolean isInstabugNotification(final Map<String, String> data) {
        return APIChecker.checkAndGet("Replies.isInstabugNotification", () -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Checking if a notification (Map) is IBG's");
            boolean isIBGNotification = RepliesWrapper.isInstabugNotification(data);
            InstabugSDKLogger.d(Constants.LOG_TAG, "Is an IBG notification: " + isIBGNotification);
            return isIBGNotification;
        }, false);
    }

    /**
     Set whether new system notification received will play the default sound from
     RingtoneManager or not (Default is {@code false})

     @param shouldPlaySound desired state of conversation sounds

     @since 8.1.0
     */
    public static void setSystemReplyNotificationSoundEnabled(final boolean shouldPlaySound) {
        APIChecker.checkAndRunInExecutor("Replies.setSystemReplyNotificationSoundEnabled", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.enableSystemNotificationSound(shouldPlaySound);
            }
        });
    }

    /**
     Set a notification channel id to a notification channel that notifications
     can be posted to.

     @param pushNotificationChannelId an id to a notification channel that notifications
     can be posted to.

     @since 8.1.0
     */
    @RequiresApi(api = Build.VERSION_CODES.O)
    public static void setPushNotificationChannelId(final String pushNotificationChannelId) {
        APIChecker.checkAndRunInExecutor("Replies.setPushNotificationChannelId", new VoidRunnable() {
            @Override
            public void run() {
                RepliesWrapper.setPushNotificationChannelId(pushNotificationChannelId);
            }
        });
    }
}
