package com.instabug.chat;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.model.Message;
import com.instabug.chat.notification.NotificationManager;
import com.instabug.chat.synchronization.NewMessagesHandler;
import com.instabug.chat.synchronization.OnNewMessagesReceivedListener;
import com.instabug.chat.synchronization.SynchronizationManager;
import com.instabug.library.IBGFeature;
import com.instabug.library.PresentationManager;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.eventpublisher.IBGDisposable;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.eventpublisher.IBGCompositeDisposable;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.tokenmapping.MappedTokenChangedEventBus;
import com.instabug.library.util.threading.PoolProvider;

import java.util.ArrayList;
import java.util.List;

@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class ChatPlugin extends Plugin implements OnNewMessagesReceivedListener {

    @Nullable
    IBGCompositeDisposable disposables;

    @VisibleForTesting
    @Nullable
    IBGDisposable mappedTokenChangedDisposable;

    @Override
    public void init(Context context) {
        super.init(context);
    }

    @Override
    public void wake() {
        // TODO map APIs
    }

    @Override
    public void sleep() {
        // TODO map APIs
    }

    @Override
    public void stop() {
        unSubscribeFromCoreEvents();
        unsubscribeFromMappedTokenChangeEvent();
        ChatPluginWrapper.release();
        NewMessagesHandler.getInstance().removeOnNewMessagesReceivedListener(this);
    }

    @Override
    public void start(final Context context) {
        PoolProvider.postIOTaskWithCheck(() -> {
            subscribeOnCoreEvents();
            subscribeToMappedTokenChangeEvent();
                NewMessagesHandler.getInstance().addOnNewMessagesReceivedListener(ChatPlugin.this);
                ChatPluginWrapper.wake(context);
                sendPushNotificationToken(false);
        });
    }

    @Override
    public long getLastActivityTime() {
        return ChatPluginWrapper.getLastActivityTime();
    }

    @Override
    public void initDefaultPromptOptionAvailabilityState() {
        ChatPluginWrapper.initDefaultPromptOptionAvailabilityState();
    }

    @Override
    @Nullable
    public ArrayList<PluginPromptOption> getPromptOptions() {
        if (contextWeakReference != null && contextWeakReference.get() != null) {
            return ChatPluginWrapper.getPromptOptions(contextWeakReference.get());
        }
        return null;
    }

    @Override
    @Nullable
    public ArrayList<PluginPromptOption> getPluginOptions(boolean ignoreBaseFeature) {
        if (contextWeakReference != null && contextWeakReference.get() != null) {
            return ChatPluginWrapper.getPromptOptions(contextWeakReference.get());
        }
        return null;
    }

    @Override
    public boolean isFeatureEnabled() {
        return InstabugCore.isFeatureEnabled(IBGFeature.REPLIES) && InstabugCore.isFeatureEnabled(IBGFeature.IN_APP_MESSAGING);
    }

    @Override
    @Nullable
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_RETURN_ANNOTATION")
    public List<Message> onNewMessagesReceived(@NonNull final List<Message> newMessages) {
        if (contextWeakReference != null) {
            final Context context = contextWeakReference.get();
            if (context != null) {
                if (InstabugCore.isAppOnForeground()) {
                    PresentationManager.getInstance().show(() -> NotificationManager.getInstance().showNotification(context,
                            newMessages));
                } else {
                    NotificationManager.getInstance().showNotification(context,
                            newMessages);
                }
            }
        }
        return null;
    }

    @VisibleForTesting
    void subscribeOnCoreEvents() {
        if (contextWeakReference != null && contextWeakReference.get() != null) {
            getOrCreateCompositeDisposables().add(subscribeToCoreEvents());
        }

    }

    @NonNull
    private IBGDisposable subscribeToCoreEvents() {
        return IBGCoreEventSubscriber
                .subscribe((event -> {
                    if (contextWeakReference != null) {
                        ChatCoreEventHandler.handleCoreEvents(contextWeakReference.get(), event);
                    }
                }));
    }

    private void unSubscribeFromCoreEvents() {
        if (disposables != null) disposables.dispose();

    }

    private void subscribeToMappedTokenChangeEvent() {
        if (mappedTokenChangedDisposable == null) {
            mappedTokenChangedDisposable = MappedTokenChangedEventBus.INSTANCE.subscribe(event -> {
                ChatsCacheManager.clearChats();
                sendPushNotificationToken(true);
                SynchronizationManager synchronizationManager = SynchronizationManager.getInstance();
                if (synchronizationManager != null) {
                    synchronizationManager.sync(true);
                }
            });
        }
    }

    private void unsubscribeFromMappedTokenChangeEvent() {
        if (mappedTokenChangedDisposable != null) {
            mappedTokenChangedDisposable.dispose();
            mappedTokenChangedDisposable = null;
        }
    }

    @SuppressLint("NULL_DEREFERENCE")
    private void sendPushNotificationToken(boolean forceSend) {
        ChatPluginWrapper.sendPushNotificationToken(forceSend);
    }

    private IBGCompositeDisposable getOrCreateCompositeDisposables() {
        return disposables != null ? disposables : (disposables = new IBGCompositeDisposable());
    }
}
