package com.instabug.bug.view.annotation;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.net.Uri;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.LinearLayout;

import androidx.annotation.Nullable;
import androidx.core.view.ViewCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;

import com.instabug.bug.R;
import com.instabug.bug.view.BugReportingActivityCallback;
import com.instabug.bug.view.reporting.ReportingContainerActivity;
import com.instabug.chat.annotation.AnnotationLayout;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.ThemeApplier;

import java.io.File;

public class BugAnnotationFragment extends InstabugBaseFragment<AnnotationPresenter> implements
        AnnotationContract.View {

    public static final String TAG = "annotation_fragment_for_bug";
    private static final String EXTRA_TITLE = "title";
    private static final String EXTRA_IMAGE_URI = "image_uri";
    private static final String EXTRA_NAME = "name"; // used for shared item transition
    @Nullable
    private String title;
    private String currentTitle = "";

    @Nullable
    private Uri imageUri;
    @Nullable
    private AnnotationLayout mAnnotationLayout;
    @Nullable
    private Callbacks listener;
    @Nullable
    private BugReportingActivityCallback activityCallback;
    @Nullable
    private Bitmap bitmap;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static BugAnnotationFragment newInstance(String title, Uri imageUri, @Nullable String name) {
        BugAnnotationFragment fragment = new BugAnnotationFragment();
        Bundle args = new Bundle();
        args.putString(EXTRA_TITLE, title);
        args.putParcelable(EXTRA_IMAGE_URI, imageUri);
        args.putString(EXTRA_NAME, name);
        fragment.setArguments(args);
        return fragment;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        postponeEnterTransition();
        super.onCreate(savedInstanceState);
        setHasOptionsMenu(true);
//        TODO uncomment this block when https://issuetracker.google.com/issues/188457866 is fixed.
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
//            if (getContext() != null) {
//                setSharedElementEnterTransition(TransitionInflater.from(getContext()).inflateTransition(android.R.transition.move));
//                setSharedElementReturnTransition(TransitionInflater.from(getContext()).inflateTransition(android.R.transition.move));
//            }
//        }
        if (getArguments() != null) {
            title = getArguments().getString(EXTRA_TITLE);
            imageUri = getArguments().getParcelable(EXTRA_IMAGE_URI);
        }
        if (activityCallback != null) {
            currentTitle = activityCallback.getToolbarTitle();
            if (title != null) {
                activityCallback.setToolbarTitle(title);
            }
            activityCallback.setToolbarUpIconClose();
        }
        presenter = new AnnotationPresenter(this);
        if (getActivity() != null && imageUri != null && imageUri.getPath() != null) {
            // todo: Move this to a BG thread
            BitmapUtils.compressBitmapAndSave(getActivity(), new File(imageUri.getPath()));
            bitmap = BitmapUtils.getBitmapFromUri(imageUri);
        }
    }

    @Override
    protected int getLayout() {
        return R.layout.ibg_bug_fragment_annotation;
    }

    @Override
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        //init annotation layout
        AnnotationLayout annotationLayout = findViewById(com.instabug.bug.R.id.annotationLayout);
        if (annotationLayout != null && getArguments() != null && getArguments().getString(EXTRA_NAME) != null) {
            View annotationImage = annotationLayout.findViewById(R.id.instabug_annotation_image);
            if (annotationImage != null) {
                ViewCompat.setTransitionName(annotationImage, getArguments().getString(EXTRA_NAME));
            }
        }
        mAnnotationLayout = annotationLayout;
        if (presenter != null && bitmap != null) {
            presenter.loadViewData(bitmap);
        }
        startPostponedEnterTransition();
        if (theme != null && theme.getBackgroundColor() != ThemeApplier.DEFAULT_COLOR) {
            rootView.setBackgroundColor(theme.getBackgroundColor());
            setActionContainerBackground(rootView);
        }
    }

    private static void setActionContainerBackground(View rootView) {
        LinearLayout actionContainer = rootView.findViewById(R.id.instabug_annotation_actions_container);
        if (actionContainer != null) {
            actionContainer.setBackgroundColor(Color.TRANSPARENT);
        }
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        listener = (Callbacks) getActivity();
        if (getActivity() instanceof BugReportingActivityCallback) {
            try {
                activityCallback = (BugReportingActivityCallback) getActivity();
            } catch (Exception e) {
                throw new RuntimeException("Must implement AnnotationFragment.Callbacks ");
            }
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        if (getActivity() instanceof ReportingContainerActivity) {
            ((ReportingContainerActivity) getActivity()).setToolbarNavigationContentDescription(R.string.ibg_core_annotation_ic_close_content_description);
        }
    }

    @Nullable
    protected String getTitle() {
        return title;
    }

    protected void onDoneButtonClicked() {
        FragmentActivity activity = getActivity();
        if (activity == null) {
            return;
        }
        if (listener != null && mAnnotationLayout != null) {
            if (imageUri != null) {
                listener.onImageEditingDone(mAnnotationLayout.getAnnotatedBitmap(), imageUri);
            }
            activity.getSupportFragmentManager().beginTransaction().remove(this).commit();
            activity.getSupportFragmentManager().popBackStack(TAG, FragmentManager
                    .POP_BACK_STACK_INCLUSIVE);
        }
    }

    @Override
    public void setAnnotationBitmap(Bitmap bitmap) {
        if (mAnnotationLayout != null) {
            mAnnotationLayout.setBitmap(bitmap);
        }
    }

    @Override
    public void showToastError() {

    }

    public interface Callbacks {
        void onImageEditingDone(@Nullable Bitmap annotatedBitmap, Uri annotatedImageUri);

        void onImageEditingCancelled();
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        menu.clear();
        inflater.inflate(R.menu.ibg_bug_instabug_bug_annoataion, menu);
        MenuItem menuItem = menu.findItem(R.id.instabug_bugreporting_annotaion_done);
        if (menuItem != null) {
            menuItem.setTitle(R.string.ibg_core_annotation_ic_done_content_description);
        }
        super.onCreateOptionsMenu(menu, inflater);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == R.id.instabug_bugreporting_annotaion_done) {
            onDoneButtonClicked();
            return true;
        } else if (item.getItemId() == android.R.id.home) {
            if (getActivity() != null) {
                getActivity().onBackPressed();
            }
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public void onDestroy() {
        if (activityCallback != null) {
            activityCallback.setToolbarUpIconClose();
            activityCallback.setToolbarTitle(currentTitle);
        }
        super.onDestroy();
    }
}
