package com.instabug.bug.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Path;
import android.graphics.PixelFormat;
import android.graphics.RectF;
import android.os.Build;
import android.util.AttributeSet;

import androidx.annotation.Nullable;

import com.instabug.bug.R;

import org.jetbrains.annotations.NotNull;

public class CorneredImageView extends androidx.appcompat.widget.AppCompatImageView {

    public static final int CORNER_NONE = 0;
    public static final int CORNER_TOP_LEFT = 1;
    public static final int CORNER_TOP_RIGHT = 2;
    public static final int CORNER_BOTTOM_RIGHT = 4;
    public static final int CORNER_BOTTOM_LEFT = 8;
    public static final int CORNER_ALL = 15;

    private final RectF cornerRect = new RectF();
    private final Path path = new Path();
    private int cornerRadius;
    private int roundedCorners;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public CorneredImageView(Context context) {
        this(context, null);
    }

    public CorneredImageView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public CorneredImageView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.CorneredImageView);
        cornerRadius = a.getDimensionPixelSize(R.styleable
                .CorneredImageView_ib_bug_cornerRadius, 0);
        roundedCorners = a.getInt(R.styleable.CorneredImageView_ib_bug_roundedCorners,
                CORNER_NONE);
        a.recycle();
    }

    public void setCornerRadius(int radius) {
        cornerRadius = radius;
        setPath();
        invalidate();
    }

    public int getRadius() {
        return cornerRadius;
    }

    public int getRoundedCorners() {
        return roundedCorners;
    }

    public void setRoundedCorners(int corners) {
        roundedCorners = corners;
        setPath();
        setBackgroundDrawable(new Border(0x000000, 10));
        invalidate();
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        setPath();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (!path.isEmpty() && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            canvas.clipPath(path);
        }
        super.onDraw(canvas);
    }

    private void setPath() {
        path.rewind();

        if (cornerRadius >= 1f && roundedCorners != CORNER_NONE) {
            final int width = getWidth();
            final int height = getHeight();
            final float twoRadius = cornerRadius * 2;
            cornerRect.set(-cornerRadius, -cornerRadius, cornerRadius, cornerRadius);

            if (isRounded(CORNER_TOP_LEFT)) {
                cornerRect.offsetTo(0f, 0f);
                path.arcTo(cornerRect, 180f, 90f);
            } else {
                path.moveTo(0f, 0f);
            }

            if (isRounded(CORNER_TOP_RIGHT)) {
                cornerRect.offsetTo(width - twoRadius, 0f);
                path.arcTo(cornerRect, 270f, 90f);
            } else {
                path.lineTo(width, 0f);
            }

            if (isRounded(CORNER_BOTTOM_RIGHT)) {
                cornerRect.offsetTo(width - twoRadius, height - twoRadius);
                path.arcTo(cornerRect, 0f, 90f);
            } else {
                path.lineTo(width, height);
            }

            if (isRounded(CORNER_BOTTOM_LEFT)) {
                cornerRect.offsetTo(0f, height - twoRadius);
                path.arcTo(cornerRect, 90f, 90f);
            } else {
                path.lineTo(0f, height);
            }

            path.close();
        }
    }

    private boolean isRounded(int corner) {
        return (roundedCorners & corner) == corner;
    }

    public static class Border extends android.graphics.drawable.Drawable {
        public android.graphics.Paint paint;
        @Nullable
        public android.graphics.Rect bounds_rect;

        public Border(int colour, int width) {
            this.paint = new android.graphics.Paint();
            this.paint.setColor(colour);
            this.paint.setStrokeWidth(width);
            this.paint.setStyle(android.graphics.Paint.Style.STROKE);
        }

        @Override
        public void onBoundsChange(android.graphics.Rect bounds) {
            this.bounds_rect = bounds;
        }

        public void draw(@NotNull android.graphics.Canvas c) {
            if (this.bounds_rect != null) {
                c.drawRect(this.bounds_rect, this.paint);
            }
        }

        public void setAlpha(int a) {
        }

        public void setColorFilter(@Nullable android.graphics.ColorFilter cf) {
        }

        @Override
        public int getOpacity() {
            return PixelFormat.OPAQUE;
        }

    }
}