package com.instabug.bug.utils;

import static android.os.Build.VERSION_CODES.JELLY_BEAN;
import static android.os.Build.VERSION_CODES.TIRAMISU;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

/**
 * @author mSobhy
 */
public class PermissionsUtils {

    private static final String PERMISSION_MESSAGE_POSTFIX = "Permission %1$s ";

    private PermissionsUtils() {
    }

    public static String mediaStoragePermission() {
        if (Build.VERSION.SDK_INT == TIRAMISU) {
            //this is only used in API 33 if the user added the permission to the manifest
            return Manifest.permission.READ_MEDIA_IMAGES;
        } else {
            return Manifest.permission.READ_EXTERNAL_STORAGE;
        }
    }

    /**
     * @param context used for checking permission based upon
     * @return whether or not media storage permission is granted
     */
    public static boolean isMediaStoragePermissionGranted(@Nullable Context context) {
        if (Build.VERSION.SDK_INT >= JELLY_BEAN) {
            if (context == null) return false;
            return isPermissionGranted(context, mediaStoragePermission());
        }
        return true;
    }

    /**
     * @param context    used for checking permission based upon
     * @param permission to check if granted or not {@link Manifest.permission}
     * @return whether this permission is granted or not
     */
    public static boolean isPermissionGranted(@NonNull Context context, @NonNull String permission) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                int hasPermission = ContextCompat.checkSelfPermission(context, permission);
                boolean isGranted = hasPermission == PackageManager.PERMISSION_GRANTED;
                InstabugSDKLogger.v(Constants.LOG_TAG, String.format(PERMISSION_MESSAGE_POSTFIX, permission) + "state is " + (isGranted ? "" : "NOT ") + "granted");
                return isGranted;
            } else {
                int res = context.checkCallingOrSelfPermission(permission);
                boolean isGranted = (res == PackageManager.PERMISSION_GRANTED);
                InstabugSDKLogger.v(Constants.LOG_TAG, String.format(PERMISSION_MESSAGE_POSTFIX, permission) + "state is " + (isGranted ? "" : "NOT ") + "granted");
                return isGranted;
            }
        } catch (Exception ex) {
            // NOTE this happens if running an app not targeting M and needs permissions (which wouldn't work)
            return true;
        }
    }

    /**
     * This activity must implement {@link Activity#onRequestPermissionsResult(int, String[], int[])}
     *
     * @param activity                           that implements  {@link Activity#onRequestPermissionsResult(int, String[], int[])}
     * @param permission                         permission to get access to
     * @param requestCode                        request code to use to get permission result
     * @param onShouldNotShowPermissionRationale runnable in case of permission denied forever (or denied once before)
     * @param afterPermissionGrantedRunnable     runnable in case of permission granted without system interference
     */
    public static void requestPermission(@NonNull Activity activity, @NonNull String permission, int requestCode
            , @Nullable Runnable onShouldNotShowPermissionRationale, @Nullable Runnable afterPermissionGrantedRunnable) {
        if (!isPermissionGranted(activity, permission)) {
            if (!ActivityCompat.shouldShowRequestPermissionRationale(activity, permission)) {
                runIfValid(onShouldNotShowPermissionRationale);
            }
            InstabugSDKLogger.d(Constants.LOG_TAG, String.format(PERMISSION_MESSAGE_POSTFIX, permission) + "not granted, requesting it");
            ActivityCompat.requestPermissions(activity, new String[]{permission},
                    requestCode);
        } else {
            InstabugSDKLogger.d(Constants.LOG_TAG, String.format(PERMISSION_MESSAGE_POSTFIX, permission) + "already granted, running after permission granted runnable");
            runIfValid(afterPermissionGrantedRunnable);
        }
    }

    private static void runIfValid(@Nullable Runnable runnable) {
        if (runnable != null) {
            runnable.run();
        }
    }
}
