package com.instabug.bug.userConsent

import com.instabug.bug.model.Bug
import com.instabug.bug.utils.deleteAttachment
import com.instabug.library.model.Attachment
import com.instabug.library.util.dropLogs

fun interface ConsentsActionHandler {
    fun handle(bug: Bug)
}

class ConsentsActionHandlerImpl : ConsentsActionHandler {

    override fun handle(bug: Bug) {
        bug.actionableConsents?.takeIf { consents ->
            consents.isNotEmpty()
        }?.forEach { actionType ->
            when (actionType) {
                ActionType.DROP_AUTO_CAPTURED_MEDIA -> dropAutoCapturedMedia(bug)
                ActionType.DROP_LOGS -> dropLogs(bug)
            }
        }
    }

    private fun dropAutoCapturedMedia(bug: Bug) {
        dropAutoCapturedMediaAttachments(bug)
        clearViewHierarchy(bug)
        clearVisualUserSteps(bug)
    }

    private fun dropAutoCapturedMediaAttachments(bug: Bug) {
        val attachments = bug.attachments
        val attachmentsToRemove = mutableListOf<Attachment>()
        if (attachments?.isNotEmpty() == true) {
            attachments.forEach { attachment ->
                if (attachment.type in listOf(
                        Attachment.Type.VIEW_HIERARCHY,
                        Attachment.Type.MAIN_SCREENSHOT,
                        Attachment.Type.AUTO_SCREEN_RECORDING_VIDEO,
                        Attachment.Type.VISUAL_USER_STEPS
                    )
                ) {
                    attachmentsToRemove.add(attachment)
                    deleteAttachment(attachment, bug.id)
                }
            }

            attachments.removeAll(attachmentsToRemove)
        }
    }

    private fun clearViewHierarchy(bug: Bug) {
        bug.viewHierarchy = null
    }

    private fun clearVisualUserSteps(bug: Bug) {
        bug.state?.setVisualUserSteps(null)
    }

    private fun dropLogs(bug: Bug) {
        bug.state?.dropLogs()
        bug.removeFileAttachments()
    }

    private fun Bug.removeFileAttachments() = attachments
        ?.filter { it.type == Attachment.Type.ATTACHMENT_FILE }
        ?.onEach { deleteAttachment(it, id) }
        ?.toList()
        ?.let(attachments::removeAll)

}
